//	James Z. Chen
//	January 1, 2005
//	All rights reserved

#ifndef __PROJECT_H
#define __PROJECT_H 1

#include "/usr/include/stdio.h"
#include "/usr/include/stdlib.h"
#include <math.h>
#include "parameter.h"
#include "maps.h"


struct EMS_RECORD {
	char  select;	// flag for selection status
	float lcfScore;	// highest LCF among templates
	float scfScore;	// highest SCF among templates
	float dcfScore;	// highest DCF among templates
	float euler;	// athimuthal Euler angle
	int   tmpid;	// template ID associated with "score"
};


struct EMS_FILAMENT {
	int segid;					// segments in the filament
	int x[EMS_FILA_LENG];		// segment axial position
	int y[EMS_FILA_LENG];
	float nx[EMS_FILA_LENG];	// segment orientation vector
	float ny[EMS_FILA_LENG];
};


struct EMS_REFERENCE {
	float *ptkStack;		// original template stack (3D)
	float *pwrStack;		// auto-corr stack of ptkStack (3D)
	char  *L1Mask;			// level-1 template masking (3D)
	char  *L2Mask;			// level-2 template masking (3D)
	char  maskID;			// masking levels
	int   *xc, *yc;			// center of alignment (xc[], yc[])
	int   editFrame;		// frame selection for editing
	float scale;			// frame display scaling
	char  *select;			// EMS_TRUE for "include", EMS_FALSE for "exclude"
};


struct EMS_PARAMETER {
	// system parameters
	char  version[EMS_VLENGTH];
	float flmBrit;			// brightness ctrl of image
	float flmCntr;			// contrast ctrl of image
	float refBrit;			// brightness ctrl of references
	float refCntr;			// contrast ctrl of references
	float spcBrit;			// brightness ctrl of spectrum
	float spcCntr;			// contrast ctrl of spectrum
	float monBrit;			// brightness ctrl of monitor 1 & 2
	float monCntr;			// contrast ctrl of monitor 1 & 2
	float mon3Brit;			// brightness ctrl of monitor 3
	float mon3Cntr;			// contrast ctrl of monitor 3

	// MICROGRAPH panel
	int   imgX, imgY;		// dimension of downsized micrograph
	long  filmSize;			// imgX * imgY
	float pixelSize;		// image pixel size (A)
	int   specSize;			// image size for spectrum display
	int   downSize;			// micrograph downsize factor
	int   dispScal;			// downsize factor for display purpose
	int   variaBox;			// variance box size (pixel)
	float prtkSize;			// particle diameter
	int   edgeExcu;			// edge exclusion size
	float varHi, varLo;		// variance thresholding factors
	int   brushSize;		// masking brush size
	float passL, passH;		// band-pass filtering limits
	unsigned char transp;	// for film masking

	// TEMPLATE panel
	int   tmpX, tmpY, tmpN;	// dimension of downsized templates
	long  tmpSize;			// tmpX * tmpY
	int   lcfMaskNum;		// active pixels in lcfMask
	int   scfMaskNum;		// active pixels in scfMask
	float lcfMaskRad;		// LCF mask radius
	float scfMaskRad;		// SCF mask radius
	float scfMaskRadc;		// SCF center blocking radius
	float rotaRange;		// rotational search range
	float rotaDelta;		// rotational search increatment

	// PARTICLES panel
	int   totCount;			// total count of particle selection
	float lcfThres;			// LCF thresholding level
	int   lcfCount;			// LCF selection
	float scfThres;			// SCF thresholding level
	int   scfCount;			// SCF selection
	float dcfThres;			// DCF thresholding level
	int   dcfCount;			// DCF selection
	float distance;			// minimum distance between particles

	// FILAMENT panel
	int filaCount;			// total count of filament selection
	float filaBrdg;			// bridging factor for segment connection
	float filaBend;			// bending tolerance in filament tracing

	// screening statistics
	float lcfMax, lcfAve;	// from LCF map
	float scfMax, scfAve;	// from SCF map
	float dcfMax, dcfAve;	// from DCF map
};


struct EMS_SYSSTATUS {
	int   emsdpMode;		// Display Panel mode
	int   emscpMode;		// Control Panel mode
	char  emsProject;		// EMS_TRUE for defined project
	char  imgInvert;		// EMS_TRUE for image contrast inversion
	char  imgInform;		// toggle between SPEC (default) and HSTG modes
	char  testScreen;		// EMS_TRUE for test screening (sum of all templates)
	char  fullScreen;		// EMS_TRUE for full screening (every template image)
	char  rotaScreen;		// EMS_TRUE for full screening (rotational search)
	char  mapRestore;		// EMS_TRUE for loading instead of computing maps
	char  showPrtk;			// EMS_TRUE to show the particle selection
	char  showMask;			// EMS_TRUE to show the film mask
	char  showZoom;			// EMS_TRUE to show the magnifier
	char  drawTools;		// EMS_PAINT/LINES/ERASE for mask manual drawing
	char  showSpec;			// EMS_TRUE to show spectrum in MONITOR-1
	char  showRefr;			// EMS_TRUE to show template in MONITOR-2
	char  newScale;			// EMS_TRUE for changed film resize factor
	char  syncMask;			// EMS_TRUE for synchronized filmMask[]
	char  mon1X;			// EMS_TRUE to show crosshair on Monitor-I
	char  mon2X;			// EMS_TRUE to show crosshair on Monitor-II
	char  mon3X;			// EMS_TRUE to show crosshair on Monitor-III
	char  filaAve;			// EMS_TRUE to use axial average as filament template
	char  editTemp;			// EMS_TRUE to edit selected reference mask (L1/L2)
	char  activeL2;			// EMS_TRUE to apply level-II masking
};


class EMS_Project {
public:
	// project parameters
	struct EMS_PARAMETER parameter;
	struct EMS_SYSSTATUS sysStatus;
	long inspectID;			// storing current inspection point
	long inspectUD;			// storing last inspection point for undo
	char projSaveStatus;	// EMS_TRUE for having saved

	char projName[NAMESIZE];		// project space name w/o ".ems"
	char filmName[NAMESIZE];		// micrograph filename w/o ".mrc"
	char emfilmFN[NAMESIZE];		// micrograph path+filename
	char templtFN[NAMESIZE];		// template path+filename
	char dirProject[NAMESIZE];		// PROJECT directory
	char dirMcrgraph[NAMESIZE];		// MCRGRAPH directory
	char dirParticle[NAMESIZE];		// PARTICLE directory
	char dirAlignmnt[NAMESIZE];		// CLASSIFY directory
	char dirTemplate[NAMESIZE];		// TEMPLATE directory
	char dirATScreen[NAMESIZE];		// ATSCREEN directory
	char dirModeling[NAMESIZE];		// MODELING directory
	char dirFrealign[NAMESIZE];		// FREALIGN directory

	// input data & essentials
	MRC_MAP micrograph;		// MRC micrograph image (2D input)
	MRC_MAP templates;		// MRC template stack (3D input)
	EMS_MAP film;			// downsized micrograph image (2D)
	EMS_MAP stdMap;			// local s.t.d. of film (2D)
	EMS_MAP std2Map;		// local s.t.d. of lcfMap (2D)
	EMS_MAP filmMask;		// resized film mask (EMS_TRUE to mask out)
	EMS_MAP dispFilm;		// micrograph image for texture mapping (fixed size)
	EMS_MAP dispMask;		// micrograph mask (for texture mapping)

	// templates
	struct EMS_REFERENCE refer;
	EMS_MAP ptk;			// one template (2D)
	EMS_MAP pwr;			// auto-corr of "prtk" (2D)

	// image cropping
	EMS_MAP imgCrop;		// image crop (2D)

	// masks for local statistics
	EMS_MAP lcfMask;		// template masking (2D)
	EMS_MAP scfMask;		// auto-corr. masking (2D)

	// LCF / SCF / DCF signature map
	EMS_MAP lcfMap;			// <film, template> map (2D LCF)
	EMS_MAP scfMap;			// <lcfMap, pwrMap> map (2D SCF)
	EMS_MAP dcfMap;			// lcfMap * scfMap (2D DCF)

	// screening peaks
	EMS_MAP lcfPeak;		// LCF selections
	EMS_MAP scfPeak;		// SCF selections
	EMS_MAP dcfPeak;		// DCF selections
	
	// keeping best score among various templates
	struct EMS_RECORD *record;

	// particle image frame size
	int sqrFrameSize;

	// filament template related
	char cropMode;			// EMS_TRUE for filament cropping
	double filaDiam;		// filament diameter
	double filaUnit;		// filament segment length
	double filaCrox;		// filament segment overlap
	double filaX, filaY;	// initial filament template center
	double filaL, filaW;	// initial filament length and width
	double filaR;			// initial filament orientation
	double filaDX, filaDY;	// filament position shift
	double filaDL, filaDW;	// filament length and width adjustment
	double filaDR;			// filament rotation adjustment
	// axial coordinates of filament selections
	struct EMS_FILAMENT filaAxis[EMS_FILA_MAXM];
	char *filaMask;
	EMS_MAP filaScore;

	EMS_Project();
	~EMS_Project();

	// project management
	char newProjSpace(char *proj_name);
	char loadProjSpace(char *proj_name);

	// particle screening
    char loadProject(char *file_name);
	char saveProject(char *file_name);
	void clearProject(void);

	// micrograph management
	char loadMicrograph(char *file_name);
	char loadMcrgraphSPD(char *file_name);
	char loadMcrgraphTIF(char *file_name);
	char saveMicrograph(char *file_name);
	void clearMicrograph(void);
	void buildFilmMask(void);
	void restoreFilmMask(void);
	void clearFilmMask(void);
	void invContrast(void);
	void clearComRecord(void);
	void clearAllRecord(void);
	void delRecord(void);
	void setDispFilm(void);

	// image processing
	void normalizeImage(void);
	void bandFiltering(void);
	void restoreOriginal(void);
	void imageFlattening(void);
	void pixelBinning(void);

	// synchronize texture display and resized micrograph
	void syncFilmMask(void);

	// template management
	char loadTemplate(char *file_name);
	void focusTemplate(int tid);
	void clearTemplate(void);
	void syncTempMask(void);
	void clearTempMask(void);
	void setIncExcState(char status);

	// particle selection
	void clearParticle(void);
	char loadParticlePTK(char *file_name);
	char loadParticlePLT(char *file_name);
	char loadParticleSPD(char *file_name);
	char saveParticleMRC(char *file_name);
	char saveDisplayMRC(char *file_name);
	char savePreciseMRC(char *file_name);
	void saveParticlePTK(char *file_name);
	void saveParticlePLT(char *file_name);
	void saveParticleSPD(char *file_name);
	char syncPrtkSelect(char *pathName, char *stakName);

	// resize micrograh and template dimentions
	void resizeMcrgraph(void);
	void resizeTemplate(void);

	// rotate & resize each page (2D) in template stack (3D)
	// taking care of template mask at the same time
	void defineTempMask(int frame, char *mask, float rotate);
	void defineTemplate(int frame, float *ptk_t, float *pwr_t, float rotate);

	// compute auto-corr function via FFT
	void calcPowerSpec(void);

	// compute varance map for micrograph
	void calcVariance(void);

	// build lcfMask & scfMask
	void profileMasking(char maskLevel);

	// computer various signature maps and select
	void runProject(char *projFile);
	void testScreen(void);
	void fullScreen(void);
	void screenDST(float threshold);

	// compute / threshold LCF map
    void calcLCF(void);
	void screenLCF(float threshold);

	// compute / threshold SCF map
	void calcSCF(void);
	void screenSCF(float threshold);

	// compute / threshold DCF map
	void calcDCF(void);
	void screenDCF(float threshold);

	// output various intermediate map data
	void writeMap2D(char *file_name, EMS_MAP &signMap);
	void writeMap3D(char *file_name, int x, int y, int z, float *map_data);

	// set up project parameters
	void resetParam(void);

	// for manual selection mode
	float prtkSelect(int x, int y);
	float prtkDelete(int x, int y);
	float prtkCoopSelect(int &x, int &y);

	// inspect particle selections
	void inspectAccept(void);
	void inspectReject(void);
	void inspectUndo(void);

	// record and restore intermediate computations
	char recordScreenMap(void);
	char restoreScreenMap(void);

	// delete ptkStack[] and pwrStack[] stacks
	void delTempStack(void);

	// template definition and filament recognition
	void setFilaModel(int x0, int y0, int x1, int y1, char flagInit);
	void findSegment(void);
	void traceFilament(void);
	void marchAlong(int tid, char flagDirct);
	void clearFilament(void);
};


extern EMS_Project *project;


#endif
