//	James Z. Chen
//	January 1, 2005
//	All rights reserved

#include "panel.h"
#include "project.h"
#include "misc.h"
#include "fftn.h"
#include "gui.h"


extern DisplayPanel_UI *EMS_DP;


EMS_Panel::EMS_Panel()
{
	// setup parameters for monitor display
	resz1 = EMS_MON_RESZ1;
	resz2 = EMS_MON_RESZ2;
	blur1 = EMS_MON_BLUR1;
	blur2 = EMS_MON_BLUR2;
	resz3 = EMS_MON_RESZ3;

	mon1rgb = mon2rgb = mon3rgb = inf_rgb = NULL;
	inf_Img = new unsigned char[EMS_INFO_WIN * EMS_INFO_WIN * 3];
	mon1Img = new unsigned char[EMS_MONTR_SIZE * EMS_MONTR_SIZE * 3];
	mon2Img = new unsigned char[EMS_MONTR_SIZE * EMS_MONTR_SIZE * 3];
	mon3Img = new unsigned char[EMS_MONFX_SIZE * EMS_MONFY_SIZE * 3];
}


EMS_Panel::~EMS_Panel()
{
	if ( mon1rgb != NULL ) delete mon1rgb;
	if ( mon1Img != NULL ) delete [] mon1Img;

	if ( mon2rgb != NULL ) delete mon2rgb;
	if ( mon2Img != NULL ) delete [] mon2Img;

	if ( mon3rgb != NULL ) delete mon3rgb;
	if ( mon3Img != NULL ) delete [] mon3Img;

	if ( inf_rgb != NULL ) delete inf_rgb;
	if ( inf_Img != NULL ) delete [] inf_Img;
}


void EMS_Panel::monitorPrtk(void)
{
	int x0, x1, y0, y1, xx, yy;
	int xp, xp0, xp1, yp, yp0, yp1;
	int dimX, dimY, tmpX, tmpY, blur_t;
	int focusX, focusY, downSize, imgSize;
	long index, count;
	unsigned char val_c;
	float *density, acc, delInt, aveInt, stdInt, patikSize;

	if ( project->micrograph.exist == EMS_FALSE ) return;

	downSize = project->parameter.downSize;
	EMS_DP->imgDisplay->getFocusXY(focusX, focusY);

	if ( (focusX<0) || (focusX>=project->parameter.imgX) ) return;
	if ( (focusY<0) || (focusY>=project->parameter.imgY) ) return;

	dimX = project->micrograph.dimX;
	dimY = project->micrograph.dimY;
	density = project->micrograph.density;

	// normalize the region of interest
	delInt = (1.0 - project->parameter.monCntr) * EMS_HSTG_SIGMA + EMS_ZERO;
	
	imgSize = EMS_MONTR_SIZE * resz1;
	x0 = focusX * downSize - imgSize/2;
	x1 = x0 + imgSize;
	y0 = focusY * downSize - imgSize/2;
	y1 = y0 + imgSize;
	
	aveInt = stdInt = 0.0;
	count = 0;
	
	for ( yy=y0; yy<y1; yy++ ) {
		for ( xx=x0; xx<x1; xx++ ) {
			
			if ( (xx>=0) && (xx<dimX) && (yy>=0) && (yy<dimY) ) {
				index = yy * dimX + xx;
				aveInt += density[index];
				stdInt += density[index] * density[index];
				count ++;
			}
		}
	}
	
	if ( count > 0 ) {
		aveInt /= count;
		stdInt /= count;
		stdInt = stdInt - aveInt * aveInt;

		if ( stdInt > 0.0 )
			stdInt = (float)(sqrt(stdInt));
		else
			stdInt = EMS_ZERO;
	}
	else {
		aveInt = 0.0;
		stdInt = 1.0;
	}

	// display current focus point in monitor-I
	// OR, spectrum of the matched template
	if ( project->sysStatus.showSpec == EMS_FALSE ) {

		index = 0;
		imgSize = EMS_MONTR_SIZE * resz1;
		x0 = focusX * downSize - imgSize/2;
		x1 = x0 + imgSize;
		y0 = focusY * downSize - imgSize/2;
		y1 = y0 + imgSize;

		for ( yy=y0; yy<y1; yy+=resz1 ) {
			for ( xx=x0; xx<x1; xx+=resz1 ) {
				
				if ( (xx<0) || (xx>=dimX) || (yy<0) || (yy>=dimY) ) {
					mon1Img[3 * index + 0] = 0;
					mon1Img[3 * index + 1] = 0;
					mon1Img[3 * index + 2] = EMS_MONTR_BLUE;
					index ++;
					continue;
				}
				
				acc = 0.0;
				blur_t = 0;
				xp0 = xx - blur1;
				xp1 = xx + blur1;
				yp0 = yy - blur1;
				yp1 = yy + blur1;
				
				for ( yp=yp0; yp<=yp1; yp++ ) {
					for ( xp=xp0; xp<=xp1; xp++ ) {
						if ( (xp<0) || (xp>=dimX) || (yp<0) || (yp>=dimY) )
							continue;
						else {
							// acc += density[yp * dimX + xp] * contrast_t;
							acc += density[yp * dimX + xp];
							blur_t ++;
						}
					}
				}

				acc = (acc/blur_t - aveInt) / (stdInt * delInt);
				acc = 255 * (acc + project->parameter.monBrit);
				
				if ( acc < 0 )
					val_c = 0;
				else if ( acc > 255 )
					val_c = 255;
				else
					val_c = (unsigned char)acc;
				
				mon1Img[3 * index + 0] = val_c;
				mon1Img[3 * index + 1] = val_c;
				mon1Img[3 * index + 2] = val_c;
				index ++;
			}
		}
		
		// overlay cross-hair if required
		if ( project->sysStatus.mon1X == EMS_TRUE ) {

			patikSize = project->parameter.prtkSize / (project->parameter.pixelSize * resz1);
			imgSize = (int)(EMS_FOCUS_RAD * patikSize);
			
			// horizontal axis
			x0 = (EMS_MONTR_SIZE - imgSize)/2;
			x1 = x0 + imgSize;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			for ( xx=x0; xx<=x1; xx++ ) {
				index = EMS_MONTR_SIZE * EMS_MONTR_SIZE / 2 + xx;
				mon1Img[3 * index + 0] = EMS_CROSS_R;
				mon1Img[3 * index + 1] = EMS_CROSS_G;
				mon1Img[3 * index + 2] = EMS_CROSS_B;
			}
			
			// vertical axis
			y0 = (EMS_MONTR_SIZE - imgSize)/2;
			y1 = y0 + imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			for ( yy=y0; yy<=y1; yy++ ) {
				index = yy * EMS_MONTR_SIZE + EMS_MONTR_SIZE / 2;
				mon1Img[3 * index + 0] = EMS_CROSS_R;
				mon1Img[3 * index + 1] = EMS_CROSS_G;
				mon1Img[3 * index + 2] = EMS_CROSS_B;
			}
			
			// two boundary bars on the horizontal axis
			imgSize = (int)(patikSize * EMS_CROSS_BAR);
			y0 = (EMS_MONTR_SIZE - imgSize) / 2;
			y1 = y0 + imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			imgSize = (int)patikSize / 2;
			for ( yy=y0; yy<=y1; yy++ ) {
				xx = EMS_MONTR_SIZE/2 - imgSize;
				if ( (xx>=0) && (xx<EMS_MONTR_SIZE) ) {
					index = yy * EMS_MONTR_SIZE + xx;
					mon1Img[3 * index + 0] = EMS_CROSS_R;
					mon1Img[3 * index + 1] = EMS_CROSS_G;
					mon1Img[3 * index + 2] = EMS_CROSS_B;
				}

				xx = EMS_MONTR_SIZE/2 + imgSize;
				if ( (xx>=0) && (xx<EMS_MONTR_SIZE) ) {
					index = yy * EMS_MONTR_SIZE + xx;
					mon1Img[3 * index + 0] = EMS_CROSS_R;
					mon1Img[3 * index + 1] = EMS_CROSS_G;
					mon1Img[3 * index + 2] = EMS_CROSS_B;
				}
			}
			
			// two boundary bars on the vertical axis
			imgSize = (int)(patikSize * EMS_CROSS_BAR);
			x0 = (EMS_MONTR_SIZE - imgSize) / 2;
			x1 = x0 + imgSize;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			imgSize = (int)patikSize / 2;
			for ( xx=x0; xx<=x1; xx++ ) {
				yy = EMS_MONTR_SIZE/2 - imgSize;
				if ( (yy>=0) && (yy<EMS_MONTR_SIZE) ) {
					index = yy * EMS_MONTR_SIZE + xx;
					mon1Img[3 * index + 0] = EMS_CROSS_R;
					mon1Img[3 * index + 1] = EMS_CROSS_G;
					mon1Img[3 * index + 2] = EMS_CROSS_B;
				}

				yy = EMS_MONTR_SIZE/2 + imgSize;
				if ( (yy>=0) && (yy<EMS_MONTR_SIZE) ) {
					index = yy * EMS_MONTR_SIZE + xx;
					mon1Img[3 * index + 0] = EMS_CROSS_R;
					mon1Img[3 * index + 1] = EMS_CROSS_G;
					mon1Img[3 * index + 2] = EMS_CROSS_B;
				}
			}
			
			// marking four corners of template-dimension
			tmpX = project->templates.dimX / resz1;
			tmpY = project->templates.dimY / resz1;
			imgSize = (int)((tmpX + tmpY) * EMS_BOUND_BAR / 2);
			
			y0 = (EMS_MONTR_SIZE - tmpY) / 2;
			y1 = (EMS_MONTR_SIZE + tmpY) / 2;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			x0 = (EMS_MONTR_SIZE - tmpX) / 2;
			x1 = x0 + imgSize;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			for ( xx=x0; xx<=x1; xx++ ) {
				index = y0 * EMS_MONTR_SIZE + xx;
				mon1Img[3 * index + 0] = EMS_BOUND_R;
				mon1Img[3 * index + 1] = EMS_BOUND_G;
				mon1Img[3 * index + 2] = EMS_BOUND_B;
				index = y1 * EMS_MONTR_SIZE + xx;
				mon1Img[3 * index + 0] = EMS_BOUND_R;
				mon1Img[3 * index + 1] = EMS_BOUND_G;
				mon1Img[3 * index + 2] = EMS_BOUND_B;
			}
			
			x1 = (EMS_MONTR_SIZE + tmpX) / 2;
			x0 = x1 - imgSize;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			for ( xx=x0; xx<=x1; xx++ ) {
				index = y0 * EMS_MONTR_SIZE + xx;
				mon1Img[3 * index + 0] = EMS_BOUND_R;
				mon1Img[3 * index + 1] = EMS_BOUND_G;
				mon1Img[3 * index + 2] = EMS_BOUND_B;
				index = y1 * EMS_MONTR_SIZE + xx;
				mon1Img[3 * index + 0] = EMS_BOUND_R;
				mon1Img[3 * index + 1] = EMS_BOUND_G;
				mon1Img[3 * index + 2] = EMS_BOUND_B;
			}
			
			x0 = (EMS_MONTR_SIZE - tmpY) / 2;
			x1 = (EMS_MONTR_SIZE + tmpY) / 2;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			y0 = (EMS_MONTR_SIZE - tmpX) / 2;
			y1 = y0 + imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			for ( yy=y0; yy<=y1; yy++ ) {
				index = yy * EMS_MONTR_SIZE + x0;
				mon1Img[3 * index + 0] = EMS_BOUND_R;
				mon1Img[3 * index + 1] = EMS_BOUND_G;
				mon1Img[3 * index + 2] = EMS_BOUND_B;
				index = yy * EMS_MONTR_SIZE + x1;
				mon1Img[3 * index + 0] = EMS_BOUND_R;
				mon1Img[3 * index + 1] = EMS_BOUND_G;
				mon1Img[3 * index + 2] = EMS_BOUND_B;
			}
			
			y1 = (EMS_MONTR_SIZE + tmpX) / 2;
			y0 = y1 - imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			for ( yy=y0; yy<=y1; yy++ ) {
				index = yy * EMS_MONTR_SIZE + x0;
				mon1Img[3 * index + 0] = EMS_BOUND_R;
				mon1Img[3 * index + 1] = EMS_BOUND_G;
				mon1Img[3 * index + 2] = EMS_BOUND_B;
				index = yy * EMS_MONTR_SIZE + x1;
				mon1Img[3 * index + 0] = EMS_BOUND_R;
				mon1Img[3 * index + 1] = EMS_BOUND_G;
				mon1Img[3 * index + 2] = EMS_BOUND_B;
			}
		}
	
		if ( mon1rgb != NULL ) delete mon1rgb;
		mon1rgb = new Fl_RGB_Image(mon1Img, EMS_MONTR_SIZE, EMS_MONTR_SIZE, 3, 0);
		EMS_DP->monitor1->image(mon1rgb);
		EMS_DP->monitor1->redraw();
		Fl::flush();
	}
	else {
		monitorSpec();
	}

	// display current focus point in monitor-II
	// OR, the matched template
	if ( project->sysStatus.showRefr == EMS_FALSE ) {
		
		index = 0;
		imgSize = EMS_MONTR_SIZE * resz2;
		x0 = focusX * downSize - imgSize/2;
		x1 = x0 + imgSize;
		y0 = focusY * downSize - imgSize/2;
		y1 = y0 + imgSize;

		for ( yy=y0; yy<y1; yy+=resz2 ) {
			for ( xx=x0; xx<x1; xx+=resz2 ) {
				
				if ( (xx<0) || (xx>=dimX) || (yy<0) || (yy>=dimY) ) {
					mon2Img[3 * index + 0] = 0;
					mon2Img[3 * index + 1] = 0;
					mon2Img[3 * index + 2] = EMS_MONTR_BLUE;
					index ++;
					continue;
				}
				
				acc = 0.0;
				blur_t = 0;
				xp0 = xx - blur2;
				xp1 = xx + blur2;
				yp0 = yy - blur2;
				yp1 = yy + blur2;
				
				for ( yp=yp0; yp<=yp1; yp++ ) {
					for ( xp=xp0; xp<=xp1; xp++ ) {
						if ( (xp<0) || (xp>=dimX) || (yp<0) || (yp>=dimY) )
							continue;
						else {
							// acc += density[yp * dimX + xp] * contrast_t;
							acc += density[yp * dimX + xp];
							blur_t ++;
						}
					}
				}

				acc = (acc/blur_t - aveInt) / (stdInt * delInt);
				acc = 255 * (acc + project->parameter.monBrit);

				if ( acc < 0 )
					val_c = 0;
				else if ( acc > 255 )
					val_c = 255;
				else
					val_c = (unsigned char)acc;
				
				mon2Img[3 * index + 0] = val_c;
				mon2Img[3 * index + 1] = val_c;
				mon2Img[3 * index + 2] = val_c;
				index ++;
			}
		}
		
		// overlay cross-hair if required
		if ( project->sysStatus.mon2X == EMS_TRUE ) {

			patikSize = project->parameter.prtkSize / (project->parameter.pixelSize * resz2);
			imgSize = (int)(EMS_FOCUS_RAD * patikSize);
			
			// horizontal axis
			x0 = (EMS_MONTR_SIZE - imgSize)/2;
			x1 = x0 + imgSize;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			for ( xx=x0; xx<=x1; xx++ ) {
				index = EMS_MONTR_SIZE * EMS_MONTR_SIZE / 2 + xx;
				mon2Img[3 * index + 0] = EMS_CROSS_R;
				mon2Img[3 * index + 1] = EMS_CROSS_G;
				mon2Img[3 * index + 2] = EMS_CROSS_B;
			}
			
			// vertical axis
			y0 = (EMS_MONTR_SIZE - imgSize)/2;
			y1 = y0 + imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			for ( yy=y0; yy<=y1; yy++ ) {
				index = yy * EMS_MONTR_SIZE + EMS_MONTR_SIZE / 2;
				mon2Img[3 * index + 0] = EMS_CROSS_R;
				mon2Img[3 * index + 1] = EMS_CROSS_G;
				mon2Img[3 * index + 2] = EMS_CROSS_B;
			}
			
			// two boundary bars on the horizontal axis
			imgSize = (int)(patikSize * EMS_CROSS_BAR);
			y0 = (EMS_MONTR_SIZE - imgSize) / 2;
			y1 = y0 + imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			imgSize = (int)patikSize / 2;
			for ( yy=y0; yy<=y1; yy++ ) {
				xx = EMS_MONTR_SIZE/2 - imgSize;
				if ( (xx>=0) && (xx<EMS_MONTR_SIZE) ) {
					index = yy * EMS_MONTR_SIZE + xx;
					mon2Img[3 * index + 0] = EMS_CROSS_R;
					mon2Img[3 * index + 1] = EMS_CROSS_G;
					mon2Img[3 * index + 2] = EMS_CROSS_B;
				}

				xx = EMS_MONTR_SIZE/2 + imgSize;
				if ( (xx>=0) && (xx<EMS_MONTR_SIZE) ) {
					index = yy * EMS_MONTR_SIZE + xx;
					mon2Img[3 * index + 0] = EMS_CROSS_R;
					mon2Img[3 * index + 1] = EMS_CROSS_G;
					mon2Img[3 * index + 2] = EMS_CROSS_B;
				}
			}
			
			// two boundary bars on the vertical axis
			imgSize = (int)(patikSize * EMS_CROSS_BAR);
			x0 = (EMS_MONTR_SIZE - imgSize) / 2;
			x1 = x0 + imgSize;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			imgSize = (int)patikSize / 2;
			for ( xx=x0; xx<=x1; xx++ ) {
				yy = EMS_MONTR_SIZE/2 - imgSize;
				if ( (yy>=0) && (yy<EMS_MONTR_SIZE) ) {
					index = yy * EMS_MONTR_SIZE + xx;
					mon2Img[3 * index + 0] = EMS_CROSS_R;
					mon2Img[3 * index + 1] = EMS_CROSS_G;
					mon2Img[3 * index + 2] = EMS_CROSS_B;
				}

				yy = EMS_MONTR_SIZE/2 + imgSize;
				if ( (yy>=0) && (yy<EMS_MONTR_SIZE) ) {
					index = yy * EMS_MONTR_SIZE + xx;
					mon2Img[3 * index + 0] = EMS_CROSS_R;
					mon2Img[3 * index + 1] = EMS_CROSS_G;
					mon2Img[3 * index + 2] = EMS_CROSS_B;
				}
			}
			
			// marking four corners of template-dimension
			tmpX = project->templates.dimX / resz2;
			tmpY = project->templates.dimY / resz2;
			imgSize = (int)((tmpX + tmpY) * EMS_BOUND_BAR / 2);
			
			y0 = (EMS_MONTR_SIZE - tmpY) / 2;
			y1 = (EMS_MONTR_SIZE + tmpY) / 2;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			x0 = (EMS_MONTR_SIZE - tmpX) / 2;
			x1 = x0 + imgSize;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			for ( xx=x0; xx<=x1; xx++ ) {
				index = y0 * EMS_MONTR_SIZE + xx;
				mon2Img[3 * index + 0] = EMS_BOUND_R;
				mon2Img[3 * index + 1] = EMS_BOUND_G;
				mon2Img[3 * index + 2] = EMS_BOUND_B;
				index = y1 * EMS_MONTR_SIZE + xx;
				mon2Img[3 * index + 0] = EMS_BOUND_R;
				mon2Img[3 * index + 1] = EMS_BOUND_G;
				mon2Img[3 * index + 2] = EMS_BOUND_B;
			}
			
			x1 = (EMS_MONTR_SIZE + tmpX) / 2;
			x0 = x1 - imgSize;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			for ( xx=x0; xx<=x1; xx++ ) {
				index = y0 * EMS_MONTR_SIZE + xx;
				mon2Img[3 * index + 0] = EMS_BOUND_R;
				mon2Img[3 * index + 1] = EMS_BOUND_G;
				mon2Img[3 * index + 2] = EMS_BOUND_B;
				index = y1 * EMS_MONTR_SIZE + xx;
				mon2Img[3 * index + 0] = EMS_BOUND_R;
				mon2Img[3 * index + 1] = EMS_BOUND_G;
				mon2Img[3 * index + 2] = EMS_BOUND_B;
			}
			
			x0 = (EMS_MONTR_SIZE - tmpY) / 2;
			x1 = (EMS_MONTR_SIZE + tmpY) / 2;
			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONTR_SIZE ) x1 = EMS_MONTR_SIZE - 1;
			
			y0 = (EMS_MONTR_SIZE - tmpX) / 2;
			y1 = y0 + imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			for ( yy=y0; yy<=y1; yy++ ) {
				index = yy * EMS_MONTR_SIZE + x0;
				mon2Img[3 * index + 0] = EMS_BOUND_R;
				mon2Img[3 * index + 1] = EMS_BOUND_G;
				mon2Img[3 * index + 2] = EMS_BOUND_B;
				index = yy * EMS_MONTR_SIZE + x1;
				mon2Img[3 * index + 0] = EMS_BOUND_R;
				mon2Img[3 * index + 1] = EMS_BOUND_G;
				mon2Img[3 * index + 2] = EMS_BOUND_B;
			}
			
			y1 = (EMS_MONTR_SIZE + tmpX) / 2;
			y0 = y1 - imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONTR_SIZE ) y1 = EMS_MONTR_SIZE - 1;
			
			for ( yy=y0; yy<=y1; yy++ ) {
				index = yy * EMS_MONTR_SIZE + x0;
				mon2Img[3 * index + 0] = EMS_BOUND_R;
				mon2Img[3 * index + 1] = EMS_BOUND_G;
				mon2Img[3 * index + 2] = EMS_BOUND_B;
				index = yy * EMS_MONTR_SIZE + x1;
				mon2Img[3 * index + 0] = EMS_BOUND_R;
				mon2Img[3 * index + 1] = EMS_BOUND_G;
				mon2Img[3 * index + 2] = EMS_BOUND_B;
			}
		}
			
		if ( mon2rgb != NULL ) delete mon2rgb;
		mon2rgb = new Fl_RGB_Image(mon2Img, EMS_MONTR_SIZE, EMS_MONTR_SIZE, 3, 0);
		EMS_DP->monitor2->image(mon2rgb);
		EMS_DP->monitor2->redraw();
		Fl::flush();
	}
	else {
		index = focusY * project->parameter.imgX + focusX;
		monitorTemp(project->record[index].tmpid);
	}
}


void EMS_Panel::monitorFila(float *filaImg, int length, int width)
{
	int x0, x1, y0, y1, xx, yy;
	int xp, xp0, xp1, yp, yp0, yp1;
	int blur_t;
	long index, imgSize, count;
	static int filaLt, filaWt;
	static float *density=NULL;
	double acc, delInt, aveInt, stdInt, patikSize;
	char accChar;

	if ( project->imgCrop.exist == EMS_FALSE ) {

		for ( index=0; index<(EMS_MONFX_SIZE*EMS_MONFY_SIZE); index++ ) {
			mon3Img[3 * index + 0] = 0;
			mon3Img[3 * index + 1] = 0;
			mon3Img[3 * index + 2] = EMS_MONTR_BLUE;
		}

		if ( mon3rgb != NULL ) delete mon3rgb;
		mon3rgb = new Fl_RGB_Image(mon3Img, EMS_MONFX_SIZE, EMS_MONFY_SIZE, 3, 0);
		EMS_DP->monitor3->image(mon3rgb);
		EMS_DP->monitor3->redraw();

		EMS_DP->dspCropX->value(0);
		EMS_DP->dspCropY->value(0);
		EMS_DP->valShftX->value(0);
		EMS_DP->valShftY->value(0);
		EMS_DP->valExtnX->value(0);
		EMS_DP->valExtnY->value(0);
		EMS_DP->valRotate->value(0);

		return;
	}

	if ( filaImg != NULL ) {
		if ( density != NULL ) delete [] density;

		filaLt = length;
		filaWt = width;
		imgSize = filaLt * filaWt;
		EMS_DP->dspCropX->value(length);
		EMS_DP->dspCropY->value(width);

		density = new float[imgSize];
		memcpy(density, filaImg, imgSize*sizeof(float));
	}

	// display filament template definition in monitor-III
	delInt = (1.0 - project->parameter.mon3Cntr) * EMS_HSTG_SIGMA + EMS_ZERO;

	if ( EMS_TRUE ) {

		x0 = (filaLt - EMS_MONFX_SIZE * resz3) / 2;
		x1 = x0 + EMS_MONFX_SIZE * resz3;
		y0 = (filaWt - EMS_MONFY_SIZE * resz3) / 2;
		y1 = y0 + EMS_MONFY_SIZE * resz3;
		
		// normalize the region of interest
		aveInt = stdInt = 0.0;
		count = 0;

		for ( yy=y0; yy<y1; yy+=resz3 ) {
			for ( xx=x0; xx<x1; xx+=resz3 ) {
				
				if ( (xx>=0) && (xx<filaLt) && (yy>=0) && (yy<filaWt) ) {
					index = yy * filaLt + xx;
					aveInt += density[index];
					stdInt += density[index] * density[index];
					count ++;
				}
			}
		}

		if ( count > 0 ) {
			aveInt /= count;
			stdInt /= count;
			stdInt = stdInt - aveInt * aveInt;

			if ( stdInt > 0.0 )
				stdInt = (float)(sqrt(stdInt));
			else
				stdInt = EMS_ZERO;
		}
		else {
			aveInt = 0.0;
			stdInt = 1.0;
		}

		// accumulate pixel values
		index = 0;
		for ( yy=y0; yy<y1; yy+=resz3 ) {
			for ( xx=x0; xx<x1; xx+=resz3 ) {
				
				if ( (xx<0) || (xx>=filaLt) || (yy<0) || (yy>=filaWt) ) {
					mon3Img[3 * index + 0] = 0;
					mon3Img[3 * index + 1] = 0;
					mon3Img[3 * index + 2] = EMS_MONTR_BLUE;
					index ++;
					continue;
				}
				
				acc = 0.0;
				blur_t = 0;
				xp0 = xx - resz3/2;
				xp1 = xp0 + resz3;
				yp0 = yy - resz3/2;
				yp1 = yp0 + resz3;

				for ( yp=yp0; yp<=yp1; yp++ ) {
					for ( xp=xp0; xp<=xp1; xp++ ) {
						if ( (xp<0) || (xp>=filaLt) || (yp<0) || (yp>=filaWt) )
							continue;
						else {
							// acc += density[yp * filaLt + xp] * contrast_t;
							acc += density[yp * filaLt + xp];
							blur_t ++;
						}
					}
				}

				acc = (acc/blur_t - aveInt) / (stdInt * delInt);
				acc = 255.0 * (acc + project->parameter.mon3Brit);
				
				if ( acc < 0 )
					mon3Img[3*index+0] = 0;
				else if ( acc > 255 )
					mon3Img[3*index+0] = 255;
				else
					mon3Img[3*index+0] = (unsigned char)acc;
				
				mon3Img[3*index+2] = mon3Img[3*index+1] = mon3Img[3*index+0];
				index ++;
			}
		}

		// calculate axial average of the selected filament template
		if ( (project->cropMode==EMS_TRUE) && (project->sysStatus.filaAve==EMS_TRUE) ) {

			x0 = (EMS_MONFX_SIZE - filaLt/resz3) / 2;
			x1 = x0 + filaLt/resz3;
			y0 = (EMS_MONFY_SIZE - filaWt/resz3) / 2;
			y1 = y0 + filaWt/resz3;

			if ( x0 < 0 ) x0 = 0;
			if ( x1 >= EMS_MONFX_SIZE ) x1 = EMS_MONFX_SIZE - 1;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONFY_SIZE ) y1 = EMS_MONFY_SIZE - 1;

			for ( yy=y0; yy<=y1; yy++ ) {

				acc = 0.0;
				for ( xx=x0; xx<=x1; xx++ )
					acc += mon3Img[3*(yy*EMS_MONFX_SIZE+xx)];

				accChar = (char)(acc / (x1-x0));
				for ( xx=x0; xx<=x1; xx++ ) {
					index = 3 * (yy * EMS_MONFX_SIZE + xx);
					mon3Img[index] = mon3Img[index+1] = mon3Img[index+2] = accChar;
				}
			}
		}
		
		// overlay cross-hair if required
		if ( project->sysStatus.mon3X == EMS_TRUE ) {

			patikSize = project->filaDiam / (project->parameter.pixelSize * resz3);
			imgSize = (int)(EMS_FOCUS_RAD * patikSize);
			
			// horizontal axis
			for ( xx=0; xx<EMS_MONFX_SIZE; xx++ ) {
				index = EMS_MONFY_SIZE * EMS_MONFX_SIZE / 2 + xx;
				mon3Img[3 * index + 0] = EMS_CROSS_R;
				mon3Img[3 * index + 1] = EMS_CROSS_G;
				mon3Img[3 * index + 2] = EMS_CROSS_B;
			}
			
			// vertical axis
			y0 = (EMS_MONFY_SIZE - imgSize)/2;
			y1 = y0 + imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONFY_SIZE ) y1 = EMS_MONFY_SIZE - 1;
			
			for ( yy=y0; yy<=y1; yy++ ) {
				index = yy * EMS_MONFX_SIZE + EMS_MONFX_SIZE / 2;
				mon3Img[3 * index + 0] = EMS_CROSS_R;
				mon3Img[3 * index + 1] = EMS_CROSS_G;
				mon3Img[3 * index + 2] = EMS_CROSS_B;
			}
			
			// measuring bars on the horizontal axis
			imgSize = (int)(patikSize * EMS_CROSS_BAR);
			y0 = (EMS_MONFY_SIZE - imgSize) / 2;
			y1 = y0 + imgSize;
			if ( y0 < 0 ) y0 = 0;
			if ( y1 >= EMS_MONFY_SIZE ) y1 = EMS_MONFY_SIZE - 1;
			
			imgSize = (int)patikSize / 2;
			for ( yy=y0; yy<=y1; yy++ ) {
				xx = EMS_MONFX_SIZE/2 - imgSize;
				if ( (xx>=0) && (xx<EMS_MONFX_SIZE) ) {
					index = yy * EMS_MONFX_SIZE + xx;
					mon3Img[3 * index + 0] = EMS_CROSS_R;
					mon3Img[3 * index + 1] = EMS_CROSS_G;
					mon3Img[3 * index + 2] = EMS_CROSS_B;
				}

				xx = EMS_MONFX_SIZE/2 + imgSize;
				if ( (xx>=0) && (xx<EMS_MONFX_SIZE) ) {
					index = yy * EMS_MONFX_SIZE + xx;
					mon3Img[3 * index + 0] = EMS_CROSS_R;
					mon3Img[3 * index + 1] = EMS_CROSS_G;
					mon3Img[3 * index + 2] = EMS_CROSS_B;
				}

				xx = EMS_MONFX_SIZE/2 - 3*imgSize;
				if ( (xx>=0) && (xx<EMS_MONFX_SIZE) ) {
					index = yy * EMS_MONFX_SIZE + xx;
					mon3Img[3 * index + 0] = EMS_CROSS_R;
					mon3Img[3 * index + 1] = EMS_CROSS_G;
					mon3Img[3 * index + 2] = EMS_CROSS_B;
				}

				xx = EMS_MONFX_SIZE/2 + 3*imgSize;
				if ( (xx>=0) && (xx<EMS_MONFX_SIZE) ) {
					index = yy * EMS_MONFX_SIZE + xx;
					mon3Img[3 * index + 0] = EMS_CROSS_R;
					mon3Img[3 * index + 1] = EMS_CROSS_G;
					mon3Img[3 * index + 2] = EMS_CROSS_B;
				}

				xx = EMS_MONFX_SIZE/2 - 5*imgSize;
				if ( (xx>=0) && (xx<EMS_MONFX_SIZE) ) {
					index = yy * EMS_MONFX_SIZE + xx;
					mon3Img[3 * index + 0] = EMS_CROSS_R;
					mon3Img[3 * index + 1] = EMS_CROSS_G;
					mon3Img[3 * index + 2] = EMS_CROSS_B;
				}

				xx = EMS_MONFX_SIZE/2 + 5*imgSize;
				if ( (xx>=0) && (xx<EMS_MONFX_SIZE) ) {
					index = yy * EMS_MONFX_SIZE + xx;
					mon3Img[3 * index + 0] = EMS_CROSS_R;
					mon3Img[3 * index + 1] = EMS_CROSS_G;
					mon3Img[3 * index + 2] = EMS_CROSS_B;
				}
			}
			
			// two boundary bars on the vertical axis
			x0 = EMS_MONFX_SIZE / 4;
			x1 = EMS_MONFX_SIZE * 3 / 4;
			
			imgSize = (int)patikSize / 2;
			for ( xx=x0; xx<=x1; xx++ ) {
				yy = EMS_MONFY_SIZE/2 - imgSize;
				if ( (yy>=0) && (yy<EMS_MONFY_SIZE) ) {
					index = yy * EMS_MONFX_SIZE + xx;
					mon3Img[3 * index + 0] = EMS_CROSS_R;
					mon3Img[3 * index + 1] = EMS_CROSS_G;
					mon3Img[3 * index + 2] = EMS_CROSS_B;
				}

				yy = EMS_MONFY_SIZE/2 + imgSize;
				if ( (yy>=0) && (yy<EMS_MONFY_SIZE) ) {
					index = yy * EMS_MONFX_SIZE + xx;
					mon3Img[3 * index + 0] = EMS_CROSS_R;
					mon3Img[3 * index + 1] = EMS_CROSS_G;
					mon3Img[3 * index + 2] = EMS_CROSS_B;
				}
			}
		}
	
		if ( mon3rgb != NULL ) delete mon3rgb;
		mon3rgb = new Fl_RGB_Image(mon3Img, EMS_MONFX_SIZE, EMS_MONFY_SIZE, 3, 0);
		EMS_DP->monitor3->image(mon3rgb);
		EMS_DP->monitor3->redraw();
		Fl::flush();
	}
}


void EMS_Panel::monitorSpec(void)
{
	int x, y, x0, y0, x1, y1, dimX, dimY;
	static int dimen[2];
	long index, dsize;
	float *rel_t, *img_t;
	float valInt, aveInt, varInt, stdInt, delInt;
	EMS_MAP origSpec, averSpec;
	
	if ( project->micrograph.exist == EMS_FALSE ) return;
	
	EMS_DP->imgDisplay->getFocusXY(x, y);
	x *= project->parameter.downSize;
	y *= project->parameter.downSize;
	dimX = project->micrograph.dimX;
	dimY = project->micrograph.dimY;
	
	dimen[0] = (int)(2.0 * project->parameter.prtkSize / project->parameter.pixelSize);
	dimen[1] = (int)(2.0 * project->parameter.prtkSize / project->parameter.pixelSize);
	dsize = dimen[0] * dimen[1];
	
	x0 = x - dimen[0]/2;
	x1 = x0 + dimen[0];
	y0 = y - dimen[1]/2;
	y1 = y0 + dimen[1];

	if ( (x0<0) || (x1>=dimX) || (y0<0) || (y1>=dimY) ) {
		msgReport("This particle is too close to the image boundary");
		
		for ( index=0; index<(EMS_MONTR_SIZE*EMS_MONTR_SIZE); index++ ) {
			mon1Img[3*index+0] = 0;
			mon1Img[3*index+1] = 0;
			mon1Img[3*index+2] = EMS_MONTR_BLUE;
		}
	}
	else {
		systemStatus(EMS_BUSY);
		
		index = 0;
		rel_t = new float[dsize];
		img_t = new float[dsize];
		
		for ( y=y0; y<y1; y++ ) {
			for ( x=x0; x<x1; x++ ) {
				if ( (x0<0) || (x1>=dimX) || (y0<0) || (y1>=dimY) )
					rel_t[index] = 0.0;
				else
					rel_t[index] = project->micrograph.density[y*dimX+x];
				
				img_t[index] = 0.0;
				index ++;
			}
		}
		
		// compute, smooth and interpolate the spectrum
		fftwN(2, dimen, rel_t, img_t, FFTW_FORWARD);
		
		for ( index=0; index<dsize; index++ ) {
			valInt = rel_t[index] * rel_t[index] + img_t[index] * img_t[index];
			rel_t[index] = sqrt(valInt);
			// rel_t[index] = log(valInt + 1.0);
		}
		
		origSpec.setup(rel_t, dimen[0], dimen[1], 0, 0);
		averSpec.initF(0.0, EMS_MONTR_SIZE, EMS_MONTR_SIZE);
		smoothSpectrum(origSpec, averSpec, EMS_SPEC_SMOT);
		
		// determine average & variance of the image
		aveInt = varInt = stdInt = delInt = 0.0;
		dsize = EMS_MONTR_SIZE * EMS_MONTR_SIZE;
		
		for ( index=0; index<dsize; index++ ) {
			aveInt += averSpec.dataF[index];
			varInt += averSpec.dataF[index] * averSpec.dataF[index];
		}
		
		aveInt /= dsize;
		varInt /= dsize;
		stdInt = varInt - aveInt * aveInt;

		if ( stdInt > 0.0 )
			stdInt = sqrt(stdInt);
		else
			stdInt = 0.0;

		delInt = (1.0 - project->parameter.spcCntr) * stdInt * EMS_SPEC_SIGMA + EMS_ZERO;
		
		// assign RGB for image display
		for ( index=0; index<dsize; index++ ) {
			valInt = 255.0 * ((averSpec.dataF[index] - aveInt) / delInt + project->parameter.spcBrit);
			
			if ( valInt < 0 )
				mon1Img[3*index+0] = (unsigned char)0;
			else if ( valInt > 255 )
				mon1Img[3*index+0] = (unsigned char)255;
			else
				mon1Img[3*index+0] = (unsigned char)(valInt);
			
			mon1Img[3*index+2] = mon1Img[3*index+1] = mon1Img[3*index+0];
		}
		
		delete [] rel_t;
		delete [] img_t;
		
		systemStatus(EMS_FREE);
	}
	
	if ( mon1rgb != NULL ) delete mon1rgb;
	mon1rgb = new Fl_RGB_Image(mon1Img, EMS_MONTR_SIZE, EMS_MONTR_SIZE, 3, 0);
	EMS_DP->monitor1->image(mon1rgb);
	EMS_DP->monitor1->redraw();
	Fl::flush();
}


void EMS_Panel::monitorTemp(int n)
{
	int xx, yy, dimX, dimY;
	int x0, x1, y0, y1;
	int xp, xp0, xp1, yp, yp0, yp1;
	int blur3, imgSize, blur_t;
	unsigned char val_c;
	long index;
	float *density, acc, ave, var, std, delInt;

	if ( (n<0) || (n>=project->parameter.tmpN) ) {
		msgReport("Best matching template is unknown for this particle");
		
		for ( index=0; index<(EMS_MONTR_SIZE*EMS_MONTR_SIZE); index++ ) {
			mon2Img[3*index+0] = 0;
			mon2Img[3*index+1] = 0;
			mon2Img[3*index+2] = EMS_MONTR_BLUE;
		}
	}
	else {
		msgReport("\0");
		dimX = project->templates.dimX;
		dimY = project->templates.dimY;
		
		// adjust intensity contrast of the template
		ave = var = std = delInt = 0.0;
		imgSize = dimX * dimY;
		density = project->templates.density + imgSize * n;
		
		for ( index=0; index<imgSize; index++ ) {
			ave += density[index];
			var += density[index] * density[index];
		}
		
		ave /= imgSize;
		var /= imgSize;
		std = var - ave * ave;

		if ( std > 0.0 )
			std = sqrt(std);
		else
			std = 0.0;

		delInt = (1.0 - project->parameter.refCntr) * std * EMS_HSTG_SIGMA + EMS_ZERO;
		
		// set up image display
		index = 0;
		blur3 = resz2 / 2;
		imgSize = EMS_MONTR_SIZE * resz2;
		
		x0 = (dimX - imgSize) / 2;
		x1 = x0 + imgSize;
		y0 = (dimY - imgSize) / 2;
		y1 = y0 + imgSize;
		
		for ( yy=y0; yy<y1; yy+=resz2 ) {
			for ( xx=x0; xx<x1; xx+=resz2 ) {
				
				if ( (xx<0) || (xx>=dimX) || (yy<0) || (yy>=dimY) ) {
					mon2Img[3 * index + 0] = 0;
					mon2Img[3 * index + 1] = 0;
					mon2Img[3 * index + 2] = EMS_MONTR_BLUE;
					index ++;
					continue;
				}
				
				acc = 0.0;
				blur_t = 0;
				xp0 = xx - blur3;
				xp1 = xx + blur3;
				yp0 = yy - blur3;
				yp1 = yy + blur3;
				
				for ( yp=yp0; yp<=yp1; yp++ ) {
					for ( xp=xp0; xp<=xp1; xp++ ) {
						if ( (xp<0) || (xp>=dimX) || (yp<0) || (yp>=dimY) )
							continue;
						else {
							acc += density[yp * dimX + xp];
							blur_t ++;
						}
					}
				}
				
				acc = 255 * ( acc / (blur_t * delInt) + project->parameter.refBrit);
				
				if ( acc < 0 )
					val_c = 0;
				else if ( acc > 255 )
					val_c = 255;
				else
					val_c = (unsigned char)acc;
				
				mon2Img[3 * index + 0] = val_c;
				mon2Img[3 * index + 1] = val_c;
				mon2Img[3 * index + 2] = val_c;
				index ++;
			}
		}
	}

	if ( mon2rgb != NULL ) delete mon2rgb;
	mon2rgb = new Fl_RGB_Image(mon2Img, EMS_MONTR_SIZE, EMS_MONTR_SIZE, 3, 0);
	EMS_DP->monitor2->image(mon2rgb);
	EMS_DP->monitor2->redraw();
	Fl::flush();
}


void EMS_Panel::redrawTemp(void)
{
	if ( project->templates.exist == EMS_TRUE ) {
		if ( (project->record!=NULL) && (project->inspectID>=0) )
			monitorTemp(project->record[project->inspectID].tmpid);
	}
}


void EMS_Panel::redrawSpec(void)
{
	if ( project->micrograph.exist == EMS_TRUE )
		monitorSpec();
}


void EMS_Panel::redrawFocus(void)
{
	if ( project->micrograph.exist == EMS_TRUE )
		monitorPrtk();
}


void EMS_Panel::drawImgInfo(char flagRerun)
{
	int x, y, x0, y0, x1, y1, upto, maxCount=0;
	int dimX, dimY;
	static int dimen[2];
	long index, dsize;
	float *rel_t, *img_t;
	float valInt, aveInt, varInt, stdInt, delInt;
	EMS_MAP origSpec;
	static EMS_MAP averSpec;
	
	msgReport("\0");

	// safe-guard data availability
	if ( project->micrograph.exist == EMS_FALSE ) return;
	
	for ( x=0; x<(EMS_INFO_WIN * EMS_INFO_WIN); x++ ) {
		inf_Img[3*x+0] = EMS_INFO_BKGD;
		inf_Img[3*x+1] = EMS_INFO_BKGD;
		inf_Img[3*x+2] = EMS_INFO_BKGD;
	}
	
	// display image intensity histogram
	if ( project->sysStatus.imgInform == EMS_HSTG_MODE ) {
		
		for ( x=EMS_INFO_EXT; x<(EMS_INFO_BIN-EMS_INFO_EXT); x++ ) {
			if ( maxCount < project->micrograph.hstGram[x] )
				maxCount = project->micrograph.hstGram[x];
		}
		
		for ( x=0; x<EMS_INFO_BIN; x++ ) {
			
			upto = (int)(1.0 * project->micrograph.hstGram[x] * (EMS_INFO_WIN-4) / maxCount);
			if ( upto > (EMS_INFO_WIN-4) ) upto = EMS_INFO_WIN - 4;
			
			for ( y=(EMS_INFO_WIN-upto); y<EMS_INFO_WIN; y++ ) {
				index = y * EMS_INFO_WIN + x + (EMS_INFO_WIN-EMS_INFO_BIN)/2;
				inf_Img[3*index+0] = EMS_HSTG_R;
				inf_Img[3*index+1] = EMS_HSTG_G;
				inf_Img[3*index+2] = EMS_HSTG_B;
			}
		}
	}
	
	// display image power spectrum
	else if ( project->sysStatus.imgInform == EMS_SPEC_MODE ) {

		EMS_DP->imgDisplay->getFocusXY(x, y);

		if ( (x<0) && (y<0) ) {
			if ( inf_rgb != NULL ) delete inf_rgb;
			inf_rgb = new Fl_RGB_Image(inf_Img, EMS_INFO_WIN, EMS_INFO_WIN, 3, 0);
			EMS_DP->imgInform->image(inf_rgb);
			EMS_DP->imgInform->redraw();
			return;
		}

		else if ( flagRerun || (dimen[0]!=project->parameter.specSize) ) {
			
			clearImgInfo();
			EMS_DP->imgDisplay->getFocusXY(x, y);
			x *= project->parameter.downSize;
			y *= project->parameter.downSize;
			
			dimX = project->micrograph.dimX;
			dimY = project->micrograph.dimY;
			dimen[0] = project->parameter.specSize;
			dimen[1] = project->parameter.specSize;
			dsize = dimen[0] * dimen[1];
			
			x0 = x - dimen[0]/2;
			x1 = x0 + dimen[0];
			y0 = y - dimen[1]/2;
			y1 = y0 + dimen[1];

			if ( (x0<0) || (x1>=dimX) || (y0<0) || (y1>=dimY) ) {
				msgReport("The selected region is beyond image domain");

				if ( inf_rgb != NULL ) delete inf_rgb;
				inf_rgb = new Fl_RGB_Image(inf_Img, EMS_INFO_WIN, EMS_INFO_WIN, 3, 0);
				EMS_DP->imgInform->image(inf_rgb);
				EMS_DP->imgInform->redraw();
				return;
			}
			
			systemStatus(EMS_BUSY);

			index = 0;
			rel_t = new float[dsize];
			img_t = new float[dsize];
			
			for ( y=y0; y<y1; y++ ) {
				for ( x=x0; x<x1; x++ ) {
					rel_t[index] = project->micrograph.density[y*dimX+x];
					img_t[index] = 0.0;
					index ++;
				}
			}
			
			// compute, smooth and interpolate the spectrum
			fftwN(2, dimen, rel_t, img_t, FFTW_FORWARD);
			
			for ( index=0; index<dsize; index++ ) {
				valInt = rel_t[index] * rel_t[index] + img_t[index] * img_t[index];
				rel_t[index] = sqrt(valInt);
				// rel_t[index] = log(valInt + 1.0);
			}
			
			origSpec.setup(rel_t, dimen[0], dimen[1], 0, 0);
			averSpec.initF(0.0, EMS_INFO_WIN, EMS_INFO_WIN);
			smoothSpectrum(origSpec, averSpec, EMS_SPEC_SMOT);
			
			delete [] rel_t;
			delete [] img_t;

			systemStatus(EMS_FREE);
		}

		// determine average & variance of the image
		aveInt = varInt = stdInt = delInt = 0.0;
		dsize = EMS_INFO_WIN * EMS_INFO_WIN;

		for ( index=0; index<dsize; index++ ) {
			aveInt += averSpec.dataF[index];
			varInt += averSpec.dataF[index] * averSpec.dataF[index];
		}

		aveInt /= dsize;
		varInt /= dsize;
		stdInt = varInt - aveInt * aveInt;

		if ( stdInt > 0.0 )
			stdInt = sqrt(stdInt);
		else
			stdInt = 0.0;

		delInt = (1.0 - project->parameter.spcCntr) * stdInt * EMS_SPEC_SIGMA + EMS_ZERO;

		// assign RGB for image display
		for ( index=0; index<dsize; index++ ) {
			valInt = 255.0 * ((averSpec.dataF[index] - aveInt) / delInt + project->parameter.spcBrit);
			
			if ( valInt < 0 )
				inf_Img[3*index+0] = (unsigned char)0;
			else if ( valInt > 255 )
				inf_Img[3*index+0] = (unsigned char)255;
			else
				inf_Img[3*index+0] = (unsigned char)(valInt);
			
			inf_Img[3*index+2] = inf_Img[3*index+1] = inf_Img[3*index+0];
		}
	}
	
	// should never happen
	else {
		errReport("Unknown information display request!");
		return;
	}
	
	if ( inf_rgb != NULL ) delete inf_rgb;
	inf_rgb = new Fl_RGB_Image(inf_Img, EMS_INFO_WIN, EMS_INFO_WIN, 3, 0);

	EMS_DP->imgInform->image(inf_rgb);
	EMS_DP->imgInform->redraw();
	Fl::flush();
}


void EMS_Panel::clearImgInfo(void)
{
	long index, imgSize;

	imgSize = EMS_INFO_WIN * EMS_INFO_WIN;

	for ( index=0; index<imgSize; index++ ) {
		inf_Img[3*index+0] = EMS_INFO_BKGD;
		inf_Img[3*index+1] = EMS_INFO_BKGD;
		inf_Img[3*index+2] = EMS_INFO_BKGD;
	}

	if ( inf_rgb != NULL ) delete inf_rgb;
	inf_rgb = new Fl_RGB_Image(inf_Img, EMS_INFO_WIN, EMS_INFO_WIN, 3, 0);

	EMS_DP->imgInform->image(inf_rgb);
	EMS_DP->imgInform->redraw();
	Fl::flush();
}


void EMS_Panel::clearMonitor(void)
{
	long index, imgSize;

	imgSize = EMS_MONTR_SIZE * EMS_MONTR_SIZE;

	for ( index=0; index<imgSize; index++ ) {
		mon1Img[3*index+0] = 0;
		mon1Img[3*index+1] = 0;
		mon1Img[3*index+2] = EMS_MONTR_BLUE;
		
		mon2Img[3*index+0] = 0;
		mon2Img[3*index+1] = 0;
		mon2Img[3*index+2] = EMS_MONTR_BLUE;
	}
	
	if ( mon1rgb != NULL ) delete mon1rgb;
	mon1rgb = new Fl_RGB_Image(mon1Img, EMS_MONTR_SIZE, EMS_MONTR_SIZE, 3, 0);
	EMS_DP->monitor1->image(mon1rgb);
	EMS_DP->monitor1->redraw();

	if ( mon2rgb != NULL ) delete mon2rgb;
	mon2rgb = new Fl_RGB_Image(mon2Img, EMS_MONTR_SIZE, EMS_MONTR_SIZE, 3, 0);
	EMS_DP->monitor2->image(mon2rgb);
	EMS_DP->monitor2->redraw();

	// reset s_factor display range
	EMS_DP->sldScore->minimum(0.0);
	EMS_DP->sldScore->maximum(1.0);

	Fl::flush();
}


void EMS_Panel::displayParam(void)
{
	char fileName[NAMESIZE];

	// system status flags
	project->sysStatus.showMask = EMS_TRUE;
	EMS_DP->btnShowMask->value(1);

	project->sysStatus.showPrtk = EMS_TRUE;
	EMS_DP->btnShowPrtk->value(0);

	project->sysStatus.showZoom = EMS_FALSE;
	EMS_DP->btnShowZoom->value(0);

	project->sysStatus.drawTools = EMS_FALSE;
	EMS_DP->btnLines->value(0);
	EMS_DP->btnCircle->value(0);
	EMS_DP->btnPaint->value(0);
	EMS_DP->btnErase->value(0);

	project->sysStatus.imgInform = EMS_HSTG_MODE;
	EMS_DP->btnSpecMode->value(0);
	EMS_DP->btnHstgMode->value(1);

	project->sysStatus.showRefr = EMS_FALSE;
	EMS_DP->btnTemplate->value(0);

	project->sysStatus.showSpec = EMS_FALSE;
	EMS_DP->btnTemplate->value(0);

	project->sysStatus.newScale = EMS_FALSE;
	project->sysStatus.syncMask = EMS_FALSE;
	project->sysStatus.mapRestore = EMS_TRUE;

	// MICROGRAPH panel
	EMS_DP->dspFilmX->value(project->micrograph.dimX);
	EMS_DP->dspFilmY->value(project->micrograph.dimY);
	EMS_DP->dspPixSize->value(project->parameter.pixelSize);
	EMS_DP->sldBrit->value(project->parameter.flmBrit);
	EMS_DP->sldCntr->value(project->parameter.flmCntr);

	EMS_DP->sldVarH->value(project->parameter.varHi);
	EMS_DP->sldVarL->value(project->parameter.varLo);
	EMS_DP->valEdge->value(project->parameter.edgeExcu);
	EMS_DP->valBrush->value(project->parameter.brushSize);
	EMS_DP->valBrush2->value(project->parameter.brushSize);
//	EMS_DP->sldTrans->value(project->parameter.transp);

	if ( project->micrograph.exist == EMS_TRUE ) {
		strcpy(fileName, project->emfilmFN);
		simpleFilePath(fileName);
		EMS_DP->dspFilmFile->value(fileName);
	}
	else
		EMS_DP->dspFilmFile->value("\0");

	// TEMPLATE panel
	EMS_DP->dspTempN->value(project->templates.dimZ);
	EMS_DP->dspTempX->value(project->templates.dimX);
	EMS_DP->dspTempY->value(project->templates.dimY);

	EMS_DP->sldLCFrad->value(project->parameter.lcfMaskRad);
	EMS_DP->sldSCFrad->value(project->parameter.scfMaskRad);
	EMS_DP->sldSCFradc->value(project->parameter.scfMaskRadc);
	EMS_DP->dspRotaRange->value(project->parameter.rotaRange);
	EMS_DP->dspRotaDelta->value(project->parameter.rotaDelta);

	if ( project->sysStatus.rotaScreen == EMS_TRUE )
		EMS_DP->btnRotate->value(1);
	else
		EMS_DP->btnRotate->value(0);

	if ( project->templates.exist == EMS_TRUE ) {
		strcpy(fileName, project->templtFN);
		simpleFilePath(fileName);
		EMS_DP->dspTempFile->value(fileName);
	}
	else
		EMS_DP->dspTempFile->value("\0");

	// PARTICLE panel
//	EMS_DP->maxLCF->value(1.0);
//	EMS_DP->aveLCF->value(0.0);
//	EMS_DP->cntLCF->value(0);
	EMS_DP->sldLCF->value(project->parameter.lcfThres);

//	EMS_DP->maxSCF->value(1.0);
//	EMS_DP->aveSCF->value(0.0);
//	EMS_DP->cntSCF->value(0);
	EMS_DP->sldSCF->value(project->parameter.scfThres);

	EMS_DP->dspPrtkSize->value(project->parameter.prtkSize);
	EMS_DP->dspDist->value(project->parameter.distance);
	EMS_DP->dspDownSize->value(project->parameter.downSize);

	EMS_DP->sldScore->maximum(project->parameter.dcfMax);
	EMS_DP->sldScore->minimum(project->parameter.dcfAve);
	EMS_DP->sldScore->value(0.0);

	// FUNCTION: DIGITAL GEL
	EMS_DP->sldGelMask->value(EMS_GEL_RADIUS);

	// message board
	EMS_DP->dspCount->value(project->parameter.totCount);
}
