//	James Z. Chen
//	January 1, 2005
//	All rights reserved

#include "images.h"
#include "../signature/include/FL/fl_ask.H"
#include "gui.h"
#include "fftn.h"

// default name of new image stack
#define EMS_NEW_STACK		"new_stack.mrc"


extern DisplayPanel_UI *EMS_DP;


EMS_Images::EMS_Images()
{
	int i;

	param.imgX = 0;
	param.imgY = 0;
	param.imgN = 0;
	attrib = attrb2 = NULL;

	frame0 = 0;
	pageSize = 1;
	selCount = 0;
	flagSaved = EMS_TRUE;
	labelDisp = EMS_TRUE;
	imgBrit = EMS_BRIGHTNESS;
	imgCntr = EMS_CONTRAST;
	textSize = EMS_TEX_NORM;

	// digital gel filtration
	gelMass = NULL;
	gelCount = 0;
	gelSelect = NULL;
	gelBinMax = 0;
	gelLmass = -3.0;
	gelHmass = +3.0;
	gelMassAve = 0.0;
	gelMassStd = 1.0;
	gelMassMin = -5.0;
	gelMassMax = +5.0;
	gelMassRad = EMS_GEL_RADIUS;

	// histogram charts
	dxBinMax = ccBinMax = 0;
	alignCount = 0;

	// parameter reset
	imgScale = 1;
	gelMassRad = EMS_GEL_RADIUS;
	maskRadius = EMS_CLS_RADIUS;
	corrThres = EMS_ALN_CORREL;
	shiftThres = EMS_ALN_SHIFT;
	flagMask = EMS_FALSE;
	flagRota = EMS_FALSE;
	flagSync = EMS_FALSE;
	flagFrealign = EMS_FALSE;

	param.precision = EMS_PRECISION;
	param.endin = EMS_ARCH_ENDIN;

	for ( i=0; i<EMS_VLENGTH; i++ ) param.version[i] = '\0';
	strcpy(param.version, EMS_VERSION);
}


EMS_Images::~EMS_Images()
{
	if ( gelMass != NULL ) {
		delete [] gelMass;
		gelMass = NULL;
	}

	if ( gelSelect != NULL ) {
		delete [] gelSelect;
		gelSelect = NULL;
	}

	if ( attrib != NULL ) {
		delete [] attrib;
		attrib = NULL;
	}

	if ( attrb2 != NULL ) {
		delete [] attrb2;
		attrb2 = NULL;
	}
}


char EMS_Images::loadImageStack(char *fileName)
{
	int index;
	char pName[NAMESIZE];

	if ( imgStack.input(fileName) == EMS_TRUE ) {
		strcpy(stackName, fileName);
		getFilePath(fileName, pName, imgsName);
		removeExt(imgsName, ".mrc");
	}
	else
		return(EMS_FALSE);

	param.imgX = imgStack.dimX;
	param.imgY = imgStack.dimY;
	param.imgN = imgStack.dimZ;
	frameSize = param.imgX * param.imgY;
	imgStack.exist = EMS_TRUE;
	flagSaved = EMS_TRUE;
	selCount = 0;

	for ( index=0; index<(EMS_HEADER-EMS_VLENGTH-20); index++ )
		param.unused[index] = '\0';

	// allocate space for particle attribute array
	if ( attrib != NULL ) delete [] attrib;
	attrib = new struct EMS_PTKPAR[param.imgN];

	for ( index=0; index<param.imgN; index++ ) {
		attrib[index].x0 = 0;
		attrib[index].y0 = 0;
		attrib[index].dx = 0.0;
		attrib[index].dy = 0.0;
		attrib[index].shift = 0.0;
		attrib[index].eulA = 0.0;
		attrib[index].eulB = 0.0;
		attrib[index].eulG = 0.0;
		attrib[index].score = 0.0;
		attrib[index].tmpid = -1;
		attrib[index].select = EMS_FALSE;

		// imaging parameters
		attrib[index].df1 = 0.0;
		attrib[index].df2 = 0.0;
		attrib[index].stigma = 0.0;
	}

	// reset for particle alignment/classification
	flagAlign = EMS_FALSE;
	flagClass = EMS_FALSE;
	flagSync = EMS_FALSE;
	flagFrealign = EMS_FALSE;

	// clear histogram / digital gel display
	ccBinMax = dxBinMax = gelBinMax = 0;
	for ( index=0; index<EMS_DX_BIN; index++ ) dxHist[index] = 0;
	for ( index=0; index<EMS_CC_BIN; index++ ) ccHist[index] = 0;
	for ( index=0; index<EMS_GEL_BIN; index++ ) gelHist[index] = 0;

	if ( emsRunMode == EMS_GUI_MODE ) {
		pageSize = 1;
		
		setDispFrames(1);
		EMS_DP->valFrame0->value(1);
		
		textSize = EMS_TEX_NORM;
		EMS_DP->dspQuality->value(0);
		
		imgScale = (param.imgX + param.imgY) / (2 * textSize);
		if ( imgScale < 1 ) imgScale = 1;
		EMS_DP->valDispScale->value(imgScale);
		
		// reset for digital gel filtration
		gelBinMax = 0;
		gelLmass = -3.0;
		gelHmass = +3.0;
		gelMassAve = 0.0;
		gelMassStd = 1.0;
		gelMassMin = -5.0;
		gelMassMax = +5.0;
		gelMassRad = EMS_GEL_RADIUS;
	}

	return(EMS_TRUE);
}


char EMS_Images::appendImageStack(char *fileName)
{
	int index, offset, y, z, dx, dy;
	MRC_MAP imgTemp;
	float *buffer, *buffer0, *bkgd, *page;

	if ( imgTemp.input(fileName) == EMS_FALSE ) return(EMS_FALSE);

	if ( (imgTemp.dimX>(2*EMS_TEX_HIGH)) || (imgTemp.dimY>(2*EMS_TEX_HIGH)) ) {
		errReport("Image size is too large to be displayed!");
		return(EMS_FALSE);
	}

	// images should share the same pixel size
	if ( fabs(imgStack.pixelX-imgTemp.pixelX)>EMS_ZERO ||
		 fabs(imgStack.pixelY-imgTemp.pixelY)>EMS_ZERO ||
		 fabs(imgStack.pixelZ-imgTemp.pixelZ)>EMS_ZERO ) {

		fl_beep(FL_BEEP_MESSAGE);
		if ( fl_ask("Pixel size differs. Proceed anyway?") == 0 )
			return(EMS_FALSE);
	}

	param.imgX = (imgStack.dimX > imgTemp.dimX)? imgStack.dimX : imgTemp.dimX;
	param.imgY = (imgStack.dimY > imgTemp.dimY)? imgStack.dimY : imgTemp.dimY;
	param.imgN = imgStack.dimZ + imgTemp.dimZ;

	frameSize = param.imgX * param.imgY;
	buffer = new float[frameSize * param.imgN];
	buffer0 = buffer;

	// background template
	bkgd = new float[frameSize];

	// copy the first stack
	dx = (param.imgX - imgStack.dimX) / 2;
	dy = (param.imgY - imgStack.dimY) / 2;

	for ( index=0; index<frameSize; index++ )
		bkgd[index] = imgStack.aveInt;

	for ( z=0; z<imgStack.dimZ; z++ ) {
		memcpy(buffer, bkgd, frameSize*sizeof(float));
		page = imgStack.density + z * imgStack.dimX * imgStack.dimY;

		for ( y=0; y<imgStack.dimY; y++ ) {
			offset = (y+dy) * param.imgX + dx;
			memcpy(buffer+offset, page+y*imgStack.dimX, imgStack.dimX*sizeof(float));
		}

		buffer += frameSize;
	}

	// copy the second stack
	dx = (param.imgX - imgTemp.dimX) / 2;
	dy = (param.imgY - imgTemp.dimY) / 2;

	for ( index=0; index<frameSize; index++ )
		bkgd[index] = imgTemp.aveInt;

	for ( z=0; z<imgTemp.dimZ; z++ ) {
		memcpy(buffer, bkgd, frameSize*sizeof(float));
		page = imgTemp.density + z * imgTemp.dimX * imgTemp.dimY;

		for ( y=0; y<imgTemp.dimY; y++ ) {
			offset = (y+dy) * param.imgX + dx;
			memcpy(buffer+offset, page+y*imgTemp.dimX, imgTemp.dimX*sizeof(float));
		}

		buffer += frameSize;
	}

	// create the merged image stack
	setNewStack(buffer0, EMS_TRUE);

	delete [] buffer0;
	delete [] bkgd;

	return(EMS_TRUE);
}


char EMS_Images::saveImageStack(char *fileName)
{
	if ( imgStack.exist == EMS_TRUE ) {
		if ( imgStack.output(fileName) ) {
			strcpy(stackName, fileName);
			flagSaved = EMS_TRUE;
			return(EMS_TRUE);
		}
	}

	return(EMS_FALSE);
}


char EMS_Images::clearImageStack(void)
{
	int index;

	imgStack.clear();

	// reset panel display
	strcpy(stackName, EMS_NEW_STACK);
	param.imgX = param.imgY = param.imgN = 0;

	if ( attrib != NULL ) {
		delete [] attrib;
		attrib = NULL;
	}

	// reset Digital Gel
	gelBinMax = 0;
	for ( index=0; index<EMS_GEL_BIN; index++ ) gelHist[index] = 0;

	return(EMS_TRUE);
}


void EMS_Images::setDispFrames(int f0)
{
	// input parameter is in base-1
	if ( f0 > 0 )
		frame0 = (f0 < param.imgN)? (f0-1) : (param.imgN-1);
	else
		frame0 = 0;
}


void EMS_Images::browseFrames(char instruct)
{
	switch ( instruct ) {
	case 0:		// to the end
		frame0 = param.imgN - pageSize;
		break;
	case +1:	// forward one page
		frame0 += pageSize;
		break;
	case -1:	// backward one page
		frame0 -= pageSize;
		break;
	default:
		frame0 ++;
	}

	if ( frame0 < 0 ) frame0 = 0;
	if ( frame0 >= param.imgN ) frame0 = param.imgN - 1;
}


void EMS_Images::clearSelect(void)
{
	int index;

	for ( index=0; index<param.imgN; index++ )
		attrib[index].select = EMS_FALSE;

	selCount = 0;
}


char EMS_Images::extractFrames(char &flagSync)
{
	int index, frames;
	float *buffer;

	if ( selCount == 0 ) return(EMS_FALSE);
	if ( selCount == param.imgN ) return(EMS_FALSE);

	buffer = new float[frameSize * selCount];

	// collect image frames
	frames = 0;

	for ( index=0; index<param.imgN; index++ ) {
		if ( attrib[index].select == EMS_TRUE ) {
			memcpy(buffer+frames*frameSize, imgStack.density+index*frameSize, frameSize*sizeof(float));
			frames ++;
		}
	}

	// synchronize PTK and STACK
	if ( strcmp(project->dirProject, EMS_PRJ_DR) )
		flagSync = syncPrtkSelect(EMS_TRUE);
	else
		flagSync = EMS_FALSE;

	// image parameter backup
	frame0 = 0;
	param.imgN = frames;
	setNewStack(buffer, EMS_FALSE);

	// reload attrib[] upon synchronization
	if ( flagSync == EMS_TRUE )
		memcpy(attrib, attrb2, sizeof(struct EMS_PTKPAR)*param.imgN);

	delete [] buffer;
	return(EMS_TRUE);
}


char EMS_Images::deleteFrames(char &flagSync)
{
	int index, frames;
	float *buffer;

	if ( selCount == 0 ) return(EMS_FALSE);
	if ( selCount == param.imgN ) return(EMS_FALSE);

	selCount = param.imgN - selCount;
	buffer = new float[frameSize * selCount];

	// collect image frames
	frames = 0;

	for ( index=0; index<param.imgN; index++ ) {
		if ( attrib[index].select == EMS_FALSE ) {
			memcpy(buffer+frames*frameSize, imgStack.density+index*frameSize, frameSize*sizeof(float));
			frames ++;
		}
	}

	// synchronize PTK and STACK
	if ( strcmp(project->dirProject, EMS_PRJ_DR) )
		flagSync = syncPrtkSelect(EMS_FALSE);
	else
		flagSync = EMS_FALSE;

	// image parameter backup
	frame0 = 0;
	param.imgN = frames;
	setNewStack(buffer, EMS_FALSE);

	// reload attrib[] upon synchronization
	if ( flagSync == EMS_TRUE )
		memcpy(attrib, attrb2, sizeof(struct EMS_PTKPAR)*param.imgN);

	delete [] buffer;
	return(EMS_TRUE);
}


char EMS_Images::syncPrtkSelect(char status)
{
	int index, imgN_t, count;
	char pName[NAMESIZE], fName[NAMESIZE], buffer[LINE_MAX];
	char file_src[NAMESIZE], file_dst[NAMESIZE], file_aln[NAMESIZE];
	FILE *fp_src, *fp_dst, *fp_aln;

	if ( !strcmp(project->dirAlignmnt, EMS_PRJ_DR) ) {
		errReport("Please set up a project space first!");
		return(EMS_FALSE);
	}

	// verify file consistency and assign file names
	getFilePath(stackName, pName, fName);

	if ( strcmp(project->dirParticle, pName) || strcmp(project->filmName, imgsName) ) {
		errReport("EM micrograph and particle stack mismatch!");
		return(EMS_FALSE);
	}

	sprintf(file_src, "%s/%s_stack.ptk", pName, imgsName);
	sprintf(file_dst, "%s/%s_stack.bak", pName, imgsName);

	// attempt to open PTK record
	fp_src = fopen(file_src, "r");
	if ( fp_src == NULL ) {
		errReport("Cannot identify particle coordinates!");
		return(EMS_FALSE);
	}

	// compare number of particles (PTK v/s MRC)
	fgets(buffer, LINE_MAX, fp_src);
	sscanf(buffer+strlen(EMS_PARTIC_H)+3, "%d", &index);

	if ( index != param.imgN ) {
		errReport("Dataset is already out of synchronization!");
		fclose(fp_src);
		return(EMS_FALSE);
	}

	// copy "*_stack.ptk" to "*_stack.bak"
	fp_dst = fopen(file_dst, "w");
	while ( fgets(buffer, LINE_MAX, fp_src) != NULL ) {
		fprintf(fp_dst, "%s", buffer);
	}

	fclose(fp_src);
	fclose(fp_dst);

	// synchronize PTK and MRC records (overwrite PTK file)
	sprintf(file_src, "%s/%s_stack.bak", pName, imgsName);
	sprintf(file_dst, "%s/%s_stack.ptk", pName, imgsName);

	fp_src = fopen(file_src, "r");
	fp_dst = fopen(file_dst, "w");
	fprintf(fp_dst, "%s : %d\n", EMS_PARTIC_H, selCount);

	// save alignment parameters
	imgN_t = param.imgN;
	sprintf(file_aln, "%s/%s.aln", project->dirAlignmnt, imgsName);
	fp_aln = fopen(file_aln, "wb");
	
	param.imgN = selCount;
	param.endin = EMS_ARCH_ENDIN;
	fwrite(&param, 1, sizeof(struct EMS_ALNPAR), fp_aln);
	param.imgN = imgN_t;
	
	// synchronize attrib[]
	if ( attrb2 != NULL ) delete [] attrb2;
	attrb2 = new struct EMS_PTKPAR[selCount];

	count = 0;
	for ( index=0; index<param.imgN; index++ ) {
		fgets(buffer, LINE_MAX, fp_src);

		if ( attrib[index].select == status ) {
			// write out PTK record
			fprintf(fp_dst, "%s", buffer);

			// write out attrib[]
			fwrite(attrib+index, 1, sizeof(struct EMS_PTKPAR), fp_aln);
			memcpy(attrb2+count, attrib+index, sizeof(struct EMS_PTKPAR));
			attrb2[count].select = EMS_FALSE;
			count ++;
		}
	}

	if ( fp_src != NULL ) fclose(fp_src);
	if ( fp_dst != NULL ) fclose(fp_dst);
	if ( fp_aln != NULL ) fclose(fp_aln);

	msgReport("Particle coordinates and images have been synchronized");
	return(EMS_TRUE);
}


void EMS_Images::paramDisplay(void)
{
	char fileName[NAMESIZE];

	// in IMAGE STACK PANEL
	if ( strcmp(stackName, EMS_NEW_STACK) ) {
		strcpy(fileName, stackName);
		simpleFilePath(fileName);
		EMS_DP->dspImageFile->value(fileName);
	}
	else
		EMS_DP->dspImageFile->value("New image stack");

	EMS_DP->dspImgX->value(param.imgX);
	EMS_DP->dspImgY->value(param.imgY);
	EMS_DP->dspImgN->value(param.imgN);
	EMS_DP->dspImgMin->value(imgStack.minInt);
	EMS_DP->dspImgMax->value(imgStack.maxInt);
	EMS_DP->dspImgAve->value(imgStack.aveInt);
	EMS_DP->dspImgVar->value(imgStack.stdInt * imgStack.stdInt);
}


char EMS_Images::filtImageStack(void)
{
	int n, x, y, xc, yc, dimen[2];
	long index, page;
	float *buffer, *imgRel, *imgImg, ampt;
	double xf, yf, L_pass, H_pass, freq, expn;

	if ( imgStack.mapSize <= 0 ) return(EMS_FALSE);

	imgRel = new float[frameSize];
	imgImg = new float[frameSize];
	buffer = new float[imgStack.mapSize];
	memcpy(buffer, imgStack.density, imgStack.mapSize*sizeof(float));

	xc = param.imgX / 2; dimen[0] = param.imgX;
	yc = param.imgY / 2; dimen[1] = param.imgY;
	L_pass = 2.0 * imgStack.pixelX / lowPass;
	H_pass = 2.0 * imgStack.pixelY / highPass;

	if ( (L_pass>1.415) || (H_pass<0.0) || (H_pass>L_pass) ) {
		errReport("Incorrect band-pass filtering parameters!");
		return(EMS_FALSE);
	}

	// process frame by frame
	for ( n=0; n<param.imgN; n++ ) {
		page = n * frameSize;
		memcpy(imgRel, buffer+page, frameSize*sizeof(float));
		
		for ( index=0; index<frameSize; index++ )
			imgImg[index] = 0.0;
		
		fftwN(2, dimen, imgRel, imgImg, FFTW_FORWARD);

		for ( y=0; y<param.imgY; y++ ) {
			yf = 1.0 * ((y + yc) % param.imgY - yc) / yc;
			
			for ( x=0; x<param.imgX; x++ ) {
				xf = 1.0 * ((x + xc) % param.imgX - xc) / xc;
				
				index = y * param.imgX + x;
				freq = sqrt(xf*xf + yf*yf);
				
				if ( freq < H_pass ) {
					expn = (freq - H_pass) / EMS_BAND_HSFT;
					ampt = (float)exp(-1.0 * expn * expn);
					imgRel[index] *= ampt;
					imgImg[index] *= ampt;
				}
				else if ( freq > L_pass ) {
					expn = (freq - L_pass) / EMS_BAND_LSFT;
					ampt = (float)exp(-1.0 * expn * expn);
					imgRel[index] *= ampt;
					imgImg[index] *= ampt;
				}
			}
		}
		
		// redefine micrograph.density[]
		fftwN(2, dimen, imgRel, imgImg, FFTW_BACKWARD);
		memcpy(buffer+page, imgRel, sizeof(float)*frameSize);
	}

	// reload image stack
	setNewStack(buffer, EMS_FALSE);

	delete [] buffer;
	delete [] imgRel;
	delete [] imgImg;

	return(EMS_TRUE);
}


char EMS_Images::normImageStack(void)
{
	int n;
	long index, page;
	float imgAve, imgStd, *buffer;

	if ( imgStack.mapSize <= 0 ) return(EMS_FALSE);

	buffer = new float[imgStack.mapSize];
	memcpy(buffer, imgStack.density, imgStack.mapSize*sizeof(float));

	// process frame by frame
	for ( n=0; n<param.imgN; n++ ) {
		imgAve = imgStd = 0.0;
		page = n * frameSize;

		for ( index=0; index<frameSize; index++ ) {
			imgAve += imgStack.density[page+index];
			imgStd += imgStack.density[page+index] * imgStack.density[page+index];
		}

		imgAve /= frameSize;
		imgStd = imgStd / frameSize - imgAve * imgAve;

		if ( imgStd > 0.0 )
			imgStd = (float)(sqrt(imgStd));
		else
			imgStd = EMS_ZERO;

		for ( index=0; index<frameSize; index++ )
			buffer[page+index] = (imgStack.density[page+index]-imgAve) / imgStd;
	}

	// reload image stack
	setNewStack(buffer, EMS_FALSE);

	delete [] buffer;
	return(EMS_TRUE);
}


char EMS_Images::scalImageStack(void)
{
	int n;
	long index, page;
	float *buffer;

	if ( imgStack.mapSize <= 0 ) return(EMS_FALSE);

	buffer = new float[imgStack.mapSize];
	memcpy(buffer, imgStack.density, imgStack.mapSize*sizeof(float));

	// process frame by frame
	for ( n=0; n<param.imgN; n++ ) {
		page = n * frameSize;

		for ( index=0; index<frameSize; index++ ) {
			buffer[page+index] = (float)(frameMult * imgStack.density[page+index] + frameAddt);
		}
	}

	// reload image stack
	setNewStack(buffer, EMS_FALSE);

	delete [] buffer;
	return(EMS_TRUE);
}


char EMS_Images::clipImageStack(void)
{
	if ( imgStack.exist == EMS_FALSE ) return(EMS_FALSE);

	imgStack.clipping(clipMin, clipMax);
	flagSaved = EMS_FALSE;

	return(EMS_TRUE);
}


char EMS_Images::cropImageStack(void)
{
	int n, dx, dy, y;
	long index, offset_src, offset_dst, newFrame, page;
	float *buffer, *bkgd;

	if ( imgStack.mapSize <= 0 ) return(EMS_FALSE);
	if ( (cropX<=0) || (cropY<=0) ) return(EMS_FALSE);
	if ( (cropX==param.imgX) && (cropY==param.imgY) ) return(EMS_FALSE);

	newFrame = cropX * cropY;
	bkgd = new float[newFrame];
	buffer = new float[newFrame * param.imgN];

	for ( index=0; index<newFrame; index++ )
		bkgd[index] = imgStack.aveInt;

	dx = (cropX - param.imgX) / 2;
	dy = (cropY - param.imgY) / 2;

	// process frame by frame
	for ( n=0; n<param.imgN; n++ ) {
		page = n * newFrame;

		if ( (dx>=0) && (dy>=0) ) {
			memcpy(buffer+page, bkgd, newFrame*sizeof(float));
			offset_src = n * frameSize;
			offset_dst = page + dy * cropX + dx;

			for ( y=0; y<param.imgY; y++ ) {
				memcpy(buffer+offset_dst, imgStack.density+offset_src, param.imgX*sizeof(float));
				offset_src += param.imgX;
				offset_dst += cropX;
			}
		}
		else if ( (dx>=0) && (dy<0) ) {
			memcpy(buffer+page, bkgd, newFrame*sizeof(float));
			offset_src = n * frameSize - dy * param.imgX;
			offset_dst = page + dx;

			for ( y=0; y<cropY; y++ ) {
				memcpy(buffer+offset_dst, imgStack.density+offset_src, param.imgX*sizeof(float));
				offset_src += param.imgX;
				offset_dst += cropX;
			}
		}
		else if ( (dx<0) && (dy>=0) ) {
			memcpy(buffer+page, bkgd, newFrame*sizeof(float));
			offset_src = n * frameSize - dx;
			offset_dst = page + dy * cropX;

			for ( y=0; y<param.imgY; y++ ) {
				memcpy(buffer+offset_dst, imgStack.density+offset_src, cropX*sizeof(float));
				offset_src += param.imgX;
				offset_dst += cropX;
			}
		}
		else {
			offset_src = n * frameSize - dy * param.imgX - dx;
			offset_dst = page;

			for ( y=0; y<cropY; y++ ) {
				memcpy(buffer+offset_dst, imgStack.density+offset_src, cropX*sizeof(float));
				offset_src += param.imgX;
				offset_dst += cropX;
			}
		}
	}

	// reload image stack
	param.imgX = cropX;
	param.imgY = cropY;
	frameSize = param.imgX * param.imgY;
	setNewStack(buffer, EMS_FALSE);

	delete [] bkgd;
	delete [] buffer;
	return(EMS_TRUE);
}


char EMS_Images::imgPixBin(void)
{
	if ( imgStack.binBox >= 2 ) {
		imgStack.bin2D();
		param.imgX = imgStack.dimX;
		param.imgY = imgStack.dimY;
		frameSize = param.imgX * param.imgY;
		flagSaved = EMS_FALSE;
		return(EMS_TRUE);
	}
	else {
		errReport("Incorrect binning size!");
		return(EMS_FALSE);
	}
}


char EMS_Images::imgResample(void)
{
	if ( imgStack.resample2D() ) {
		param.imgX = imgStack.dimX;
		param.imgY = imgStack.dimY;
		frameSize = param.imgX * param.imgY;
		flagSaved = EMS_FALSE;
		return(EMS_TRUE);
	}
	else
		return(EMS_FALSE);
}


char EMS_Images::imgRotaSum(void)
{
	int x, y, x0, x1, y0, y1, n;
	long page, index;
	double xc, yc, xf, yf, dA, dB, dC, dD, dE, dF;
	double rotate, rotaStep, cosAng, sinAng;
	float *buffer1, *buffer2;

	if ( imgStack.exist == EMS_FALSE ) return(EMS_FALSE);

	xc = 0.5 * param.imgX;
	yc = 0.5 * param.imgY;
	rotaStep = 4.0 * RAD2DEG / (param.imgX + param.imgY);

	// set up temporary storage
	buffer1 = new float[frameSize];
	buffer2 = new float[frameSize];

	for ( index=0; index<frameSize; index++ ) {
		buffer1[index] = 0.0;
		buffer2[index] = 0.0;
	}

	// pixel-wise sum first
	for ( n=0; n<param.imgN; n++ ) {
		page = n * frameSize;

		for ( index=0; index<frameSize; index++ )
			buffer1[index] += imgStack.density[page+index];
	}

	// rotational average afterwards
	for ( rotate=0.0; rotate<(360.0-EMS_ZERO); rotate+=rotaStep ) {
		cosAng = cos(rotate);
		sinAng = sin(rotate);
		
		for ( y=0; y<param.imgY; y++ ) {
			for ( x=0; x<param.imgX; x++ ) {
				index = y * param.imgX + x;
				
				xf = (x - xc) * cosAng + (y - yc) * sinAng + xc;
				yf = (xc - x) * sinAng + (y - yc) * cosAng + yc;
				x0 = (int)floor(xf); x1 = x0 + 1;
				y0 = (int)floor(yf); y1 = y0 + 1;
				
				if ( (x0<0) || (x1>=param.imgX) || (y0<0) || (y1>=param.imgY) ) {
					buffer2[index] += imgStack.aveInt;
					continue;
				}
				
				dA = buffer1[y0 * param.imgX + x0];
				dB = buffer1[y0 * param.imgX + x1];
				dC = buffer1[y1 * param.imgX + x0];
				dD = buffer1[y1 * param.imgX + x1];
				dE = (x1-xf) * dA + (xf-x0) * dB;
				dF = (x1-xf) * dC + (xf-x0) * dD;
				buffer2[index] += (float)((y1-yf) * dE + (yf-y0) * dF);
			}
		}
	}
	
	// rescaling accumulated intensity
	for ( index=0; index<frameSize; index++ )
		buffer2[index] *= (float)(rotaStep / (param.imgN * 360.0));

	// image parameter backup
	param.imgN = 1;
	frame0 = 0;
	setNewStack(buffer2, EMS_TRUE);

	delete [] buffer1;
	delete [] buffer2;

	return(EMS_TRUE);
}


char EMS_Images::imgRotaAve(void)
{
	int x, y, x0, x1, y0, y1, n;
	long page, index;
	double xc, yc, xf, yf, dA, dB, dC, dD, dE, dF;
	double rotate, rotaStep, cosAng, sinAng;
	float *buffer;

	if ( imgStack.exist == EMS_FALSE ) return(EMS_FALSE);

	xc = 0.5 * param.imgX;
	yc = 0.5 * param.imgY;
	rotaStep = 4.0 * RAD2DEG / (param.imgX + param.imgY);

	// set up temporary storage
	buffer = new float[imgStack.mapSize];

	for ( index=0; index<imgStack.mapSize; index++ )
		buffer[index] = 0.0;

	// calculate rotational average frame by frame
	for ( n=0; n<param.imgN; n++ ) {
		page = n * frameSize;

		for ( rotate=0.0; rotate<(360.0-EMS_ZERO); rotate+=rotaStep ) {
			cosAng = cos(rotate);
			sinAng = sin(rotate);
			
			for ( y=0; y<param.imgY; y++ ) {
				for ( x=0; x<param.imgX; x++ ) {
					index = y * param.imgX + x;
					
					xf = (x - xc) * cosAng + (y - yc) * sinAng + xc;
					yf = (xc - x) * sinAng + (y - yc) * cosAng + yc;
					x0 = (int)floor(xf); x1 = x0 + 1;
					y0 = (int)floor(yf); y1 = y0 + 1;
					
					if ( (x0<0) || (x1>=param.imgX) || (y0<0) || (y1>=param.imgY) ) {
						buffer[page+index] += imgStack.aveInt;
						continue;
					}
					
					dA = imgStack.density[page + y0 * param.imgX + x0];
					dB = imgStack.density[page + y0 * param.imgX + x1];
					dC = imgStack.density[page + y1 * param.imgX + x0];
					dD = imgStack.density[page + y1 * param.imgX + x1];
					dE = (x1-xf) * dA + (xf-x0) * dB;
					dF = (x1-xf) * dC + (xf-x0) * dD;
					buffer[page+index] += (float)((y1-yf) * dE + (yf-y0) * dF);
				}
			}
		}
	}

	// rescaling accumulated pixel intensity
	for ( index=0; index<imgStack.mapSize; index++ )
		buffer[index] *= (float)(rotaStep / 360.0);

	setNewStack(buffer, EMS_TRUE);

	delete [] buffer;
	return(EMS_TRUE);
}


char EMS_Images::imgSumAll(void)
{
	long n, index, page;
	float *buffer;

	if ( param.imgN <= 1 ) return(EMS_FALSE);

	// clear accumulator
	buffer = new float[frameSize];

	for ( index=0; index<frameSize; index++ )
		buffer[index] = 0.0;

	for ( n=0; n<param.imgN; n++ ) {
		page = n * frameSize;

		for ( index=0; index<frameSize; index++ )
			buffer[index] += imgStack.density[page+index];
	}

	// calculate pixel average
	for ( index=0; index<frameSize; index++ )
		buffer[index] /= param.imgN;

	// image parameter backup
	param.imgN = 1;
	frame0 = 0;
	setNewStack(buffer, EMS_TRUE);

	delete [] buffer;
	return(EMS_TRUE);
}


void EMS_Images::setNewStack(float *buffer, char flagNewStack)
{
	float px, py, pz;
	long index;

	px = imgStack.pixelX;
	py = imgStack.pixelY;
	pz = imgStack.pixelZ;

	imgStack.clear();
	imgStack.setDimensn(param.imgX, param.imgY, param.imgN, px, py, pz);
	imgStack.setDensity(buffer);

	if ( flagNewStack == EMS_TRUE )
		strcpy(stackName, EMS_NEW_STACK);

	// clear frame selection list
	if ( attrib != NULL ) delete [] attrib;
	attrib = new struct EMS_PTKPAR[param.imgN];

	for ( index=0; index<param.imgN; index++ ) {
		attrib[index].x0 = 0;
		attrib[index].y0 = 0;
		attrib[index].dx = 0.0;
		attrib[index].dy = 0.0;
		attrib[index].shift = 0.0;
		attrib[index].eulA = 0.0;
		attrib[index].eulB = 0.0;
		attrib[index].eulG = 0.0;
		attrib[index].score = 0.0;
		attrib[index].tmpid = -1;
		attrib[index].select = EMS_FALSE;

		attrib[index].df1 = 0.0;
		attrib[index].df2 = 0.0;
		attrib[index].stigma = 0.0;
	}

	// clear histogram / digital gel display
	ccBinMax = dxBinMax = gelBinMax = 0;
	for ( index=0; index<EMS_DX_BIN; index++ ) dxHist[index] = 0;
	for ( index=0; index<EMS_CC_BIN; index++ ) ccHist[index] = 0;
	for ( index=0; index<EMS_GEL_BIN; index++ ) gelHist[index] = 0;

	// auto-determine frames in page
	pageSize = 1;
	selCount = 0;

	// need to be saved
	flagAlign = EMS_FALSE;
	flagSaved = EMS_FALSE;
}


inline void EMS_Images::resetGelThres(void)
{
	gelLmass = (float)(gelMassAve - 3.0 * gelMassStd);
	gelHmass = (float)(gelMassAve + 3.0 * gelMassStd);
}


char EMS_Images::calcGelMass(void)
{
	int x, y, z, pixCount;
	long index, page;
	double sgn_ave, sgn_std;
	double xc, yc, r2, R2;
	char *gelMask;

	if ( imgStack.mapSize <= 0 ) return(EMS_FALSE);

	// setup mass masking
	gelMask = new char[frameSize];

	xc = gelMassRad * param.imgX / 2;
	yc = gelMassRad * param.imgY / 2;
	r2 = ( xc <= yc )? xc : yc;
	R2 = r2 * r2;
	pixCount = 0;

	for ( y=0; y<param.imgY; y++ ) {
		for ( x=0; x<param.imgX; x++ ) {

			index = y * param.imgX + x;
			r2 = (x-xc) * (x-xc) + (y-yc) * (y-yc);

			if ( r2 <= R2 ) {
				gelMask[index] = EMS_TRUE;
				pixCount ++;
			}
			else
				gelMask[index] = EMS_FALSE;
		}
	}

	// safe-guard pixel count
	if ( pixCount == 0 ) {
		errReport("Masking radius is too small!");
		delete [] gelMask;
		return(EMS_FALSE);
	}

	// calculate density mass frame by frame
	gelMassAve = gelMassStd = 0.0;
	gelMassMin = +1.0E9; gelMassMax = -1.0E9;

	if ( gelMass != NULL ) delete [] gelMass;
	gelMass = new float[param.imgN];

	for ( z=0; z<param.imgN; z++ ) {
		page = z * frameSize;
		gelMass[z] = 0.0;

		// SIGMA-normalization
		sgn_ave = sgn_std = 0.0;

		for ( index=0; index<frameSize; index++ ) {
			if ( gelMask[index] == EMS_FALSE ) {
				sgn_ave += imgStack.density[page+index];
				sgn_std += imgStack.density[page+index] * imgStack.density[page+index];
			}
		}

		sgn_ave /= frameSize - pixCount;
		sgn_std = sgn_std / (frameSize-pixCount) - sgn_ave * sgn_ave;

		if ( sgn_std > 0.0 )
			sgn_std = sqrt(sgn_std);
		else
			sgn_std = EMS_ZERO;

		// accumulate mass of SIGMA-normalizaed particle image
		for ( index=0; index<frameSize; index++ ) {
			if ( gelMask[index] == EMS_TRUE )
				gelMass[z] += (float)((imgStack.density[page+index] - sgn_ave) / sgn_std);
		}

		gelMassAve += gelMass[z];
		gelMassStd += gelMass[z] * gelMass[z];

		if ( gelMass[z] > gelMassMax ) gelMassMax = gelMass[z];
		if ( gelMass[z] < gelMassMin ) gelMassMin = gelMass[z];
	}

	gelMassAve /= param.imgN;
	gelMassStd = gelMassStd / param.imgN - gelMassAve * gelMassAve;

	if ( gelMassStd > 0.0 )
		gelMassStd = (float)(sqrt(gelMassStd));
	else
		gelMassStd = EMS_ZERO;

	delete [] gelMask;

	// setup gel histogram
	gelBinMax = 0;

	for ( index=0; index<EMS_GEL_BIN; index++ )
		gelHist[index] = 0;

	for ( z=0; z<param.imgN; z++ ) {

		index = (int)(2*(gelMass[z] - gelMassAve) / gelMassStd) + 2*EMS_GEL_RANGE;

		if ( index < 0 )
			index = 0;
		else if ( index >= EMS_GEL_BIN )
			index = EMS_GEL_BIN - 1;

		gelHist[index] ++;

		if ( gelHist[index] > gelBinMax )
			gelBinMax ++;
	}

	gelLmass = (float)(gelMassAve - 3.0 * gelMassStd);
	gelHmass = (float)(gelMassAve + 3.0 * gelMassStd);
	
	return(EMS_TRUE);
}


char EMS_Images::gelFiltration(void)
{
	int index;

	if ( imgStack.mapSize <= 0 ) return(EMS_FALSE);
	if ( gelMass == NULL ) return(EMS_FALSE);

	if ( gelSelect!= NULL ) delete [] gelSelect;
	gelSelect = new char[param.imgN];
	gelCount = param.imgN;

	for ( index=0; index<param.imgN; index++ ) {
		if ( (gelMass[index]<gelLmass) || (gelMass[index]>gelHmass) ) {
			gelSelect[index] = EMS_FALSE;
			gelCount --;
		}
		else
			gelSelect[index] = EMS_TRUE;
	}

	selCount = gelCount;
	return(EMS_TRUE);
}


void EMS_Images::setGelLmass(double mass)
{
	float massL, massH;

	if ( imgStack.mapSize <= 0 ) return;
	if ( gelMass == NULL ) return;

	massL = gelMassAve - EMS_GEL_RANGE * gelMassStd;
	massH = gelMassAve + EMS_GEL_RANGE * gelMassStd;

	if ( mass < massL )
		gelLmass = massL;
	else if ( mass > massH )
		gelLmass = massH;
	else
		gelLmass = (float)mass;

	if ( gelLmass >= gelHmass )
		gelHmass = gelLmass + EMS_ZERO;
}


void EMS_Images::setGelHmass(double mass)
{
	float massL, massH;

	if ( imgStack.mapSize <= 0 ) return;
	if ( gelMass == NULL ) return;

	massL = gelMassAve - EMS_GEL_RANGE * gelMassStd;
	massH = gelMassAve + EMS_GEL_RANGE * gelMassStd;

	if ( mass < massL )
		gelHmass = massL;
	else if ( mass > massH )
		gelHmass = massH;
	else
		gelHmass = (float)mass;

	if ( gelHmass <= gelLmass )
		gelLmass = gelHmass - EMS_ZERO;
}


char EMS_Images::gelCommit(void)
{
	int index;

	if ( imgStack.exist == EMS_FALSE ) return(EMS_FALSE);
	if ( (attrib==NULL) || (gelSelect==NULL) ) return(EMS_FALSE);

	for ( index=0; index<param.imgN; index++ )
		attrib[index].select = gelSelect[index];

	return(EMS_TRUE);
}


void EMS_Images::resetParam(void)
{
	imgScale = 1;
	EMS_DP->valDispScale->value(1);

	gelMassRad = EMS_GEL_RADIUS;
	EMS_DP->sldGelMask->value(EMS_GEL_RADIUS);

	maskRadius = EMS_CLS_RADIUS;
	param.precision = EMS_PRECISION;
	corrThres = EMS_ALN_CORREL;
	shiftThres = EMS_ALN_SHIFT;
	phaseThres = EMS_ALN_PHASE;
	falgnThres = EMS_ALN_SHIFT;
	parInName[0] = '\0';
}
