#ifndef __PARSER_H
#define __PARSER_H

#include <string>
#include <map>
#include <iostream>

/* ----------------------------------------------------------------------
    options parser is an object to take care of parsing commandline
        options for us. options are added with AddOption method, parsed
        with Parse and individual options can be queried with GetOption.
    Internally, the object emulates a simple dictionary, storing 
        a key (the option name) in one array, a value (the setting of the
        option) in a second; a value is retrieved by cross-indexing both
        arrays.
    Note that the object will not do any type checking (yet!) and will
        always return strings.
    ----------------------------------------------------------------------*/



class OptionsParser {

public:
  // constructor. 
  OptionsParser();
  // destructor: use default destructor
  // ~OptionsParser();

  // add an option
  bool AddOption(const std::string option);
  bool AddFlag(const std::string option);
  // parse the commandline
  void Parse(int argc, char* argv[]);
  // retrieve the value referenced by a specific key
  std::string GetOption(const std::string option);
  bool GetFlag(const std::string option);
  // check the error state of parsing
  bool Error(void);
  // return the error message
  std::string GetError(void);

private:
  // dictionary is implemented via an STL::map
  std::map<std::string,std::string> contents;

  // error state signals when something goes wrong
  bool error;
  // and this contains the actual error message
  std::string error_msg;

};

// constructor. init all internal values.

OptionsParser::OptionsParser() {
  
  contents.clear();
  error=false;
  error_msg="";
  
}

// add an options to be parsed to the internal dictionary

bool OptionsParser::AddOption(const std::string option) {
  
  // first check whether this key exists already!
  std::map<std::string,std::string>::const_iterator key_exists;
  key_exists = contents.find(option);
  if (key_exists == contents.end()) {
    // insert an empty new option, containing only the key...
    contents.insert(make_pair(option,""));
    return true;
  } else {
    // key found, so we already set it!
    error=true;
    error_msg="key "+option+" exists already!";
    std::cout << contents[option] << std::endl;
    return false;
  }
  
}

// retrieve an option, indexed by option name

std::string OptionsParser::GetOption(const std::string option) {
  
  std::map<std::string,std::string>::const_iterator value_location;
  
  value_location=contents.find(option);
  if (value_location == contents.end())
    return "";
  else
    return value_location->second;
  
}

// add an flag that does not store a value, but that is checked for
//    existence only
bool OptionsParser::AddFlag(const std::string option) {

  //check whether this option exists already... create iterator
  std::map<std::string,std::string>::const_iterator key_exists;
  // and attempt to retrieve the option by key name
  key_exists = contents.find(option);
  // check whether we found the option
  if (key_exists == contents.end()) {
    // at end of map, so the key does not exist yet....
    //     insert new option, containing the "flag" key...
    contents.insert(make_pair(option,"flag"));
    return true;
  } else {
    // key exists already, so return an error....
    error=true;
    error_msg="key "+option+" exists already!";
    std::cout << contents[option] << std::endl;
    return false;
  }
}

// retrieve existence of a flag, indexed by option name

bool OptionsParser::GetFlag(const std::string option) {

  std::map<std::string,std::string>::const_iterator value_location;

  value_location=contents.find(option);
  if (value_location == contents.end())
    return "";
  else {
    if (value_location->second == "flag")
      return false;
    else
      return true;
  }
}


// parse the commandline passed in and set the internal values

void OptionsParser::Parse(int argc, char* argv[]) {

  // check for all arguments starting with a "-"
  while ((argc > 1) && (argv[1][0]=='-')) {

    // now get the option name argv, stripped of the -
    std::string key=argv[1];
    key.erase(0,1);

    // first check that the key is one already set up with AddOption.
    // create an iterator
    std::map<std::string,std::string>::const_iterator key_exists;
    // then find the key
    key_exists=contents.find(key);

    // end() means key does not exist...
    if (key_exists == contents.end()) {

      // set error state
      error=true;
      // and message
      error_msg="unknown options key -"+key;
      // and return
      return;

    } else {
    
      // first, check whether this key represents a flag
      if ((key_exists->second) == "flag") {
	// yes, this is a flag. set map value to something other than
	//    flag, to show it was set
	contents[key]="true";
	// and continue to the next option
	argc-=1;
	argv+=1;
      } else {
	// check whether this key is already set to something. if so,
	//    it was specified twice!
	if ((key_exists->second) == "") {
	  // empty string, which is the value we initted to. set the
	  //    new value, then.
	  
	  // get the value that comes right after the key.
	  std::string value=argv[2];
	  
	  // now check that value is not another key by mistake
	  if (value[0] == '-') {
	    // set error state and message
	    error=true;
	    error_msg="no value specified for option "+key;
	    return;
	    
	  } else {
	    // value seems okay, store it.
	    contents[key]=value;
	    // and continue to the next one.
	    argc-=2;
	    argv+=2;
	    
	  }
	}
      }
    }
  }
}

// return errorstate

bool OptionsParser::Error(void) {
  return error;
}

// return the errormessage 

std::string OptionsParser::GetError(void) {
  return error_msg;
}

#endif
