//	James Z. Chen
//	January 1, 2005
//	All rights reserved

#include "project.h"
#include "panel.h"
#include "misc.h"
#include "fftn.h"
#include "gui.h"
#include "algebra.h"
#include "../signature/include/FL/fl_ask.H"


extern DisplayPanel_UI *EMS_DP;


EMS_Project::EMS_Project()
{
	int i;

	record = NULL;
	inspectID = inspectUD = -1;
	projSaveStatus = EMS_TRUE;

	for ( i=0; i<EMS_VLENGTH; i++ ) parameter.version[i] = '\0';
	strcpy(parameter.version, EMS_VERSION);

	// template image
	refer.ptkStack = NULL;
	refer.pwrStack = NULL;
	refer.L1Mask = NULL;
	refer.L2Mask = NULL;
	refer.select = NULL;
	refer.xc = refer.yc = NULL;
	refer.editFrame = -1;
	refer.scale = 1.0;
	refer.maskID = EMS_MASK_L1;

	// system status
	sysStatus.emsdpMode = DP_MCRGRAPH_MODE;
	sysStatus.emsProject = EMS_FALSE;
	sysStatus.imgInvert = EMS_FALSE;
	sysStatus.imgInform = EMS_HSTG_MODE;
	sysStatus.fullScreen = EMS_TRUE;
	sysStatus.rotaScreen = EMS_TRUE;
	sysStatus.testScreen = EMS_FALSE;
	sysStatus.mapRestore = EMS_FALSE;
	sysStatus.drawTools = EMS_FALSE;
	sysStatus.showMask = EMS_TRUE;
	sysStatus.showPrtk = EMS_TRUE;
	sysStatus.showZoom = EMS_FALSE;
	sysStatus.showSpec = EMS_FALSE;
	sysStatus.showRefr = EMS_FALSE;
	sysStatus.newScale = EMS_TRUE;
	sysStatus.syncMask = EMS_FALSE;
	sysStatus.mon1X = EMS_TRUE;
	sysStatus.mon2X = EMS_TRUE;
	sysStatus.mon3X = EMS_TRUE;
	sysStatus.filaAve = EMS_FALSE;
	sysStatus.editTemp = EMS_FALSE;
	sysStatus.activeL2 = EMS_FALSE;

	// MICROGRAPH panel
	parameter.imgX = 0;
	parameter.imgY = 0;
	parameter.filmSize = 0;
	parameter.pixelSize = EMS_PIXEL_SIZE;
	parameter.prtkSize = EMS_PRTK_SIZE;
	parameter.downSize = EMS_DOWNSIZE;
	parameter.variaBox = EMS_VARIABOX;
	parameter.specSize = EMS_SPEC_SIZE;
	parameter.flmBrit = EMS_BRIGHTNESS;
	parameter.flmCntr = EMS_CONTRAST;
	parameter.spcBrit = EMS_BRIGHTNESS;
	parameter.spcCntr = EMS_CONTRAST;
	parameter.varHi = EMS_VAR_HI;
	parameter.varLo = EMS_VAR_LO;
	parameter.edgeExcu = EMS_EDGE_EXCU;
	parameter.brushSize = EMS_BRUSH;
	parameter.passL = EMS_L_PASS;
	parameter.passH = EMS_H_PASS;
	parameter.transp = EMS_MSK_A;

	// TEMPLATE panel
	parameter.tmpX = 0;
	parameter.tmpY = 0;
	parameter.tmpN = 0;
	parameter.tmpSize = 0;
	parameter.lcfMaskNum = 0;
	parameter.scfMaskNum = 0;
	parameter.rotaRange = EMS_ROTA_RANGE;
	parameter.rotaDelta = EMS_ROTA_DELTA;
	parameter.lcfMaskRad = EMS_LCF_MASKR;
	parameter.scfMaskRad = EMS_SCF_MASKR;
	parameter.scfMaskRadc = EMS_SCF_MASKC;
	parameter.refBrit = EMS_BRIGHTNESS;
	parameter.refCntr = EMS_CONTRAST;

	// PARTICLE panel
	parameter.lcfAve = 0.0;
	parameter.lcfMax = 0.0;
	parameter.scfAve = 0.0;
	parameter.scfMax = 0.0;
	parameter.dcfAve = 0.0;
	parameter.dcfMax = 0.0;
	parameter.lcfCount = 0;
	parameter.scfCount = 0;
	parameter.dcfCount = 0;
	parameter.totCount = 0;
	parameter.lcfThres = EMS_LCF_THRES;
	parameter.scfThres = EMS_SCF_THRES;
	parameter.dcfThres = EMS_DCF_THRES;
	parameter.distance = EMS_PRTK_DIST;
	parameter.monBrit = EMS_BRIGHTNESS;
	parameter.monCntr = EMS_CONTRAST;
	sqrFrameSize = EMS_FRAME_SIZE;

	// FILAMENT panel
	filaX = filaY = 0.0;
	filaL = filaW = filaR = 0.0;
	filaDX = filaDY = 0.0;
	filaDL = filaDW = filaDR = 0.0;
	filaMask = NULL;
	parameter.filaBrdg = EMS_FILA_BRDG;
	parameter.filaBend = EMS_FILA_BEND;
	parameter.filaCount = 0;
	parameter.mon3Brit = EMS_BRIGHTNESS;
	parameter.mon3Cntr = EMS_CONTRAST;

	// default filenames
	strcpy(dirProject, EMS_PRJ_DR);
	strcpy(dirMcrgraph, EMS_PRJ_DR);
	strcpy(dirTemplate, EMS_PRJ_DR);
	strcpy(dirATScreen, EMS_PRJ_DR);
	strcpy(dirParticle, EMS_PRJ_DR);
	strcpy(dirAlignmnt, EMS_PRJ_DR);
	strcpy(dirModeling, EMS_PRJ_DR);
	strcpy(dirFrealign, EMS_PRJ_DR);
}


EMS_Project::~EMS_Project()
{
	delRecord();
	delTempStack();
}


char EMS_Project::newProjSpace(char *proj_name)
{
	char pName[NAMESIZE], dirName[NAMESIZE];
	FILE *fp;

	// remove the path if necessary
	getFilePath(proj_name, pName, projName);
	removeExt(projName, ".ems");

	if ( strlen(projName) == 0 )
		strcpy(project->projName, EMS_PRJ_FN);

	// create PROJECT directory
	sprintf(dirProject, "%s/%s.ems", pName, projName);
	makeDirectory(dirProject);
	sysStatus.emsProject = EMS_TRUE;
	// create MCRGRAPH directory
	sprintf(dirName, "%s/%s", dirProject, EMS_DIR_MCRGRAPH);
	strcpy(dirMcrgraph, dirName);
	makeDirectory(dirName);
	// create PARTICLE directory
	sprintf(dirName, "%s/%s", dirProject, EMS_DIR_PARTICLE);
	strcpy(dirParticle, dirName);
	makeDirectory(dirName);
	// create TEMPLATE directory
	sprintf(dirName, "%s/%s", dirProject, EMS_DIR_TEMPLATE);
	strcpy(dirTemplate, dirName);
	makeDirectory(dirName);
	// create ATSCREEN directory
	sprintf(dirName, "%s/%s", dirProject, EMS_DIR_ATSCREEN);
	strcpy(dirATScreen, dirName);
	makeDirectory(dirName);

	// write project space record
	sprintf(dirName, "%s/%s.sys", dirProject, projName);
	if ( (fp=fopen(dirName, "w")) == NULL ) return(EMS_FALSE);

	fprintf(fp, "%s\n", projName);
	fprintf(fp, "%s\n", dirProject);
	fclose(fp);

	return(EMS_TRUE);
}


char EMS_Project::loadProjSpace(char *proj_name)
{
	int nameSize;
	FILE *fp;

	// read project space record
	if ( (fp=fopen(proj_name, "r")) == NULL ) return(EMS_FALSE);

	fgets(projName, NAMESIZE, fp);
	nameSize = strlen(projName) - 1;
	projName[nameSize] = '\0';

	fgets(dirProject, NAMESIZE, fp);
	nameSize = strlen(dirProject) - 1;
	dirProject[nameSize] = '\0';
	fclose(fp);

	// set directory structure
	sprintf(dirMcrgraph, "%s/%s", dirProject, EMS_DIR_MCRGRAPH);
	sprintf(dirParticle, "%s/%s", dirProject, EMS_DIR_PARTICLE);
	sprintf(dirTemplate, "%s/%s", dirProject, EMS_DIR_TEMPLATE);
	sprintf(dirATScreen, "%s/%s", dirProject, EMS_DIR_ATSCREEN);

	return(EMS_TRUE);
}


char EMS_Project::loadProject(char *file_name)
{
	int nameSize, index, noteSize;
	char buffer[LINE_MAX], emfilm_fn[NAMESIZE], templt_fn[NAMESIZE], keynote_fn[NAMESIZE];
	char loadM_status, loadT_status, keyNote[LINE_MAX+1];
	FILE *fp;

	if ( (fp = fopen(file_name, "rb")) == NULL ) {
		errReport("Cannot open the project file!");
		return(EMS_FALSE);
	}

	// see recordScreenMap() for header format
	fgets(buffer, LINE_MAX, fp);
	nameSize = strlen(buffer) - 1;
	buffer[nameSize] = '\0';

	// set projName & filmName
	for ( index=0; index<nameSize; index++ ) {
		if ( !strncmp(buffer+index, " : ", 3) ) break;
	}

	if ( index == nameSize ) {
		errReport("This is not a screening project!");
		return(EMS_FALSE);
	}

	strcpy(projName, buffer);
	projName[index] = '\0';
	strcpy(filmName, buffer+index+3);

	// read project directory
	fgets(buffer, LINE_MAX, fp);
	nameSize = strlen(buffer) - 1;
	buffer[nameSize] = '\0';
	strcpy(dirProject, buffer);

	// read micrograph filename
	fgets(buffer, LINE_MAX, fp);
	nameSize = strlen(buffer) - 1;
	buffer[nameSize] = '\0';
	strcpy(emfilm_fn, buffer);

	// read template filename
	fgets(buffer, LINE_MAX, fp);
	nameSize = strlen(buffer) - 1;
	buffer[nameSize] = '\0';
	strcpy(templt_fn, buffer);

	// assign directory names
	if ( strcmp(dirProject, EMS_PRJ_DR ) ) {
		sprintf(dirMcrgraph, "%s/%s", dirProject, EMS_DIR_MCRGRAPH);
		sprintf(dirTemplate, "%s/%s", dirProject, EMS_DIR_TEMPLATE);
		sprintf(dirATScreen, "%s/%s", dirProject, EMS_DIR_ATSCREEN);
		sprintf(dirParticle, "%s/%s", dirProject, EMS_DIR_PARTICLE);
		sysStatus.emsProject = EMS_TRUE;
	}

	// load system status and parameters
	fread(&sysStatus, sizeof(struct EMS_SYSSTATUS), 1, fp);
	fread(&parameter, sizeof(struct EMS_PARAMETER), 1, fp);
	fclose(fp);

	// load micrograph, template & screening map
	loadM_status = loadMicrograph(emfilm_fn);
	loadT_status = loadTemplate(templt_fn);
	sysStatus.mapRestore = restoreScreenMap();

	// try to load keynote
	if ( emsRunMode == EMS_GUI_MODE ) {

		EMS_DP->dspKeyNote->value("\0");
		sprintf(keynote_fn, "%s/%s.knt", dirMcrgraph, filmName);
		
		if ( (fp=fopen(keynote_fn, "rb")) != NULL ) {
			fread(&noteSize, sizeof(int), 1, fp);
			fread(keyNote, sizeof(char), noteSize, fp);
			fclose(fp);
			
			keyNote[noteSize] = '\0';
			EMS_DP->dspKeyNote->value(keyNote);
		}
	}

	if ( (loadM_status==EMS_TRUE) && (loadT_status==EMS_TRUE) ) {
		msgReport("Screening project loading is complete");
		return(EMS_TRUE);
	}
	else if ( (loadM_status==EMS_FALSE) && (loadT_status==EMS_TRUE) ) {
		errReport("Cannot open the EM micrograph file!");
		return(EMS_FALSE);
	}
	else if ( (loadM_status==EMS_TRUE) && (loadT_status==EMS_FALSE) ) {
		errReport("Cannot open the template file!");
		return(EMS_FALSE);
	}
	else {
		errReport("Cannot open the micrograph and template files!");
		return(EMS_FALSE);
	}
}


char EMS_Project::saveProject(char *file_name)
{
	int noteSize;
	char status, keynote_fn[NAMESIZE], keyNote[LINE_MAX+1];
	FILE *fp;
/*
	// set project ID (totally unnecessary)
	strcpy(projName, file_name);
	nameSize = strlen(projName);

	// remove ".prj" extension
	if ( !strncmp(projName+nameSize-4, ".prj", 4) )
		projName[nameSize-4] = '\0';
*/
	status = recordScreenMap();

	// try to save keynote
	sprintf(keynote_fn, "%s/%s.knt", dirMcrgraph, filmName);
	strncpy(keyNote, EMS_DP->dspKeyNote->value(), LINE_MAX);
	keyNote[LINE_MAX] = '\0';
	noteSize = strlen(keyNote);

	if ( noteSize > 0 ) {
		if ( (fp=fopen(keynote_fn, "wb")) != NULL ) {
			fwrite(&noteSize, sizeof(int), 1, fp);
			fwrite(keyNote, sizeof(char), noteSize, fp);
			fclose(fp);
		}
		else
			errReport("Unable to save the keynote!");
	}

	return(status);
}


void EMS_Project::clearProject(void)
{
	ptk.clear();
	pwr.clear();
	film.clear();
	filmMask.clear();

	lcfMap.clear();
	scfMap.clear();
	dcfMap.clear();
	stdMap.clear();
	std2Map.clear();
	lcfMask.clear();
	scfMask.clear();

	lcfPeak.clear();
	scfPeak.clear();
	dcfPeak.clear();

	dispFilm.clear();
	dispMask.clear();

	delRecord();
	delTempStack();

	templates.clear();
	micrograph.clear();

	resetParam();
}


char EMS_Project::loadMicrograph(char *file_name)
{
	int index, nameSize;
	char pathName[NAMESIZE];

	if ( micrograph.input(file_name) == EMS_FALSE )
		return(EMS_FALSE);
	else
		strcpy(emfilmFN, file_name);

	if ( micrograph.dimZ != 1 ) {
		errReport("Micrograph data has to be a 2D image!");
		return(EMS_FALSE);
	}

	micrograph.exist = EMS_TRUE;
	micrograph.normalize();
	micrograph.histogram();
	resizeMcrgraph();

	// save micrograph data to project space
	if ( emsRunMode == EMS_GUI_MODE ) {
		index = 0;
		nameSize = strlen(file_name);

		// get film filename and use it for related files in the project space
		getFilePath(file_name, pathName, filmName);
		removeExt(filmName, ".mrc");

		setDispFilm();
		EMS_DP->imgDisplay->setFilmTexture(dispFilm);
		
		// display film mask
		dispMask.initC(EMS_FALSE, dispFilm.dimX, dispFilm.dimY);
		clearFilmMask();
		
		// display image quality information
		cPanel->drawImgInfo(0);
	}

	return(EMS_TRUE);
}


char EMS_Project::loadMcrgraphTIF(char *file_name)
{
	int index;
	char pathName[NAMESIZE];

	if ( micrograph.input_tif(file_name) == EMS_FALSE )
		return(EMS_FALSE);
	else
		strcpy(emfilmFN, file_name);

	if ( micrograph.dimZ != 1 ) {
		errReport("Micrograph data has to be a 2D image!");
		return(EMS_FALSE);
	}

	micrograph.exist = EMS_TRUE;
	micrograph.normalize();
	micrograph.histogram();
	resizeMcrgraph();

	// save micrograph data to project space
	if ( emsRunMode == EMS_GUI_MODE ) {
		index = 0;

		// get film filename and use it for related files in the project space
		getFilePath(file_name, pathName, filmName);
		removeExt(filmName, ".tif");

		setDispFilm();
		EMS_DP->imgDisplay->setFilmTexture(dispFilm);
		
		// display film mask
		dispMask.initC(EMS_FALSE, dispFilm.dimX, dispFilm.dimY);
		clearFilmMask();
		
		// display image quality information
		cPanel->drawImgInfo(0);
	}

	return(EMS_TRUE);
}


char EMS_Project::loadMcrgraphSPD(char *file_name)
{
	int index;
	char pathName[NAMESIZE];

	if ( micrograph.input_spd(file_name) == EMS_FALSE )
		return(EMS_FALSE);
	else
		strcpy(emfilmFN, file_name);

	if ( micrograph.dimZ != 1 ) {
		errReport("Micrograph data has to be a 2D image!");
		return(EMS_FALSE);
	}

	micrograph.exist = EMS_TRUE;
	micrograph.normalize();
	micrograph.histogram();
	resizeMcrgraph();

	// save micrograph data to project space
	if ( emsRunMode == EMS_GUI_MODE ) {
		index = 0;

		// get film filename and use it for related files in the project space
		getFilePath(file_name, pathName, filmName);
		removeExt(filmName, ".spd");

		setDispFilm();
		EMS_DP->imgDisplay->setFilmTexture(dispFilm);
		
		// display film mask
		dispMask.initC(EMS_FALSE, dispFilm.dimX, dispFilm.dimY);
		clearFilmMask();
		
		// display image quality information
		cPanel->drawImgInfo(0);
	}

	return(EMS_TRUE);
}


char EMS_Project::saveMicrograph(char *file_name)
{
	char pName[NAMESIZE];
	MRC_MAP mrcMap;

	mrcMap.setDimensn(micrograph.dimX, micrograph.dimY, 1, micrograph.pixelX, micrograph.pixelY, micrograph.pixelZ);
	mrcMap.setDensity(micrograph.density);

	if ( mrcMap.output(file_name) == EMS_TRUE ) {
		strcpy(emfilmFN, file_name);
		getFilePath(file_name, pName, filmName);
		removeExt(filmName, ".mrc");
		return(EMS_TRUE);
	}
	else
		return(EMS_FALSE);
}


void EMS_Project::setDispFilm(void)
{
	int x, xx, y, yy, xt, yt, disp_dimX, disp_dimY;
	float *dst, acc, area;
	
	parameter.dispScal = 1 + (micrograph.dimX + micrograph.dimY - 2)/(2 * EMS_FILM_TEX);
	disp_dimX = micrograph.dimX / parameter.dispScal;
	disp_dimY = micrograph.dimY / parameter.dispScal;

	// to make texture mapping function happy
	disp_dimX -= disp_dimX % 4;
	disp_dimY -= disp_dimY % 4;
	
	dst = new float[disp_dimX * disp_dimY];
	area = parameter.dispScal * parameter.dispScal;
	
	for ( y=0; y<disp_dimY; y++ ) {
		for ( x=0; x<disp_dimX; x++ ) {
			
			acc = 0.0;
			for ( yy=0; yy<parameter.dispScal; yy++ ) {
				yt = y * parameter.dispScal + yy;
				for ( xx=0; xx<parameter.dispScal; xx++ ) {
					xt = x * parameter.dispScal + xx;
					acc += micrograph.density[yt*micrograph.dimX+xt];
				}
			}
			
			dst[y * disp_dimX + x] = acc / area;
		}
	}
	
	dispFilm.setup(dst, disp_dimX, disp_dimY, disp_dimX, disp_dimY);
	EMS_DP->imgDisplay->initDisplay(dispFilm.dimX, dispFilm.dimY);

	delete [] dst;
}


void EMS_Project::clearMicrograph(void)
{
	delRecord();
	micrograph.clear();
	clearFilament();

	film.clear();
	stdMap.clear();
	std2Map.clear();
	filmMask.clear();

	dispFilm.clear();
	dispMask.clear();

	lcfMap.clear();
	scfMap.clear();
	dcfMap.clear();
	lcfPeak.clear();
	scfPeak.clear();
	dcfPeak.clear();
	lcfMask.clear();
	scfMask.clear();

	parameter.imgX = 0;
	parameter.imgY = 0;
	parameter.filmSize = 0;
	parameter.totCount = 0;
	inspectID = inspectUD = -1;

//	parameter.downSize = EMS_DOWNSIZE;
//	parameter.prtkSize = EMS_PRTK_SIZE;
//	parameter.pixelSize = EMS_PIXEL_SIZE;
	sysStatus.mapRestore = EMS_FALSE;

	if ( emsRunMode == EMS_GUI_MODE ) {
		EMS_DP->dspFilmFile->value("\0");
		EMS_DP->dspFilmX->value(0);
		EMS_DP->dspFilmY->value(0);
		EMS_DP->dspDownSize->value(parameter.downSize);
		EMS_DP->dspPrtkSize->value(parameter.prtkSize);
		EMS_DP->dspPixSize->value(parameter.pixelSize);
		EMS_DP->dspCount->value(0);
		cPanel->clearImgInfo();
		cPanel->clearMonitor();
		cPanel->monitorFila(NULL, 0, 0);
		EMS_DP->imgDisplay->clearFocus();
		EMS_DP->imgDisplay->clearFilmMarker();
		EMS_DP->imgDisplay->clearFilaBoxing();
	}
}


void EMS_Project::buildFilmMask(void)
{
	int x, y, x0, y0, x1, y1, xx, yy;
	long index, index2, maskNum;
	float varHi_t, varLo_t, aveVar;
	double xc, yc, r2, R2, dist, dist2;
	EMS_MAP stdDisp, locMask;

	if ( dispFilm.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}
	else
		msgReport("Inspecting micrograph ...");

	// clear existing mask
	dispMask.resetC(EMS_FALSE);

	for ( y=0; y<dispMask.dimY; y++ ) {
		for ( x=0; x<parameter.edgeExcu; x++ ) {
			dispMask.dataC[y * dispMask.dimX + x] = EMS_TRUE;
			dispMask.dataC[y * dispMask.dimX + (dispMask.dimX-1-x)] = EMS_TRUE;
		}
	}

	for ( y=0; y<parameter.edgeExcu; y++ ) {
		for ( x=0; x<dispMask.dimX; x++ ) {
			dispMask.dataC[y * dispMask.dimX + x] = EMS_TRUE;
			dispMask.dataC[(dispMask.dimY-1-y) * dispMask.dimX + x] = EMS_TRUE;
		}
	}

	xc = yc = parameter.variaBox / 2.0;
	R2 = xc * xc - EMS_ZERO;
	maskNum = 0;
	
	locMask.initC(EMS_FALSE, parameter.variaBox, parameter.variaBox);
	
	for ( y=0; y<parameter.variaBox; y++ ) {
		for ( x=0; x<parameter.variaBox; x++ ) {
			
			index = y * parameter.variaBox + x;
			r2 = (x - xc) * (x - xc) + (y - yc) * (y - yc);
			
			if ( r2 <= R2 ) {
				maskNum ++;
				locMask.dataC[index] = (char)EMS_TRUE;
			}
		}
	}
	
	localSTD(locMask, dispFilm, stdDisp, (float)(1.0/maskNum));
	
	aveVar = 0.0;
	for ( index=0; index<dispFilm.mapSize; index++ )
		aveVar += stdDisp.dataF[index];

	aveVar /= dispFilm.mapSize;
	varHi_t = parameter.varHi * aveVar;
	varLo_t = parameter.varLo * aveVar;

	// variance thresholding
	dist = EMS_PRTK_HALF * parameter.prtkSize / (parameter.pixelSize * parameter.dispScal);
	dist2 = dist * dist;

	// exclude a zone of particle-size around each high/low variance spot
	for ( y=0; y<dispMask.dimY; y++ ) {
		for ( x=0; x<dispMask.dimX; x++ ) {
			index = y * dispMask.dimX + x;
			
			if ( dispMask.dataC[index] == EMS_FALSE ) {
				if ( (stdDisp.dataF[index]>=varHi_t) || (stdDisp.dataF[index]<=varLo_t) ) {

					dispMask.dataC[index] = EMS_TRUE;
					
					x0 = (int)floor(x - dist);
					if ( x0 < 0 ) x0 = 0;
					
					x1 = (int)ceil(x + dist);
					if ( x1 >= dispMask.dimX ) x1 = dispMask.dimX - 1;
					
					y0 = (int)floor(y - dist);
					if ( y0 < 0 ) y0 = 0;
					
					y1 = (int)ceil(y + dist);
					if ( y1 >= dispMask.dimY ) y1 = dispMask.dimY - 1;
					
					for ( yy=y0; yy<=y1; yy++ ) {
						for ( xx=x0; xx<=x1; xx++ ) {
							index2 = yy * dispMask.dimX + xx;

							if ( (((xx-x)*(xx-x)+(yy-y)*(yy-y))<=dist2) && (index!=index2) ) {
								if ( (stdDisp.dataF[index2]<varHi_t) && (stdDisp.dataF[index2]>varLo_t) )
									dispMask.dataC[index2] = EMS_TRUE;
							}
						}
					}
				}
			}
		}
	}

	sysStatus.syncMask = EMS_FALSE;
	EMS_DP->imgDisplay->setMaskTexture(dispMask);
	msgReport("Micrograph inspection is complete");
}


void EMS_Project::syncFilmMask(void)
{
	int x, y, x0, y0, x1, y1;
	long index;
	double v00, v01, v10, v11, m00, m11;
	double xf, yf, scaling, maskVal;
	char *mask_map;

	if ( emsRunMode == EMS_COM_MODE ) return;
	if ( sysStatus.syncMask == EMS_TRUE ) return;

	if ( film.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}

	mask_map = new char[film.mapSize];
	scaling = 1.0 * parameter.downSize / parameter.dispScal;

	for ( y=0; y<film.dimY; y++ ) {
		yf = y * scaling;
		for ( x=0; x<film.dimX; x++ ) {
			xf = x * scaling;
			index = y * film.dimX + x;
			
			x0 = (int)floor(xf); x1 = x0 + 1;
			if ( x1 >= dispFilm.dimX ) {
				mask_map[index] = EMS_TRUE;
				continue;
			}
			
			y0 = (int)floor(yf); y1 = y0 + 1;
			if ( y1 >= dispFilm.dimY ) {
				mask_map[index] = EMS_TRUE;
				continue;
			}
			
			v00 = dispMask.dataC[y0 * dispMask.dimX + x0];
			v10 = dispMask.dataC[y0 * dispMask.dimX + x1];
			v01 = dispMask.dataC[y1 * dispMask.dimX + x0];
			v11 = dispMask.dataC[y1 * dispMask.dimX + x1];
			m00 = (x1-xf) * v00 + (xf-x0) * v10;
			m11 = (x1-xf) * v01 + (xf-x0) * v11;
			maskVal = (y1-yf) * m00 + (yf-y0) * m11;

			if ( maskVal >= 0.5 )
				mask_map[index] = EMS_TRUE;
			else
				mask_map[index] = EMS_FALSE;
		}
	}

	filmMask.setup(mask_map, film.dimX, film.dimY, 0, 0);

	sysStatus.syncMask = EMS_TRUE;
	delete [] mask_map;
}


void EMS_Project::restoreFilmMask(void)
{
	long index;
	int x, y, x0, y0, x1, y1;
	double v00, v01, v10, v11, m00, m11;
	double xf, yf, scaling, maskVal;

	if ( (film.exist==EMS_FALSE) || (record==NULL) ) return;
	
	scaling = 1.0 * parameter.dispScal / parameter.downSize;
	filmMask.initC(EMS_FALSE, parameter.imgX, parameter.imgY);
	dispMask.resetC(EMS_FALSE);

	// set up dispMask[]
	for ( y=0; y<dispMask.dimY; y++ ) {
		yf = y * scaling;
		for ( x=0; x<dispMask.dimX; x++ ) {
			xf = x * scaling;
			index = y * dispMask.dimX + x;
			
			x0 = (int)floor(xf); x1 = x0 + 1;
			if ( x1 >= filmMask.dimX ) {
				dispMask.dataC[index] = EMS_TRUE;
				continue;
			}
			
			y0 = (int)floor(yf); y1 = y0 + 1;
			if ( y1 >= filmMask.dimY ) {
				dispMask.dataC[index] = EMS_TRUE;
				continue;
			}

			v00 = filmMask.dataC[y0 * filmMask.dimX + x0];
			v10 = filmMask.dataC[y0 * filmMask.dimX + x1];
			v01 = filmMask.dataC[y1 * filmMask.dimX + x0];
			v11 = filmMask.dataC[y1 * filmMask.dimX + x1];
			m00 = (x1-xf) * v00 + (xf-x0) * v10;
			m11 = (x1-xf) * v01 + (xf-x0) * v11;
			maskVal = (y1-yf) * m00 + (yf-y0) * m11;

			if ( maskVal >= 0.5 )
				dispMask.dataC[index] = EMS_TRUE;
		}
	}

	EMS_DP->imgDisplay->setMaskTexture(dispMask);
}


void EMS_Project::clearFilmMask(void)
{
	int x, y;

	// safe-guard data availability
	if ( micrograph.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}
	if ( dispMask.exist == EMS_FALSE ) {
		errReport("EM micrograph mask does not exist yet!");
		return;
	}

	dispMask.resetC(EMS_FALSE);

	for ( y=0; y<dispMask.dimY; y++ ) {
		for ( x=0; x<parameter.edgeExcu; x++ ) {
			dispMask.dataC[y * dispMask.dimX + x] = EMS_TRUE;
			dispMask.dataC[y * dispMask.dimX + (dispMask.dimX-1-x)] = EMS_TRUE;
		}
	}

	for ( y=0; y<parameter.edgeExcu; y++ ) {
		for ( x=0; x<dispMask.dimX; x++ ) {
			dispMask.dataC[y * dispMask.dimX + x] = EMS_TRUE;
			dispMask.dataC[(dispMask.dimY-1-y) * dispMask.dimX + x] = EMS_TRUE;
		}
	}

	sysStatus.syncMask = EMS_FALSE;
	EMS_DP->imgDisplay->setMaskTexture(dispMask);
}


void EMS_Project::invContrast(void)
{
	long index;

	// invert original density data
	for ( index=0; index<micrograph.mapSize; index++ )
		micrograph.density[index] *= -1;

	// invert downsized image
	for ( index=0; index<film.mapSize; index++ ) {
		film.dataF[index] *= -1;
		film.rel[index] *= -1;
		film.img[index] *= -1;
	}

	// invert micrograph texture map
	if ( emsRunMode == EMS_GUI_MODE ) {
		
		for ( index=0; index<dispFilm.mapSize; index++ ) {
			dispFilm.dataF[index] *= -1;
			dispFilm.rel[index] *= -1;
			dispFilm.img[index] *= -1;
		}
	}
}


char EMS_Project::loadTemplate(char *file_name)
{
	int index;

	strcpy(templtFN, file_name);
	if ( templates.input(templtFN) == EMS_FALSE ) return(EMS_FALSE);

	if ( (templates.dimX<EMS_TEMP_MIN) || (templates.dimY<EMS_TEMP_MIN) ) {
		errReport("Template size is too small!");
		clearTemplate();
		return(EMS_FALSE);
	}

	templates.exist = EMS_TRUE;
	parameter.tmpX = templates.dimX / parameter.downSize;
	parameter.tmpY = templates.dimY / parameter.downSize;
	parameter.tmpN = templates.dimZ;
	parameter.tmpSize = parameter.tmpX * parameter.tmpY;

	delTempStack();
	refer.ptkStack = new float[templates.mapSize];
	refer.pwrStack = new float[templates.mapSize];
	refer.L1Mask = new char[templates.mapSize];
	refer.L2Mask = new char[templates.mapSize];
	refer.select = new char[parameter.tmpN];
	refer.xc = new int[parameter.tmpN];
	refer.yc = new int[parameter.tmpN];

	// store templates and compute auto-corr function for each one
	memcpy(refer.ptkStack, templates.density, templates.mapSize*sizeof(float));
	calcPowerSpec();

	// setup alignment center & inclusion flag
	for ( index=0; index<parameter.tmpN; index++ ) {
		refer.xc[index] = templates.dimX / 2;
		refer.yc[index] = templates.dimY / 2;
		refer.select[index] = EMS_TRUE;
	}

	// LCF masking at original size
	refer.maskID = EMS_MASK_L1;
	profileMasking(EMS_MASK_L2);
	profileMasking(EMS_MASK_L1);

	// setup templates and spectra display
	if ( emsRunMode == EMS_GUI_MODE )
		EMS_DP->imgDisplay->setTempTexture();

	return(EMS_TRUE);
}


void EMS_Project::focusTemplate(int tid)
{
	// GUI takes base-one
	refer.editFrame = tid - 1;
}


void EMS_Project::clearTemplate(void)
{
	parameter.tmpX = 0;
	parameter.tmpY = 0;
	parameter.tmpN = 0;
	parameter.tmpSize = 0;
	parameter.lcfMaskNum = 0;
	parameter.scfMaskNum = 0;
	parameter.lcfCount = 0;
	parameter.scfCount = 0;
	parameter.dcfCount = 0;
	parameter.totCount = 0;

	ptk.clear();
	pwr.clear();
	lcfMap.clear();
	scfMap.clear();
	dcfMap.clear();
	stdMap.clear();
	std2Map.clear();
	lcfMask.clear();
	scfMask.clear();
	lcfPeak.clear();
	scfPeak.clear();
	dcfPeak.clear();

	delTempStack();
	templates.clear();

	refer.editFrame = -1;
	inspectID = inspectUD = -1;
	refer.maskID = EMS_MASK_L1;
	sysStatus.mapRestore = EMS_FALSE;

	if ( emsRunMode == EMS_GUI_MODE ) {
		EMS_DP->dspCount->value(0);
		cPanel->clearMonitor();
	}
}


void EMS_Project::syncTempMask(void)
{
	long pageSize, index, index_src, index_dst;
	char *msk;

	if ( (refer.editFrame<0) || (refer.editFrame>=parameter.tmpN) ) return;

	if ( refer.maskID == EMS_MASK_L1 )
		msk = refer.L1Mask;
	else if ( refer.maskID == EMS_MASK_L2 )
		msk = refer.L2Mask;

	pageSize = templates.dimX * templates.dimY;
	index_src = refer.editFrame * pageSize;

	for ( index=(refer.editFrame+1); index<parameter.tmpN; index++ ) {
		index_dst = index * pageSize;
		memcpy(msk+index_dst, msk+index_src, sizeof(char)*pageSize);
	}
}


void EMS_Project::clearTempMask(void)
{
	long index;
	char *msk;

	if ( refer.maskID == EMS_MASK_L1 )
		msk = refer.L1Mask;
	else if ( refer.maskID == EMS_MASK_L2 )
		msk = refer.L2Mask;

	for ( index=0; index<templates.mapSize; index++ )
		msk[index] = EMS_TRUE;
}


void EMS_Project::setIncExcState(char status)
{
	int index;

	for ( index=0; index<parameter.tmpN; index++ )
		refer.select[index] = status;
}


char EMS_Project::loadParticlePTK(char *file_name)
{
	char buffer[LINE_MAX];
	int x, y;
	long index;
	FILE *fp;
	
	fp = fopen(file_name, "r");
	
	if ( fp == NULL ) {
		errReport("Cannot load the particle selection!");
		return(EMS_FALSE);
	}

	// read header (no action)
	fgets(buffer, LINE_MAX, fp);

	// read particle selecion
	while ( fgets(buffer, LINE_MAX, fp) != NULL ) {
		sscanf(buffer, "%d %d", &x, &y);

		x /= parameter.downSize;
		y /= parameter.downSize;

		if ( (x<0) || (x>=parameter.imgX) || (y<0) || (y>=parameter.imgY) ) {
			errReport("Incorrect particle coordinates!");
			return(EMS_FALSE);
		}

		index = y * parameter.imgX + x;
		if ( record[index].select <= EMS_SEL_NON ) {
			record[index].select = EMS_SEL_MAN;
			parameter.totCount ++;
		}
	}
	
	fclose(fp);
	return(EMS_TRUE);
}


char EMS_Project::loadParticlePLT(char *file_name)
{
	char buffer[LINE_MAX];
	long index;
	float xf, yf;
	int x, y;
	FILE *fp;
	
	fp = fopen(file_name, "r");
	
	if ( fp == NULL ) {
		errReport("Cannot load the particle selection!");
		return(EMS_FALSE);
	}

	// read particle selecion
	while ( fgets(buffer, LINE_MAX, fp) != NULL ) {
		sscanf(buffer, "%f %f", &xf, &yf);

		x = (int)yf / parameter.downSize;
		y = (int)(micrograph.dimY - xf) / parameter.downSize;

		if ( (x<0) || (x>=parameter.imgX) || (y<0) || (y>=parameter.imgY) ) {
			errReport("Incorrect particle coordinates!");
			return(EMS_FALSE);
		}

		index = y * parameter.imgX + x;
		if ( record[index].select <= EMS_SEL_NON ) {
			record[index].select = EMS_SEL_MAN;
			parameter.totCount ++;
		}
	}
	
	fclose(fp);
	return(EMS_TRUE);
}


char EMS_Project::loadParticleSPD(char *file_name)
{
	char buffer[LINE_MAX];
	int sid, er2;
	float x, y;
	long index;
	FILE *fp;
	
	fp = fopen(file_name, "r");
	
	if ( fp == NULL ) {
		errReport("Cannot load the particle selection!");
		return(EMS_FALSE);
	}

	// read particle selecion
	while ( fgets(buffer, LINE_MAX, fp) != NULL ) {

		// skip comment lines
		while ( buffer[1] == ';' )
			fgets(buffer, LINE_MAX, fp);

		sscanf(buffer, "%d %d %f %f", &sid, &er2, &x, &y);

		x /= parameter.downSize;
		y /= parameter.downSize;

		if ( (x<0) || (x>=parameter.imgX) || (y<0) || (y>=parameter.imgY) ) {
			errReport("Incorrect particle coordinates!");
			return(EMS_FALSE);
		}

		index = (int)y * parameter.imgX + (int)x;
		if ( record[index].select <= EMS_SEL_NON ) {
			record[index].select = EMS_SEL_MAN;
			parameter.totCount ++;
		}
	}
	
	fclose(fp);
	return(EMS_TRUE);
}


void EMS_Project::saveParticlePTK(char *file_name)
{
	int x, y, x_t, y_t, id_t, count;
	float a_t, b_t, g_t, score;
	long index;
	FILE *fp;
	
	fp = fopen(file_name, "w");
	
	if ( fp == NULL ) {
		errReport("Cannot write out the selected particles!");
		return;
	}

	// count number of particles
	count = 0;

	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {
			if ( record[y*parameter.imgX+x].select > EMS_SEL_NON )
				count ++;
		}
	}

	// write header
	fprintf(fp, "%s : %d\n", EMS_PARTIC_H, count);

	// write data
	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {
			index = y * parameter.imgX + x;
			if ( record[index].select > EMS_SEL_NON ) {
				x_t = x * parameter.downSize;
				y_t = y * parameter.downSize;
				id_t = record[index].tmpid + 1;
				a_t = record[index].euler;
				b_t = 0.0;
				g_t = 0.0;
				score = record[index].dcfScore;

				fprintf(fp, "%8d %8d %4d %6.2f %6.2f %6.2f %6.3f\n",
					x_t, y_t, id_t, a_t, b_t, g_t, score);
			}
		}
	}
	
	fclose(fp);
}


char EMS_Project::saveParticleMRC(char *file_name)
{
	int x, y, count;
	long index;
	char status, flagPrecise;
	char fileName[NAMESIZE], buffer[LINE_MAX];
	FILE *fp;

	if ( sqrFrameSize < 4 ) {
		errReport("Incorrect particle image frame size!");
		return(EMS_FALSE);
	}

	// attempt to open and match "{filmName}_stack.ptk" file
	sprintf(fileName, "%s/%s_stack.ptk", dirParticle, filmName);
	flagPrecise = EMS_FALSE;

	if ( (fp=fopen(fileName, "r")) != NULL ) {
		// compare number of particles (PTK v/s totCount)
		fgets(buffer, LINE_MAX, fp);
		sscanf(buffer+strlen(EMS_PARTIC_H)+3, "%d", &count);
		
		// verify each entry
		if ( count == parameter.totCount ) {
			while ( fgets(buffer, LINE_MAX, fp) != NULL ) {
				sscanf(buffer, "%d %d", &x, &y);
				
				x /= parameter.downSize;
				y /= parameter.downSize;
				
				if ( (x<0) || (x>=parameter.imgX) || (y<0) || (y>=parameter.imgY) ) {
					flagPrecise = EMS_FALSE;
					break;
				}
				
				index = y * parameter.imgX + x;
				if ( record[index].select <= EMS_SEL_NON ) {
					flagPrecise = EMS_FALSE;
					break;
				}
				else
					flagPrecise = EMS_TRUE;
			}
		}

		fclose(fp);
	}

	if ( flagPrecise ) {
		// crop image frame according to ${filmName_stack.ptk} coordinates
		status = savePreciseMRC(file_name);
	}
	else {
		// crop image frame according to record[] coordinates (maybe downsized)
		status = saveDisplayMRC(file_name);
	}

	return(status);
}


char EMS_Project::savePreciseMRC(char *file_name)
{
	int x, y, x0, y0, x1, y1, xx, yy, entry;
	long index, pageSize, dataSize, src, dst;
	float *map_data;
	char fileName[NAMESIZE], buffer[LINE_MAX];
	FILE *fp;

	// open PTK coordinates file
	sprintf(fileName, "%s/%s_stack.ptk", dirParticle, filmName);

	fp = fopen(fileName, "r");
	fgets(buffer, LINE_MAX, fp);
	sscanf(buffer+strlen(EMS_PARTIC_H)+3, "%d", &entry);

	// write out particle image frame by frame
	pageSize = sqrFrameSize * sqrFrameSize;
	map_data = new float[pageSize * entry];

	for ( index=0; index<entry; index++ ) {

		fgets(buffer, LINE_MAX, fp);
		sscanf(buffer, "%d %d", &x, &y);

		x0 = x - sqrFrameSize/2;
		x1 = x0 + sqrFrameSize;
		y0 = y - sqrFrameSize/2;
		y1 = y0 + sqrFrameSize;
		
		for ( yy=y0; yy<y1; yy++ ) {
			for ( xx=x0; xx<x1; xx++ ) {
				src = yy * micrograph.dimX + xx;
				dst = index * pageSize + (yy-y0) * sqrFrameSize + (xx-x0);
				
				if ( (xx<0) || (xx>=micrograph.dimX) || (yy<0) || (yy>=micrograph.dimY) )
					map_data[dst] = micrograph.aveInt;
				else
					map_data[dst] = micrograph.density[src];
			}
		}
		
	}

	// invert intensity contrast if necessary
	if ( sysStatus.imgInvert == EMS_TRUE ) {
		dataSize = pageSize * entry;
		for ( index=0; index<dataSize; index++ )
			map_data[index] *= -1.0f;
	}

	// write out particle frames
	writeMap3D(file_name, sqrFrameSize, sqrFrameSize, entry, map_data);

	delete [] map_data;
	fclose(fp);

	return(EMS_TRUE);
}


char EMS_Project::saveDisplayMRC(char *file_name)
{
	int x, y, x0, y0, x1, y1, xx, yy;
	int x_t, y_t, id_t, count=0;
	long index, pageSize, dataSize, src, dst;
	float *map_data, a_t, b_t, g_t, score;
	char fileName[NAMESIZE];
	FILE *fp;

	// count selections matching "inclusive" templates
	count = 0;
	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {
			index = y * parameter.imgX + x;

			if ( (record[index].select>EMS_SEL_NON) && (record[index].tmpid<0) )
				count ++;
			else if ( (record[index].select>EMS_SEL_NON) && refer.select[record[index].tmpid] )
				count ++;
		}
	}

	if ( count == 0 ) {
		errReport("No particle has been selected!");
		return(EMS_FALSE);
	}

	// prepare "{filmName}_stack.ptk" write-out
	pageSize = sqrFrameSize * sqrFrameSize;
	sprintf(fileName, "%s/%s_stack.ptk", dirParticle, filmName);

	if ( (fp=fopen(fileName, "w")) == NULL ) {
		errReport("Cannot write out the selected particles!");
		return(EMS_FALSE);
	}

	map_data = new float[pageSize * count];
	fprintf(fp, "%s : %d\n", EMS_PARTIC_H, count);

	// write out particle image frame by frame
	count = 0;
	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {
			index = y * parameter.imgX + x;

			if ( (record[index].select>EMS_SEL_NON) && (record[index].tmpid<0) ) {
				x0 = x * parameter.downSize - sqrFrameSize/2;
				x1 = x0 + sqrFrameSize;
				y0 = y * parameter.downSize - sqrFrameSize/2;
				y1 = y0 + sqrFrameSize;
			}
			else if ( (record[index].select>EMS_SEL_NON) && refer.select[record[index].tmpid] ) {
				x0 = x * parameter.downSize - sqrFrameSize/2;
				x1 = x0 + sqrFrameSize;
				y0 = y * parameter.downSize - sqrFrameSize/2;
				y1 = y0 + sqrFrameSize;
			}
			else
				continue;

			for ( yy=y0; yy<y1; yy++ ) {
				for ( xx=x0; xx<x1; xx++ ) {
					src = yy * micrograph.dimX + xx;
					dst = count * pageSize + (yy-y0) * sqrFrameSize + (xx-x0);

					if ( (xx<0) || (xx>=micrograph.dimX) || (yy<0) || (yy>=micrograph.dimY) )
						map_data[dst] = micrograph.aveInt;
					else
						map_data[dst] = micrograph.density[src];
				}
			}

			// write out corresponding PTK coordinates
			x_t = x * parameter.downSize;
			y_t = y * parameter.downSize;
			id_t = record[index].tmpid + 1;
			a_t = record[index].euler;
			b_t = 0.0;
			g_t = 0.0;
			score = record[index].dcfScore;
			count ++;

			fprintf(fp, "%8d %8d %4d %6.2f %6.2f %6.2f %6.3f\n",
				x_t, y_t, id_t, a_t, b_t, g_t, score);
		}
	}

	// invert intensity contrast if necessary
	if ( sysStatus.imgInvert == EMS_TRUE ) {
		dataSize = pageSize * count;
		for ( index=0; index<dataSize; index++ )
			map_data[index] *= -1.0f;
	}

	// write out particle frames
	writeMap3D(file_name, sqrFrameSize, sqrFrameSize, count, map_data);

	delete [] map_data;
	fclose(fp);

	return(EMS_TRUE);
}


void EMS_Project::saveParticlePLT(char *file_name)
{
	int x, y;
	float xf, yf;
	FILE *fp;
	
	fp = fopen(file_name, "w");
	
	if ( fp == NULL ) {
		errReport("Cannot write out the selected particles!");
		return;
	}

	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {
			if ( record[y*parameter.imgX+x].select > EMS_SEL_NON ) {
				xf = micrograph.dimY - y * parameter.downSize;
				yf = x * parameter.downSize;
				fprintf(fp, "   %8.3f       %8.3f       1.000000    \n", xf, yf);
			}
		}
	}

	fclose(fp);
}


void EMS_Project::saveParticleSPD(char *file_name)
{
	int x, y, sid;
	float xf, yf;
	FILE *fp;
	
	fp = fopen(file_name, "w");
	
	if ( fp == NULL ) {
		errReport("Cannot write out the selected particles!");
		return;
	}

	// write header
	fprintf(fp, " ; EMS particle selection, output in SPD format\n");
	sid = 0;

	// write data
	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {
			if ( record[y*parameter.imgX+x].select > EMS_SEL_NON ) {
				sid ++;
				xf = x * parameter.downSize;
				yf = y * parameter.downSize;
				fprintf(fp, "%5d 2 %12.6f %12.6f\n", sid, xf, yf);
			}
		}
	}
	
	fclose(fp);
}


void EMS_Project::clearParticle(void)
{
	long index;
	
	if ( (record==NULL) || (parameter.totCount<=0) ) return;
	
	for ( index=0; index<parameter.filmSize; index++ )
		record[index].select = EMS_SEL_NON;
	
	parameter.totCount = 0;
	inspectID = inspectUD = -1;

	if ( emsRunMode == EMS_GUI_MODE )
		EMS_DP->dspCount->value(0);
}


char EMS_Project::syncPrtkSelect(char *pathName, char *stakName)
{
	char fileName[NAMESIZE];

	// verify file consistency
	if ( strcmp(filmName, stakName) || strcmp(dirParticle, pathName) ) {
		errReport("EM micrograph and particle stack mismatch!");
		return(EMS_FALSE);
	}

	// reload particle PTK record
	sprintf(fileName, "%s/%s_stack.ptk", pathName, filmName);
	clearParticle();
	loadParticlePTK(fileName);

	projSaveStatus = EMS_FALSE;
	msgReport("Micrograph and particle stack have been synchronized.");

	return(EMS_TRUE);
}


void EMS_Project::resizeMcrgraph(void)
{
	int x, xx, y, yy;
	long index;
	float *dst, acc, area, ave=0.0, std=0.0, var=0.0;

	if ( micrograph.exist == EMS_FALSE ) return;
	
	parameter.imgX = micrograph.dimX / parameter.downSize;
	parameter.imgY = micrograph.dimY / parameter.downSize;
	parameter.filmSize = parameter.imgX * parameter.imgY;

	dst = new float[parameter.filmSize];
	area = parameter.downSize * parameter.downSize;
	
	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {
			acc = 0.0;
			
			for ( yy=0; yy<parameter.downSize; yy++ ) {
				for ( xx=0; xx<parameter.downSize; xx++ ) {
					index = (y*parameter.downSize+yy)*micrograph.dimX + x*parameter.downSize + xx;
					acc += micrograph.density[index];
				}
			}

			acc /= area;
			ave += acc;
			var += acc * acc;
			dst[y * parameter.imgX + x] = acc;
		}
	}

	ave /= parameter.filmSize;
	var /= parameter.filmSize;
	std = var - ave * ave;

	if ( std > 0.0 )
		std = (float)(sqrt(std));
	else
		std = EMS_ZERO;

	// normalize resized film image
	for ( index=0; index<parameter.filmSize; index++ )
		dst[index] = (dst[index] - ave) / std;

	film.setup(dst, parameter.imgX, parameter.imgY, parameter.imgX, parameter.imgY);
	sysStatus.syncMask = EMS_FALSE;

	// allocate memory for bookkeeper and clear record
	delRecord();
	record = new struct EMS_RECORD[parameter.filmSize];
	clearAllRecord();

	// clear relevant maps
	stdMap.clear();
	std2Map.clear();
	lcfMap.clear();
	lcfPeak.clear();
	scfMap.clear();
	scfPeak.clear();
	dcfMap.clear();
	dcfPeak.clear();

	delete [] dst;
}


void EMS_Project::resizeTemplate(void)
{
	if ( templates.exist == EMS_FALSE ) return;

	parameter.tmpX = templates.dimX / parameter.downSize;
	parameter.tmpY = templates.dimY / parameter.downSize;
	parameter.tmpSize = parameter.tmpX * parameter.tmpY;
}


void EMS_Project::defineTempMask(int frame, char *mask, float rotate)
{
	int x, y, x0, y0, x1, y1, xx, yy, xt, yt, dimX, dimY, dim;
	double xc, yc, r2, area, scf_rd, scf_r2, scf_rcd, scf_rc2;
	double xf, yf, dA, dB, dC, dD, dE, dF, msk_acc, cosAng, sinAng;
	long index, page;
	char *msk_t;

	dimX = templates.dimX;
	dimY = templates.dimY;
	area = parameter.downSize * parameter.downSize;
	
	// in the original dimension
	page = frame * dimX * dimY;
	xc = 0.5 * parameter.tmpX * parameter.downSize;
	yc = 0.5 * parameter.tmpY * parameter.downSize;

	rotate *= DEG2RAD;
	cosAng = cos(rotate);
	sinAng = sin(rotate);

	// masking at downsized scale
	msk_t = new char[parameter.tmpSize];
	parameter.lcfMaskNum = 0;

	// rotate and accumulate pixel values
	for ( y=0; y<parameter.tmpX; y++ ) {
		for ( x=0; x<parameter.tmpY; x++ ) {

			msk_acc = 0.0;
			index = y * parameter.tmpX + x;

			for ( yy=0; yy<parameter.downSize; yy++ ) {
				yt = y * parameter.downSize + yy;
				for ( xx=0; xx<parameter.downSize; xx++ ) {
					xt = x * parameter.downSize + xx;

					xf = (xt - xc) * cosAng + (yt - yc) * sinAng + refer.xc[frame];
					yf = (xc - xt) * sinAng + (yt - yc) * cosAng + refer.yc[frame];
					x0 = (int)floor(xf); x1 = x0 + 1;
					y0 = (int)floor(yf); y1 = y0 + 1;

					// safe-guard pixel location
					if ( (x0<0) || (x1>=dimX) || (y0<0) || (y1>=dimY) ) {
						msk_acc = 0.0;
						xx = parameter.downSize;
						yy = parameter.downSize;
						break;
					}
					
					dA = mask[page + y0 * dimX + x0];
					dB = mask[page + y0 * dimX + x1];
					dC = mask[page + y1 * dimX + x0];
					dD = mask[page + y1 * dimX + x1];
					dE = (x1-xf) * dA + (xf-x0) * dB;
					dF = (x1-xf) * dC + (xf-x0) * dD;
					msk_acc += (y1-yf) * dE + (yf-y0) * dF;
				}
			}

			if ( (msk_acc / area) >= 0.5 ) {
				msk_t[index] = EMS_TRUE;
				parameter.lcfMaskNum ++;
			}
			else
				msk_t[index] = EMS_FALSE;
		}
	}

	lcfMask.setup(msk_t, parameter.tmpX, parameter.tmpY, 0, 0);

	// SCF masking, sometimes redundant
	scfMask.initC(EMS_FALSE, parameter.tmpX, parameter.tmpY);
	parameter.scfMaskNum = 0;

	// setup masking radius
	dimX /= parameter.downSize;
	dimY /= parameter.downSize;
	dim = (dimX <= dimY)? dimX : dimY;
	scf_rd = 0.5 * dim * parameter.scfMaskRad;
	scf_rcd = 0.5 * dim * parameter.scfMaskRadc;
	scf_r2 = scf_rd * scf_rd;
	scf_rc2 = scf_rcd * scf_rcd;

	xc = parameter.tmpX / 2.0;
	yc = parameter.tmpY / 2.0;

	for ( y=0; y<parameter.tmpY; y++ ) {
		for ( x=0; x<parameter.tmpX; x++ ) {
			r2 = (x - xc) * (x - xc) + (y - yc) * (y - yc);
			
			if ( (r2 >= scf_rc2) && (r2 <= scf_r2) ) {
				scfMask.dataC[y*parameter.tmpX+x] = (char)EMS_TRUE;
				parameter.scfMaskNum ++;
			}
		}
	}

	delete [] msk_t;
}


void EMS_Project::defineTemplate(int frame, float *ptk_t, float *pwr_t, float rotate)
{
	int x, xx, y, yy, xt, yt, x0, x1, y0, y1, dimX, dimY;
	long page, index;
	double xc, yc, xf, yf, dA, dB, dC, dD, dE, dF;
	double ptk_acc, pwr_acc, cosAng, sinAng, area;
	double ave, var, std, ave2, var2, std2, val;
	
	dimX = templates.dimX;
	dimY = templates.dimY;
	area = parameter.downSize * parameter.downSize;

	// in the original dimension
	page = frame * dimX * dimY;
	xc = 0.5 * parameter.tmpX * parameter.downSize;
	yc = 0.5 * parameter.tmpY * parameter.downSize;

	rotate *= DEG2RAD;
	cosAng = cos(rotate);
	sinAng = sin(rotate);

	for ( y=0; y<parameter.tmpY; y++ ) {
		for ( x=0; x<parameter.tmpX; x++ ) {
			index = y * parameter.tmpX + x;
			ptk_acc = pwr_acc = 0.0;

			if ( (lcfMask.dataC[index]==EMS_FALSE) && (scfMask.dataC[index]==EMS_FALSE) ) {
				ptk_t[index] = 0.0;
				pwr_t[index] = 0.0;
				continue;
			}

			// accumulate ptk[]
			if ( lcfMask.dataC[index] == EMS_TRUE ) {
				
				for ( yy=0; yy<parameter.downSize; yy++ ) {
					yt = y * parameter.downSize + yy;
					for ( xx=0; xx<parameter.downSize; xx++ ) {
						xt = x * parameter.downSize + xx;
						
						xf = (xt - xc) * cosAng + (yt - yc) * sinAng + refer.xc[frame];
						yf = (xc - xt) * sinAng + (yt - yc) * cosAng + refer.yc[frame];
						x0 = (int)floor(xf); x1 = x0 + 1;
						y0 = (int)floor(yf); y1 = y0 + 1;

						dA = refer.ptkStack[page + y0 * dimX + x0];
						dB = refer.ptkStack[page + y0 * dimX + x1];
						dC = refer.ptkStack[page + y1 * dimX + x0];
						dD = refer.ptkStack[page + y1 * dimX + x1];
						dE = (x1-xf) * dA + (xf-x0) * dB;
						dF = (x1-xf) * dC + (xf-x0) * dD;
						ptk_acc += (y1-yf) * dE + (yf-y0) * dF;
					}
				}
			}

			// accumulate pwr[]
			if ( scfMask.dataC[index] == EMS_TRUE ) {
				
				for ( yy=0; yy<parameter.downSize; yy++ ) {
					yt = y * parameter.downSize + yy;
					for ( xx=0; xx<parameter.downSize; xx++ ) {
						xt = x * parameter.downSize + xx;
						
						xf = (xt - xc) * cosAng + (yt - yc) * sinAng + dimX/2;
						yf = (xc - xt) * sinAng + (yt - yc) * cosAng + dimY/2;
						x0 = (int)floor(xf); x1 = x0 + 1;
						y0 = (int)floor(yf); y1 = y0 + 1;

						dA = refer.pwrStack[page + y0 * dimX + x0];
						dB = refer.pwrStack[page + y0 * dimX + x1];
						dC = refer.pwrStack[page + y1 * dimX + x0];
						dD = refer.pwrStack[page + y1 * dimX + x1];
						dE = (x1-xf) * dA + (xf-x0) * dB;
						dF = (x1-xf) * dC + (xf-x0) * dD;
						pwr_acc += (y1-yf) * dE + (yf-y0) * dF;
					}
				}
			}

			ptk_t[index] = (float)(ptk_acc / area);
			pwr_t[index] = (float)(pwr_acc / area);
		}
	}

	// normalize ptk[] and pwr[] to N(0, 1)
	ave = var = ave2 = var2 = 0.0;

	for ( index=0; index<parameter.tmpSize; index++ ) {

		if ( lcfMask.dataC[index] == EMS_TRUE ) {
			val = ptk_t[index];
			ave += val;
			var += val * val;
		}
		
		if ( scfMask.dataC[index] == EMS_TRUE ) {
			val = pwr_t[index];
			ave2 += val;
			var2 += val * val;
		}
	}

	if ( parameter.lcfMaskNum > 0 ) {
		ave /= parameter.lcfMaskNum;
		var /= parameter.lcfMaskNum;
		std = var - ave * ave;

		if ( std > 0.0 )
			std = sqrt(std);
		else
			std = EMS_ZERO;
	}

	if ( parameter.scfMaskNum > 0 ) {
		ave2 /= parameter.scfMaskNum;
		var2 /= parameter.scfMaskNum;
		std2 = var2 - ave2 * ave2;

		if ( std2 > 0.0 )
			std2 = sqrt(std2);
		else
			std2 = EMS_ZERO;
	}

	for ( index=0; index<parameter.tmpSize; index++ ) {
		
		if ( lcfMask.dataC[index] == EMS_TRUE )
			ptk_t[index] = (float)((ptk_t[index] - ave) / std);
		else
			ptk_t[index] = 0.0;
		
		if ( scfMask.dataC[index] == EMS_TRUE )
			pwr_t[index] = (float)((pwr_t[index] - ave2) / std2);
		else
			pwr_t[index] = 0.0;
	}
}


void EMS_Project::normalizeImage(void)
{
	micrograph.normalize();
	resizeMcrgraph();
}


void EMS_Project::bandFiltering(void)
{
	int x, y, xc, yc, dimen[2];
	long index, dSize;
	float *imgRel, *imgImg, ampt;
	double xf, yf, L_pass, H_pass, freq, expn;
	char fftName[NAMESIZE];
	FILE *fp;

	dimen[0] = micrograph.dimX;
	dimen[1] = micrograph.dimY;
	dSize = micrograph.mapSize;

	xc = dimen[0] / 2;
	yc = dimen[1] / 2;
	L_pass = 2.0 * parameter.pixelSize / parameter.passL;
	H_pass = 2.0 * parameter.pixelSize / parameter.passH;

	if ( (L_pass>1.415) || (H_pass<0.0) || (H_pass>L_pass) ) {
		errReport("Incorrect band-pass filtering parameters!");
		return;
	}

	// load density data or its FT from project space
	imgRel = new float[dSize];
	imgImg = new float[dSize];

	for ( index=0; index<dSize; index++ )
		imgRel[index] = imgImg[index] = 0.0;

	// try to load density FT data from project space
	sprintf(fftName, "%s/%s_fft.img", dirMcrgraph, filmName);
	fp = fopen(fftName, "rb");

	// if non-exist, load density itself and apply FT
	if ( fp == NULL ) {
		memcpy(imgRel, micrograph.density, sizeof(float)*dSize);
		fftwN(2, dimen, imgRel, imgImg, FFTW_FORWARD);

		// store FT data for future use
		fp = fopen(fftName, "wb");
		fwrite(imgRel, sizeof(float), dSize, fp);
		fwrite(imgImg, sizeof(float), dSize, fp);
		fclose(fp);
	}
	else {
		fread(imgRel, sizeof(float), dSize, fp);
		fread(imgImg, sizeof(float), dSize, fp);
		fclose(fp);
	}

	for ( y=0; y<dimen[1]; y++ ) {
		yf = 1.0 * ((y + yc) % dimen[1] - yc) / yc;

		for ( x=0; x<dimen[0]; x++ ) {
			xf = 1.0 * ((x + xc) % dimen[0] - xc) / xc;

			index = y * dimen[0] + x;
			freq = sqrt(xf*xf + yf*yf);

			if ( freq < H_pass ) {
				expn = (freq - H_pass) / EMS_BAND_HSFT;
				ampt = (float)exp(-1.0 * expn * expn);
				imgRel[index] *= ampt;
				imgImg[index] *= ampt;
			}
			else if ( freq > L_pass ) {
				expn = (freq - L_pass) / EMS_BAND_LSFT;
				ampt = (float)exp(-1.0 * expn * expn);
				imgRel[index] *= ampt;
				imgImg[index] *= ampt;
			}
		}
	}

	// redefine micrograph.density[]
	fftwN(2, dimen, imgRel, imgImg, FFTW_BACKWARD);
	memcpy(micrograph.density, imgRel, sizeof(float)*dSize);

	resizeMcrgraph();

	delete [] imgRel;
	delete [] imgImg;
}


void EMS_Project::restoreOriginal(void)
{
/*
	char fileName[NAMESIZE];
	FILE *fp;

	sprintf(fileName, "%s/%s.img", dirMcrgraph, filmName);
	fp = fopen(fileName, "rb");

	if ( fp == NULL ) {
		errReport("Cannot load image data from the project space!");
		return;
	}

	fread(micrograph.density, sizeof(float), micrograph.mapSize, fp);
	fclose(fp);

	resizeMcrgraph();
*/
}


void EMS_Project::imageFlattening(void)
{
	int x, y, x0, y0, x1, y1, xx, yy, xt, yt;
	int x2, y2, x3, y3, x4, y4, xy, x2y, xy2, x2y2, x3y, xy3;
	int dimX, dimY, dimM, tileX, tileY, tileSize, tileN;
	long index, mcount, ncount, tileArea, numThres;
	char *tileMask;
	double sum1, sum2, tileAve, tileStd, dval, xf, yf, *tileMean;
	double mA[EMS_FLAT_TERM][EMS_FLAT_TERM], vB[EMS_FLAT_TERM];

	tileN = 0;
	dimX = micrograph.dimX;
	dimY = micrograph.dimY;

	dimM = ( dimX >= dimY )? dimX : dimY;
	tileSize = (int)ceil(1.0 * dimM / EMS_FLAT_TILE);
	tileX = dimX / tileSize;
	tileY = dimY / tileSize;
	tileArea = tileSize * tileSize;
	numThres = (int)(tileArea * EMS_AREA_THRES);

	tileMask = new char[tileX * tileY];
	tileMean = new double[tileX * tileY];

	// identify background intensity within each tile
	syncFilmMask();

	for ( y=0; y<tileY; y++ ) {
		y0 = y * tileSize;
		y1 = y0 + tileSize;

		for ( x=0; x<tileX; x++ ) {
			x0 = x * tileSize;
			x1 = x0 + tileSize;

			// calculate AVE and STD
			sum1 = sum2 = 0.0;
			mcount = 0;

			for ( yy=y0; yy<y1; yy++ ) {
				yt = yy / parameter.downSize;
				for ( xx=x0; xx<x1; xx++ ) {
					xt = xx / parameter.downSize;

					// count unmasked pixels in the current tile
					if ( filmMask.dataC[yt*parameter.imgX+yt] == EMS_FALSE )
						mcount ++;

					dval = micrograph.density[yy*dimX+xx];
					sum1 += dval;
					sum2 += dval * dval;
				}
			}

			index = y * tileX + x;
			if ( mcount < numThres ) {
				tileMask[index] = EMS_TRUE;
				tileMean[index] = 0.0;
				continue;
			}
			else {
				tileN ++;
				tileMask[index] = EMS_FALSE;
			}

			tileAve = sum1 / tileArea;
			tileStd = sum2 / tileArea - tileAve * tileAve;

			if ( tileStd > 0.0 )
				tileStd = sqrt(tileStd);
			else
				tileStd = 0.0;

			// average over members within ONE STD, using unmasked pixels only
			sum1 = 0.0;
			ncount = 0;

			for ( yy=y0; yy<y1; yy++ ) {
				yt = yy / parameter.downSize;
				for ( xx=x0; xx<x1; xx++ ) {
					xt = xx / parameter.downSize;

					if ( filmMask.dataC[yt*parameter.imgX+yt] == EMS_FALSE ) {
						dval = micrograph.density[yy*dimX+xx];
						if ( fabs(dval - tileAve) <= tileStd ) {
							sum1 += dval;
							ncount ++;
						}
					}
				}
			}

			tileMean[y*tileX+x] = sum1 / ncount;
		}
	}

	// 2nd-order LSQ fitting of the background sampling
	if ( tileN >= EMS_FLAT_TERM ) {
		
		// reset matrix A and vector B
		for ( y=0; y<EMS_FLAT_TERM; y++ ) {
			vB[y] = 0.0;
			for ( x=0; x<EMS_FLAT_TERM; x++ )
				mA[x][y] = 0.0;
		}
		
		for ( y=0; y<tileY; y++ ) {
			y2 = y * y;
			y3 = y2 * y;
			y4 = y3 * y;
			
			for ( x=0; x<tileX; x++ ) {
				x2 = x * x;
				x3 = x2 * x;
				x4 = x3 * x;
				
				xy = x * y;
				x2y = x2 * y;
				xy2 = x * y2;
				x2y2 = x2 * y2;
				x3y = x3 * y;
				xy3 = x * y3;
				
				index = y * tileX + x;
				if ( tileMask[index] == EMS_TRUE ) continue;
				
				mA[0][0] += 1;
				mA[0][1] += x;
				mA[0][2] += y;
				mA[0][3] += xy;
				mA[0][4] += x2;
				mA[0][5] += y2;
				
				mA[1][0] += x;
				mA[1][1] += x2;
				mA[1][2] += xy;
				mA[1][3] += x2y;
				mA[1][4] += x3;
				mA[1][5] += xy2;
				
				mA[2][0] += y;
				mA[2][1] += xy;
				mA[2][2] += y2;
				mA[2][3] += xy2;
				mA[2][4] += x2y;
				mA[2][5] += y3;
				
				mA[3][0] += xy;
				mA[3][1] += x2y;
				mA[3][2] += xy2;
				mA[3][3] += x2y2;
				mA[3][4] += x3y;
				mA[3][5] += xy3;
				
				mA[4][0] += x2;
				mA[4][1] += x3;
				mA[4][2] += x2y;
				mA[4][3] += x3y;
				mA[4][4] += x4;
				mA[4][5] += x2y2;
				
				mA[5][0] += y2;
				mA[5][1] += xy2;
				mA[5][2] += y3;
				mA[5][3] += xy3;
				mA[5][4] += x2y2;
				mA[5][5] += y4;
				
				vB[0] += tileMean[index];
				vB[1] += x * tileMean[index];
				vB[2] += y * tileMean[index];
				vB[3] += xy * tileMean[index];
				vB[4] += x2 * tileMean[index];
				vB[5] += y2 * tileMean[index];
			}
		}
		
		linearEquas(mA, vB);
		
		// background removing
		for ( y=0; y<dimY; y++ ) {
			yf = 1.0 * y / tileSize - 0.5;
			for ( x=0; x<dimX; x++ ) {
				xf = 1.0 * x / tileSize - 0.5;
				index = y * dimX + x;
				dval = vB[0] + vB[1]*xf + vB[2]*yf + vB[3]*xf*yf + vB[4]*xf*xf + vB[5]*yf*yf;
				micrograph.density[index] -= (float)dval;
			}
		}
		
		resizeMcrgraph();
	}
	else
		errReport("Insufficient data points for background estimation!");

	delete [] tileMask;
	delete [] tileMean;
}


void EMS_Project::pixelBinning(void)
{
	micrograph.bin2D();
	parameter.pixelSize *= micrograph.binBox;
	resizeMcrgraph();
}


void EMS_Project::calcPowerSpec(void)
{
	int x, y, z, x_t, y_t;
	int dimX, dimY, dimen[2];
	long dsize, psize, index, index_src, index_dst;
	float *cr, *ci, *pwr_t, ww;
	EMS_MAP ptk_t;
	
	dimX = templates.dimX;
	dimY = templates.dimY;
	psize = dimX * dimY;

	dimen[0] = dimX * EMS_TMP_EXP;
	dimen[1] = dimY * EMS_TMP_EXP;
	dsize = dimen[0] * dimen[1];
	ww = (float)(sqrt(dsize));

	cr = new float[dsize];	// real part of []x[]
	ci = new float[dsize];	// imag part of []x[]
	pwr_t = new float[dsize];

	// process one template image at a time
	for ( z=0; z<parameter.tmpN; z++ ) {
		ptk_t.setup(refer.ptkStack+z*psize, dimX, dimY, dimen[0], dimen[1]);

		for ( index=0; index<dsize; index++ ) {
			cr[index] = ptk_t.rel[index] * ptk_t.rel[index] + ptk_t.img[index] * ptk_t.img[index];
			cr[index] *= ww;
			ci[index] = 0.0;
		}
		
		fftwN(2, dimen, cr, ci, FFTW_BACKWARD);
		
		// quadrant swapping
		for ( y=0; y<dimen[1]; y++ ) {
			y_t = (y + dimen[1]/2) % dimen[1];
			for ( x=0; x<dimen[0]; x++ ) {
				x_t = (x + dimen[0]/2) % dimen[0];
				pwr_t[y_t*dimen[0]+x_t] = cr[y*dimen[0]+x];
			}
		}

		for ( y=0; y<dimY; y++ ) {
			index_dst = z * psize + y * dimX;
			index_src = ((dimen[1] - dimY)/2 + y) * dimen[0] + (dimen[0] - dimX)/2;
			memcpy(refer.pwrStack+index_dst, pwr_t+index_src, dimX * sizeof(float));
		}
	}

	delete [] cr;
	delete [] ci;
	delete [] pwr_t;
}


void EMS_Project::calcVariance(void)
{
	if ( micrograph.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}

	if ( lcfMask.exist == EMS_FALSE ) {
		errReport("Template dimension is unknown yet!");
		return;
	}

	localSTD(lcfMask, film, stdMap, (float)(1.0/parameter.lcfMaskNum));
}


void EMS_Project::profileMasking(char maskLevel)
{
	int x, y, z, dim;
	long index;
	char *msk;
	double xc, yc, r2, lcf_rd, lcf_r2;

	if ( templates.exist == EMS_FALSE ) {
		errReport("Template dimension is unknown yet!");
		return;
	}

	// masking at the original scale
	xc = (double)(templates.dimX / 2);
	yc = (double)(templates.dimY / 2);
	dim = (templates.dimX <= templates.dimY)? templates.dimX : templates.dimY;
	
	lcf_rd = 0.5 * dim * parameter.lcfMaskRad;
	lcf_r2 = lcf_rd * lcf_rd;
	
	// generate L1/L2 mask frame by frame
	if ( maskLevel == EMS_MASK_L1 )
		msk = refer.L1Mask;
	else
		msk = refer.L2Mask;

	for ( z=0; z<templates.dimZ; z++ ) {
		index = z * templates.dimX * templates.dimY;
		
		for ( y=0; y<templates.dimY; y++ ) {
			for ( x=0; x<templates.dimX; x++ ) {
				r2 = (x-refer.xc[z])*(x-refer.xc[z]) + (y-refer.yc[z])*(y-refer.yc[z]);
				
				if ( r2 <= lcf_r2 )
					msk[index] = (char)EMS_TRUE;
				else
					msk[index] = (char)EMS_FALSE;
				
				index ++;
			}
		}
	}
}


void EMS_Project::runProject(char *projFile)
{
	if ( loadProject(projFile) == EMS_TRUE )
		fullScreen();
}


void EMS_Project::testScreen(void)
{
	int x, y, z, dim;
	long index, maskNum;
	double xc, yc, r2, lcf_rd, lcf_r2;
	double scf_rd, scf_r2, scf_rcd, scf_rc2;
	float *ptk_t, *pwr_t, *ptk_s, *pwr_s, rotate, scaling;

	// safe-guard data availability
	if ( emsRunMode != EMS_GUI_MODE ) {
		errReport("Screening test functions only under the GUI mode!");
		return;
	}
	if ( micrograph.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}
	if ( templates.exist == EMS_FALSE ) {
		errReport("Templates have not been loaded yet!");
		return;
	}

	// safe-guard downsize factor
	if ( parameter.downSize < 1 ) {
		errReport("Inappropriate downsize factor!");
		parameter.downSize = EMS_DOWNSIZE;
		EMS_DP->dspDownSize->value(EMS_DOWNSIZE);
		return;
	}

	// search each template at various orientations for a new project
	if ( sysStatus.mapRestore == EMS_FALSE ) {

		if ( sysStatus.newScale == EMS_TRUE ) {
			resizeMcrgraph();
			sysStatus.newScale = EMS_FALSE;
		}

		// setup masking radii
		x = templates.dimX / parameter.downSize;
		y = templates.dimY / parameter.downSize;
		dim = ( x <= y )? x : y;

		lcf_rd = 0.5 * dim * parameter.lcfMaskRad;
		scf_rd = 0.5 * dim * parameter.scfMaskRad;
		scf_rcd = 0.5 * dim * parameter.scfMaskRadc;

		// redefine template size for rotational average
		parameter.tmpX = parameter.tmpY = 2 * (int)(sqrt(x * x + y * y));
		parameter.tmpSize = parameter.tmpX * parameter.tmpY;

		ptk_t = new float[parameter.tmpSize];
		pwr_t = new float[parameter.tmpSize];
		ptk_s = new float[parameter.tmpSize];
		pwr_s = new float[parameter.tmpSize];
		
		for ( index=0; index<parameter.tmpSize; index++ ) {
			ptk_s[index] = 0.0;
			pwr_s[index] = 0.0;
		}
		
		// setup LCF (level-I) and SCF masks
		xc = parameter.tmpX / 2.0;
		yc = parameter.tmpY / 2.0;
		lcf_r2 = lcf_rd * lcf_rd;
		scf_r2 = scf_rd * scf_rd;
		scf_rc2 = scf_rcd * scf_rcd;
		
		lcfMask.initC(EMS_FALSE, parameter.tmpX, parameter.tmpY);
		scfMask.initC(EMS_FALSE, parameter.tmpX, parameter.tmpY);
		parameter.lcfMaskNum = parameter.scfMaskNum = 0;
		
		for ( y=0; y<parameter.tmpY; y++ ) {
			for ( x=0; x<parameter.tmpX; x++ ) {
				
				index = y * parameter.tmpX + x;
				r2 = (x - xc) * (x - xc) + (y - yc) * (y - yc);
				
				if ( r2 <= lcf_r2 ) {
					parameter.lcfMaskNum ++;
					lcfMask.dataC[index] = (char)EMS_TRUE;
				}
				
				if ( (r2 >= scf_rc2) && (r2 <= scf_r2) ) {
					parameter.scfMaskNum ++;
					scfMask.dataC[index] = (char)EMS_TRUE;
				}
			}
		}

		// compute rotational average of all the reference images
		scaling = (float)(EMS_ROTA_AVERG / (parameter.tmpN * 360.0));

		for ( z=0; z<parameter.tmpN; z++ ) {
			for ( rotate=0.0; rotate<(360.0-EMS_ZERO); rotate+=EMS_ROTA_AVERG ) {
				
				defineTemplate(z, ptk_t, pwr_t, rotate);

				for ( index=0; index<parameter.tmpSize; index++ ) {
					ptk_s[index] += ptk_t[index] * scaling;
					pwr_s[index] += pwr_t[index] * scaling;
				}
			}
		}

		ptk.setup(ptk_s, parameter.tmpX, parameter.tmpY, parameter.imgX, parameter.imgY);
		pwr.setup(pwr_s, parameter.tmpX, parameter.tmpY, parameter.imgX, parameter.imgY);
		
		// compute LCF / SCF maps
		calcVariance();
		calcLCF();
		calcSCF();

		// avoid multiplying two negatives
		for ( index=0; index<parameter.filmSize; index++ ) {
			if ( lcfMap.dataF[index] < 0.0 )
				lcfMap.dataF[index] = 0.0;
			if ( scfMap.dataF[index] < 0.0 )
				scfMap.dataF[index] = 0.0;
		}

		// compute DCF map
		calcDCF();
		
		// accumulate selection from various templates
		for ( index=0; index<parameter.filmSize; index++ ) {
			record[index].lcfScore = lcfMap.dataF[index];
			record[index].scfScore = scfMap.dataF[index];
			record[index].dcfScore = dcfMap.dataF[index];
			record[index].euler = 0.0;
			record[index].tmpid = -1;
		}
		
		sysStatus.mapRestore = EMS_TRUE;

		delete [] ptk_t;
		delete [] pwr_t;
		delete [] ptk_s;
		delete [] pwr_s;
	}

	// generate LCF / SCF / DCF statistics
	maskNum = 0;
	parameter.lcfAve = 0.0;
	parameter.scfAve = 0.0;
	parameter.dcfAve = 0.0;
	parameter.lcfMax = -1.0E9;
	parameter.scfMax = -1.0E9;
	parameter.dcfMax = -1.0E9;

	syncFilmMask();

	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( filmMask.dataC[index] == EMS_FALSE ) {
			parameter.lcfAve += record[index].lcfScore;
			parameter.scfAve += record[index].scfScore;
			parameter.dcfAve += record[index].dcfScore;

			if ( parameter.lcfMax < record[index].lcfScore ) parameter.lcfMax = record[index].lcfScore;
			if ( parameter.scfMax < record[index].scfScore ) parameter.scfMax = record[index].scfScore;
			if ( parameter.dcfMax < record[index].dcfScore ) parameter.dcfMax = record[index].dcfScore;

			maskNum ++;
		}
	}

	if ( maskNum > 0 ) {
		parameter.lcfAve /= maskNum;
		parameter.scfAve /= maskNum;
		parameter.dcfAve /= maskNum;
	}

	// display map statistics
	EMS_DP->maxLCF->value(parameter.lcfMax);
	EMS_DP->aveLCF->value(parameter.lcfAve);
	EMS_DP->maxSCF->value(parameter.scfMax);
	EMS_DP->aveSCF->value(parameter.scfAve);
	EMS_DP->sldScore->maximum(parameter.dcfMax);
	EMS_DP->sldScore->minimum(parameter.dcfAve);
	
	// screen SIGNATURE maps
	screenLCF(parameter.lcfThres);
	screenSCF(parameter.scfThres);
	screenDCF(-1);
	screenDST(parameter.distance);
	
	msgReport("Screening test is complete");
}


void EMS_Project::fullScreen(void)
{
	int x, y, z;
	long index, maskNum;
	float *ptk_t, *pwr_t, rotate;
	EMS_MAP lcf_L1;

	// safe-guard data availability
	if ( micrograph.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}
	if ( templates.exist == EMS_FALSE ) {
		errReport("Templates have not been loaded yet!");
		return;
	}

	// safe-guard rotation parameters
	if ( (parameter.rotaRange<0) || (parameter.rotaRange>360.0) ) {
		errReport("Rotation range should be in the range of [0 360]!");
		parameter.rotaRange = EMS_ROTA_RANGE;
		EMS_DP->dspRotaRange->value(EMS_ROTA_RANGE);
		return;
	}
	if ( (parameter.rotaDelta<0) || (parameter.rotaDelta>360.0) ) {
		errReport("Angular increment should be in the range of [0 360]!");
		parameter.rotaDelta = EMS_ROTA_DELTA;
		EMS_DP->dspRotaDelta->value(EMS_ROTA_DELTA);
		return;
	}

	// safe-guard downsize factor
	if ( parameter.downSize < 1 ) {
		errReport("Inappropriate downsize factor!");
		parameter.downSize = EMS_DOWNSIZE;
		EMS_DP->dspDownSize->value(EMS_DOWNSIZE);
		return;
	}

	// restore project parameter and maps if possible
	if ( emsRunMode == EMS_COM_MODE ) {
		sysStatus.newScale = EMS_TRUE;
		sysStatus.fullScreen = EMS_TRUE;
		sysStatus.testScreen = EMS_FALSE;
		sysStatus.mapRestore = EMS_FALSE;
	}
	else {
		for ( index=0; index<parameter.filmSize; index++ ) {
			if ( record[index].select == EMS_SEL_COM ) {
				record[index].select = EMS_SEL_NON;
			}
		}
	}

	// search each template at various orientations for a new project
	if ( sysStatus.mapRestore == EMS_FALSE ) {

		if ( sysStatus.newScale == EMS_TRUE ) {
			resizeMcrgraph();
			sysStatus.newScale = EMS_FALSE;
		}
		else {
			for ( index=0; index<parameter.filmSize; index++ ) {
				record[index].lcfScore = -1.0;
				record[index].scfScore = -1.0;
				record[index].dcfScore = -1.0;
				record[index].euler = 0.0;
				record[index].tmpid = -1;
			}
		}

		// redefine template size for rotational resampling
		x = templates.dimX / parameter.downSize;
		y = templates.dimY / parameter.downSize;
		parameter.tmpX = parameter.tmpY = 2 * (int)(sqrt(x * x + y * y));
		parameter.tmpSize = parameter.tmpX * parameter.tmpY;

		ptk_t = new float[parameter.tmpSize];
		pwr_t = new float[parameter.tmpSize];
		
		for ( z=0; z<parameter.tmpN; z++ ) {
			for ( rotate=0.0; rotate<parameter.rotaRange; rotate+=parameter.rotaDelta ) {
				
				sprintf(message, "Searching for template %d at rotation %-6.2f", z+1, rotate);
				msgReport(message);

				// setup level-I mask (LCF), PTK and PWR 2D array
				defineTempMask(z, refer.L1Mask, rotate);
				defineTemplate(z, ptk_t, pwr_t, rotate);
				ptk.setup(ptk_t, parameter.tmpX, parameter.tmpY, parameter.imgX, parameter.imgY);
				pwr.setup(pwr_t, parameter.tmpX, parameter.tmpY, parameter.imgX, parameter.imgY);

				// level-I screening: compute LCF / SCF maps
				calcVariance();
				calcLCF();
				calcSCF();
				
				// avoid multiplying two negatives
				for ( index=0; index<parameter.filmSize; index++ ) {
					if ( lcfMap.dataF[index] < 0.0 )
						lcfMap.dataF[index] = 0.0;
					if ( scfMap.dataF[index] < 0.0 )
						scfMap.dataF[index] = 0.0;
				}
				
				// level-II screening if activated
				if ( project->sysStatus.activeL2 == EMS_TRUE ) {

					// backup LCF score from level-I screening
					copyMap(lcf_L1, lcfMap);

					// setup level-II mask and screen
					defineTempMask(z, refer.L2Mask, rotate);
					calcVariance();
					calcLCF();

					// L2-weighting LCF, SCF and DCF scores
					multiplyMap(lcfMap, lcf_L1, lcfMap);
				}
				
				// compute DCF maps (final score)
				calcDCF();

				// accumulate selection from various templates
				for ( index=0; index<parameter.filmSize; index++ ) {
					
					if ( lcfMap.dataF[index] > record[index].lcfScore )
						record[index].lcfScore = lcfMap.dataF[index];
					
					if ( scfMap.dataF[index] > record[index].scfScore )
						record[index].scfScore = scfMap.dataF[index];
					
					if ( dcfMap.dataF[index] > record[index].dcfScore ) {
						record[index].dcfScore = dcfMap.dataF[index];
						record[index].tmpid = z;
						record[index].euler = rotate;
					}
				}

				if ( sysStatus.rotaScreen == EMS_FALSE ) break;
			}
		}

		delete [] ptk_t;
		delete [] pwr_t;
	}

	// generate LCF / SCF / DCF statistics
	if ( emsRunMode == EMS_GUI_MODE ) {

		maskNum = 0;
		parameter.lcfAve = 0.0;
		parameter.scfAve = 0.0;
		parameter.dcfAve = 0.0;
		parameter.lcfMax = -1.0E9;
		parameter.scfMax = -1.0E9;
		parameter.dcfMax = -1.0E9;
		
		syncFilmMask();

		for ( index=0; index<parameter.filmSize; index++ ) {
			if ( filmMask.dataC[index] == EMS_FALSE ) {
				parameter.lcfAve += record[index].lcfScore;
				parameter.scfAve += record[index].scfScore;
				parameter.dcfAve += record[index].dcfScore;
				
				if ( parameter.lcfMax < record[index].lcfScore ) parameter.lcfMax = record[index].lcfScore;
				if ( parameter.scfMax < record[index].scfScore ) parameter.scfMax = record[index].scfScore;
				if ( parameter.dcfMax < record[index].dcfScore ) parameter.dcfMax = record[index].dcfScore;
				
				maskNum ++;
			}
		}

		if ( maskNum > 0 ) {
			parameter.lcfAve /= maskNum;
			parameter.scfAve /= maskNum;
			parameter.dcfAve /= maskNum;
		}

		// display map statistics
		EMS_DP->maxLCF->value(parameter.lcfMax);
		EMS_DP->aveLCF->value(parameter.lcfAve);
		EMS_DP->maxSCF->value(parameter.scfMax);
		EMS_DP->aveSCF->value(parameter.scfAve);
		EMS_DP->sldScore->maximum(parameter.dcfMax);
		EMS_DP->sldScore->minimum(parameter.dcfAve);
		
		// screen correlation score maps
		screenLCF(parameter.lcfThres);
		screenSCF(parameter.scfThres);
		screenDCF(-1);
		screenDST(parameter.distance);
	}

	msgReport("Particle screening completes");

	// store project parameter file and maps if necessary
	if ( sysStatus.mapRestore == EMS_FALSE )
		sysStatus.mapRestore = recordScreenMap();
}


void EMS_Project::screenDST(float threshold)
{
	int x, y, x0, x1, y0, y1, xx, yy, dist, dist2;
	long index, index_t;
	float pixSize;

	if ( record == NULL ) return;
	if ( dcfPeak.exist == EMS_FALSE ) return;

	// load peaks from DCF screening
	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( record[index].select == EMS_SEL_COM )
			record[index].select = EMS_SEL_NON;

		if ( dcfPeak.dataC[index] == EMS_TRUE ) {
			if ( record[index].select == EMS_SEL_NON )
				record[index].select = EMS_SEL_COM;
		}
	}

	// searching for local maximum
	pixSize = parameter.pixelSize * parameter.downSize;
	dist = (int)ceil(EMS_DCF_RADIUS * parameter.prtkSize / pixSize);
	dist2 = dist * dist;

	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {

			index = y * parameter.imgX + x;
			if ( record[index].select == EMS_SEL_NON ) continue;

			x0 = x - dist;
			if ( x0 < 0 ) x0 = 0;

			x1 = x + dist;
			if ( x1 >= parameter.imgX ) x1 = parameter.imgX - 1;

			y0 = y - dist;
			if ( y0 < 0 ) y0 = 0;

			y1 = y + dist;
			if ( y1 >= parameter.imgY ) y1 = parameter.imgY - 1;

			for ( yy=y0; yy<=y1; yy++ ) {
				for ( xx=x0; xx<=x1; xx++ ) {

					index_t = yy * parameter.imgX + xx;
					if ( index_t == index ) continue;
					if ( record[index_t].select == EMS_SEL_NON ) continue;
					if ( (yy-y)*(yy-y)+(xx-x)*(xx-x) > dist2 ) continue;

					if ( record[index_t].dcfScore > record[index].dcfScore ) {
						record[index].select = EMS_SEL_NON;

						// bail out of the double-loop
						xx = x1 + 1;
						yy = y1 + 1;
						break;
					}
				}
			}

			if ( record[index].select > EMS_SEL_NON ) {
				for ( yy=y0; yy<=y1; yy++ ) {
					for ( xx=x0; xx<=x1; xx++ ) {
						index_t = yy * parameter.imgX + xx;

						if ( index_t == index )
							continue;
						else if ( (yy-y)*(yy-y)+(xx-x)*(xx-x) > dist2 )
							continue;
						else
							record[index_t].select = EMS_SEL_NON;
					}
				}
			}
		}
	}

	// excluding particles too close to each other
	dist = (int)ceil((threshold + parameter.prtkSize) / pixSize);
	dist2 = dist * dist;

	for ( y=0; y<parameter.imgY; y++ ) {
		for ( x=0; x<parameter.imgX; x++ ) {

			index = y * parameter.imgX + x;
			if ( record[index].select == EMS_SEL_NON ) continue;

			x0 = x - dist;
			if ( x0 < 0 ) x0 = 0;

			x1 = x + dist;
			if ( x1 >= parameter.imgX ) x1 = parameter.imgX - 1;

			y0 = y - dist;
			if ( y0 < 0 ) y0 = 0;

			y1 = y + dist;
			if ( y1 >= parameter.imgY ) y1 = parameter.imgY - 1;

			for ( yy=y0; yy<=y1; yy++ ) {
				for ( xx=x0; xx<=x1; xx++ ) {
					
					index_t = yy * parameter.imgX + xx;
					if ( index_t == index ) continue;
					if ( record[index_t].select == EMS_SEL_NON ) continue;
					if ( (yy-y)*(yy-y)+(xx-x)*(xx-x) > dist2 ) continue;
					
					record[index].select = record[index_t].select = EMS_SEL_NON;
				}
			}
		}
	}

	// count total particle selection
	parameter.totCount = 0;

	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( record[index].select > EMS_SEL_NON ) {
			parameter.totCount ++;
		}
	}

	EMS_DP->dspCount->value(parameter.totCount);
}


void EMS_Project::calcLCF(void)
{
	long index;

	if ( parameter.lcfMaskNum == 0 ) {
		errReport("LCF mask has zero count!");
		return;
	}

	localCOR(ptk, film, stdMap, lcfMap, (float)(1.0/parameter.lcfMaskNum));

	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( lcfMap.dataF[index] > 1.0 )
			lcfMap.dataF[index] = 1.0;
		else if ( lcfMap.dataF[index] < -1.0 )
			lcfMap.dataF[index] = -1.0;
	}
}


void EMS_Project::screenLCF(float threshold)
{
	long index;

	if ( record == NULL ) return;

	syncFilmMask();

	parameter.lcfCount = 0;
	lcfPeak.initC(EMS_FALSE, parameter.imgX, parameter.imgY);

	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( (filmMask.dataC[index]==EMS_FALSE) && (record[index].lcfScore>=threshold) ) {
			lcfPeak.dataC[index] = EMS_TRUE;
			parameter.lcfCount ++;
		}
	}

	EMS_DP->cntLCF->value(parameter.lcfCount);
}


void EMS_Project::calcSCF(void)
{
	long index;

	if ( parameter.scfMaskNum == 0 ) {
		errReport("SCF mask has zero count!");
		return;
	}

	lcfMap.imageFFT(parameter.imgX, parameter.imgY);
	localSTD(scfMask, lcfMap, std2Map, (float)(1.0/parameter.scfMaskNum));
	localCOR(pwr, lcfMap, std2Map, scfMap, (float)(1.0/parameter.scfMaskNum));

	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( scfMap.dataF[index] > 1.0 )
			scfMap.dataF[index] = 1.0;
		else if ( scfMap.dataF[index] < -1.0 )
			scfMap.dataF[index] = -1.0;
	}
}


void EMS_Project::screenSCF(float threshold)
{
	long index;

	if ( record == NULL ) return;
	if ( lcfPeak.exist == EMS_FALSE ) return;

	parameter.scfCount = 0;
	scfPeak.initC(EMS_FALSE, parameter.imgX, parameter.imgY);

	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( (lcfPeak.dataC[index]==EMS_TRUE) && (record[index].scfScore>=threshold) ) {
			scfPeak.dataC[index] = EMS_TRUE;
			parameter.scfCount ++;
		}
	}

	EMS_DP->cntSCF->value(parameter.scfCount);
}


void EMS_Project::calcDCF(void)
{
	multiplyMap(lcfMap, scfMap, dcfMap);
}


void EMS_Project::screenDCF(float threshold)
{
	long index;

	if ( record == NULL ) return;
	if ( scfPeak.exist == EMS_FALSE ) return;

	// define DCF as a combination of LCF and SCF
	if ( threshold < 0 )
		threshold = parameter.lcfThres * parameter.scfThres;

	parameter.dcfCount = 0;
	dcfPeak.initC(EMS_FALSE, parameter.imgX, parameter.imgY);

	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( (scfPeak.dataC[index]==EMS_TRUE) && (record[index].dcfScore>=threshold) ) {
			dcfPeak.dataC[index] = EMS_TRUE;
			parameter.dcfCount ++;
		}
	}
}


float EMS_Project::prtkSelect(int x, int y)
{
	long index;

	if ( (x<0) || (x>=parameter.imgX) ) return(-1.0);
	if ( (y<0) || (y>=parameter.imgY) ) return(-1.0);

	index = y * parameter.imgX + x;

	if ( record[index].select == EMS_SEL_NON ) {
		record[index].select = EMS_SEL_MAN;
		parameter.totCount ++;
		projSaveStatus = EMS_FALSE;
	}

	return(record[index].dcfScore);
}


float EMS_Project::prtkCoopSelect(int &x, int &y)
{
	int xt, yt, x0, x1, y0, y1, dist;
	float s_factor, pixSize;

	if ( (x<0) || (x>=parameter.imgX) ) return(-1.0);
	if ( (y<0) || (y>=parameter.imgY) ) return(-1.0);
	
	pixSize = parameter.pixelSize * parameter.downSize;
	s_factor = record[y * parameter.imgX + x].dcfScore;
	
	if ( s_factor > 0.0 ) {

		dist = (int)ceil(EMS_PRTK_HALF * parameter.prtkSize / pixSize);

		x0 = x - dist;
		if ( x0 < 0 ) x0 = 0;

		x1 = x + dist;
		if ( x1 >= parameter.imgX ) x1 = parameter.imgX - 1;

		y0 = y - dist;
		if ( y0 < 0 ) y0 = 0;

		y1 = y + dist;
		if ( y1 >= parameter.imgY ) y1 = parameter.imgY - 1;
		
		for ( yt=y0; yt<=y1; yt++ ) {
			for ( xt=x0; xt<=x1; xt++ ) {
				if ( record[yt*parameter.imgX+xt].dcfScore > s_factor ) {
					s_factor = record[yt*parameter.imgX+xt].dcfScore;
					x = xt;
					y = yt;
				}
			}
		}
	}

	else {
		msgReport("CO-OP function is off: S-factor is unknown or too low");
	}

	project->prtkSelect(x, y);
	return(s_factor);
}


float EMS_Project::prtkDelete(int x, int y)
{
	int x0, x1, y0, y1, xx, yy;
	long index;
	float s_factor = -1.0;

	if ( (x<0) || (x>=parameter.imgX) ) return(-1.0);
	if ( (y<0) || (y>=parameter.imgY) ) return(-1.0);

	x0 = x - EMS_DST_TOLERANCE;
	x1 = x + EMS_DST_TOLERANCE;
	y0 = y - EMS_DST_TOLERANCE;
	y1 = y + EMS_DST_TOLERANCE;

	if ( x0 < 0 ) x0 = 0;
	if ( x1 >= parameter.imgX ) x1 = parameter.imgX - 1;
	if ( y0 < 0 ) y0 = 0;
	if ( y1 >= parameter.imgY ) y1 = parameter.imgY - 1;

	for ( yy=y0; yy<=y1; yy++ ) {
		for ( xx=x0; xx<=x1; xx++ ) {
			index = yy * parameter.imgX + xx;

			if ( record[index].select > EMS_SEL_NON ) {
				record[index].select = EMS_SEL_NON;
				parameter.totCount --;
				projSaveStatus = EMS_FALSE;

				if ( record[index].dcfScore > s_factor )
					s_factor = record[index].dcfScore;
			}
		}
	}

	return(s_factor);
}


void EMS_Project::inspectAccept(void)
{
	int x, y;

	if ( inspectID >= 0 )
		record[inspectID].select = EMS_SEL_OUI;
	else
		inspectID = 0;

	// backup for UNDO
	inspectUD = inspectID;

	for ( inspectID; inspectID<parameter.filmSize; inspectID++ ) {
		if ( (record[inspectID].select==EMS_SEL_COM) || (record[inspectID].select==EMS_SEL_MAN) ) {
			x = inspectID % parameter.imgX;
			y = inspectID / parameter.imgX;
			EMS_DP->imgDisplay->setInspectFocus(x, y, 0);
			EMS_DP->sldScore->value(record[inspectID].dcfScore);

			if ( sysStatus.showSpec == EMS_TRUE )
				cPanel->monitorSpec();

			if ( sysStatus.showRefr == EMS_TRUE )
				cPanel->monitorTemp(record[inspectID].tmpid);

			break;
		}
	}

	if ( inspectID >= parameter.filmSize ) {
		EMS_DP->imgDisplay->setInspectFocus(-1, -1, 0);
		msgReport("All selections have been inspected");
		inspectID = -1;
	}
}


void EMS_Project::inspectReject(void)
{
	int x, y;

	if ( inspectID >= 0 ) {
		record[inspectID].select = EMS_SEL_NON;
		parameter.totCount --;
		EMS_DP->dspCount->value(parameter.totCount);
	}
	else
		inspectID = 0;

	// backup for UNDO
	inspectUD = inspectID;

	for ( inspectID; inspectID<parameter.filmSize; inspectID++ ) {
		if ( (record[inspectID].select==EMS_SEL_COM) || (record[inspectID].select==EMS_SEL_MAN) ) {
			x = inspectID % parameter.imgX;
			y = inspectID / parameter.imgX;
			EMS_DP->imgDisplay->setInspectFocus(x, y, 0);
			EMS_DP->sldScore->value(record[inspectID].dcfScore);

			if ( sysStatus.showSpec == EMS_TRUE )
				cPanel->monitorSpec();

			if ( sysStatus.showRefr == EMS_TRUE )
				cPanel->monitorTemp(record[inspectID].tmpid);

			break;
		}
	}

	if ( inspectID >= parameter.filmSize ) {
		EMS_DP->imgDisplay->setInspectFocus(-1, -1, 0);
		msgReport("All selections have been inspected");
		inspectID = -1;
	}
}


void EMS_Project::inspectUndo(void)
{
	int x, y;
	
	if ( (inspectUD<=0) || (inspectUD>=parameter.filmSize) ) return;
	
	// retract one step ONLY
	inspectID = inspectUD;
	record[inspectID].select = EMS_SEL_COM;
	
	x = inspectID % parameter.imgX;
	y = inspectID / parameter.imgX;
	EMS_DP->imgDisplay->setInspectFocus(x, y, 0);
	EMS_DP->sldScore->value(record[inspectID].dcfScore);
	
	if ( sysStatus.showSpec == EMS_TRUE )
		cPanel->monitorSpec();
	
	if ( sysStatus.showRefr == EMS_TRUE )
		cPanel->monitorTemp(record[inspectID].tmpid);
}


char EMS_Project::recordScreenMap(void)
{
	char record_fn[NAMESIZE], projct_fn[NAMESIZE], mask_fn[NAMESIZE];
	FILE *project_fp, *record_fp, *mask_fp;

	sprintf(projct_fn, "%s/%s.prj", dirATScreen, filmName);
	sprintf(record_fn, "%s/%s.map", dirATScreen, filmName);
	sprintf(mask_fn, "%s/%s.msk", dirMcrgraph, filmName);

	// write ".prj" file
	if ( (project_fp = fopen(projct_fn, "wb")) != NULL ) {
		fprintf(project_fp, "%s : %s\n", projName, filmName);
		fprintf(project_fp, "%s\n", dirProject);
		fprintf(project_fp, "%s\n", emfilmFN);
		fprintf(project_fp, "%s\n", templtFN);
		fwrite(&sysStatus, sizeof(struct EMS_SYSSTATUS), 1, project_fp);
		fwrite(&parameter, sizeof(struct EMS_PARAMETER), 1, project_fp);
		fclose(project_fp);
	}
	else {
		errReport("Cannot record project parameters!");
		return(EMS_FALSE);
	}

	// write ".map" file
	if ( (record_fp = fopen(record_fn, "wb")) != NULL ) {
		fwrite(record, parameter.filmSize, sizeof(struct EMS_RECORD), record_fp);
		fclose(record_fp);
	}
	else {
		errReport("Cannot record screening map!");
		return(EMS_FALSE);
	}

	// write ".msk" file
	if ( emsRunMode == EMS_GUI_MODE ) {
		if ( (mask_fp = fopen(mask_fn, "wb")) != NULL ) {
			fwrite(dispMask.dataC, dispMask.mapSize, sizeof(char), mask_fp);
			fclose(mask_fp);
		}
		else {
			errReport("Cannot record film mask!");
			return(EMS_FALSE);
		}
	}

	return(EMS_TRUE);
}


char EMS_Project::restoreScreenMap(void)
{
	char record_fn[NAMESIZE], mask_fn[NAMESIZE];
	FILE *record_fp, *mask_fp;

	// load ".map" file
	sprintf(record_fn, "%s/%s.map", dirATScreen, filmName);

	if ( (record_fp = fopen(record_fn, "rb")) != NULL ) {
		fread(record, parameter.filmSize, sizeof(struct EMS_RECORD), record_fp);
		fclose(record_fp);
	}
	else {
		msgReport("Need to re-compute the screening map");
		return(EMS_FALSE);
	}

	// restore film mask
	if ( emsRunMode == EMS_GUI_MODE ) {
		sprintf(mask_fn, "%s/%s.msk", dirMcrgraph, filmName);

		if ( (mask_fp = fopen(mask_fn, "rb")) != NULL ) {
			fread(dispMask.dataC, dispMask.mapSize, sizeof(char), mask_fp);
			fclose(record_fp);

			EMS_DP->imgDisplay->setMaskTexture(dispMask);
			sysStatus.syncMask = EMS_FALSE;
			syncFilmMask();
		}
		else {
			errReport("Cannot restore film mask!");
			return(EMS_FALSE);
		}
	}

	return(EMS_TRUE);
}


void EMS_Project::writeMap2D(char *file_name, EMS_MAP &signMap)
{
	float pixSize;
	MRC_MAP mrcMap;

	pixSize = parameter.pixelSize * parameter.downSize;
	mrcMap.setDimensn(signMap.dimX, signMap.dimY, 1, pixSize, pixSize, 1.0f);
	mrcMap.setDensity(signMap.dataF);
	mrcMap.output(file_name);
}


void EMS_Project::writeMap3D(char *file_name, int x, int y, int z, float *map_data)
{
	MRC_MAP mrcMap;

	mrcMap.setDimensn(x, y, z, parameter.pixelSize, parameter.pixelSize, parameter.pixelSize);
	mrcMap.setDensity(map_data);
	mrcMap.output(file_name);
}


void EMS_Project::clearAllRecord(void)
{
	long index;

	if ( record == NULL )
		return;
	else
		inspectID = -1;

	for ( index=0; index<parameter.filmSize; index++ ) {
		record[index].select = EMS_SEL_NON;
		record[index].lcfScore = -1.0;
		record[index].scfScore = -1.0;
		record[index].dcfScore = -1.0;
		record[index].euler = 0.0;
		record[index].tmpid = -1;
	}
}


void EMS_Project::clearComRecord(void)
{
	long index;

	if ( record == NULL ) return;

	for ( index=0; index<parameter.filmSize; index++ ) {
		if ( record[index].select == EMS_SEL_COM ) {
			record[index].select = EMS_SEL_NON;
			record[index].lcfScore = -1.0;
			record[index].scfScore = -1.0;
			record[index].dcfScore = -1.0;
			record[index].euler = 0.0;
			record[index].tmpid = -1;
		}
	}
}


void EMS_Project::delRecord(void)
{
	if ( record != NULL ) {
		delete [] record;
		record = NULL;
	}
}


void EMS_Project::delTempStack(void)
{
	if ( refer.ptkStack != NULL ) {
		delete [] refer.ptkStack;
		refer.ptkStack = NULL;
	}

	if ( refer.pwrStack != NULL ) {
		delete [] refer.pwrStack;
		refer.pwrStack = NULL;
	}

	if ( refer.L1Mask != NULL ) {
		delete [] refer.L1Mask;
		refer.L1Mask = NULL;
	}

	if ( refer.L2Mask != NULL ) {
		delete [] refer.L2Mask;
		refer.L2Mask = NULL;
	}

	if ( refer.select != NULL ) {
		delete [] refer.select;
		refer.select = NULL;
	}

	if ( refer.xc != NULL ) {
		delete [] refer.xc;
		refer.xc = NULL;
	}

	if ( refer.yc != NULL ) {
		delete [] refer.yc;
		refer.yc = NULL;
	}
}


void EMS_Project::resetParam(void)
{
	// PROJECT NAME
	sysStatus.emsProject = EMS_FALSE;
	strcpy(projName, EMS_PRJ_FN);
	EMS_DP->dspProj->value(EMS_PRJ_FN);
	EMS_DP->dspFilm->value("\0");

	strcpy(dirProject, EMS_PRJ_DR);
	strcpy(dirMcrgraph, EMS_PRJ_DR);
	strcpy(dirParticle, EMS_PRJ_DR);
	strcpy(dirAlignmnt, EMS_PRJ_DR);
	strcpy(dirTemplate, EMS_PRJ_DR);
	strcpy(dirModeling, EMS_PRJ_DR);
	strcpy(dirFrealign, EMS_PRJ_DR);
	strcpy(dirATScreen, EMS_PRJ_DR);

	// system status flags
	sysStatus.imgInform = EMS_HSTG_MODE;
	EMS_DP->btnSpecMode->value(0);
	EMS_DP->btnHstgMode->value(1);

	sysStatus.drawTools = EMS_FALSE;
	EMS_DP->btnLines->value(0);
	EMS_DP->btnCircle->value(0);
	EMS_DP->btnPaint->value(0);
	EMS_DP->btnErase->value(0);

	sysStatus.showMask = EMS_TRUE;
	EMS_DP->btnShowMask->value(1);

	sysStatus.showPrtk = EMS_TRUE;
	EMS_DP->btnShowPrtk->value(0);

	sysStatus.showZoom = EMS_FALSE;
	EMS_DP->btnShowZoom->value(0);

	sysStatus.testScreen = EMS_FALSE;
	sysStatus.fullScreen = EMS_TRUE;
	sysStatus.rotaScreen = EMS_TRUE;
	EMS_DP->btnRotate->value(1);

	sysStatus.showRefr = EMS_FALSE;
	EMS_DP->btnRefr->value(0);

	sysStatus.showSpec = EMS_FALSE;
	EMS_DP->btnSpec->value(0);

	sysStatus.editTemp = EMS_FALSE;
	EMS_DP->btnEditFrame->value(0);

	sysStatus.activeL2 = EMS_FALSE;
	EMS_DP->btnL2Mask->value(0);

	sysStatus.newScale = EMS_TRUE;
	sysStatus.syncMask = EMS_FALSE;
	sysStatus.mapRestore = EMS_FALSE;

	// MICROGRAPH panel
	EMS_DP->dspFilmFile->value("\0");
	EMS_DP->dspFilmX->value(0);
	EMS_DP->dspFilmY->value(0);

	parameter.imgX = 0;
	parameter.imgY = 0;
	parameter.filmSize = 0;

	parameter.prtkSize = EMS_PRTK_SIZE;
	EMS_DP->dspPrtkSize->value(EMS_PRTK_SIZE);

	parameter.pixelSize = EMS_PIXEL_SIZE;
	EMS_DP->dspPixSize->value(EMS_PIXEL_SIZE);

	parameter.variaBox = EMS_VARIABOX;
	EMS_DP->valVarbox->value(EMS_VARIABOX);

	parameter.downSize = EMS_DOWNSIZE;
	EMS_DP->dspDownSize->value(EMS_DOWNSIZE);

	parameter.specSize = EMS_SPEC_SIZE;
	EMS_DP->valSpecSize->value(EMS_SPEC_SIZE);

	parameter.flmBrit = EMS_BRIGHTNESS;
	EMS_DP->sldBrit->value(EMS_BRIGHTNESS);

	parameter.flmCntr = EMS_CONTRAST;
	EMS_DP->sldCntr->value(EMS_CONTRAST);

	parameter.spcBrit = EMS_BRIGHTNESS;
	EMS_DP->specBrit->value(EMS_BRIGHTNESS);

	parameter.spcCntr = EMS_CONTRAST;
	EMS_DP->specCntr->value(EMS_CONTRAST);

	parameter.edgeExcu = EMS_EDGE_EXCU;
	EMS_DP->valEdge->value(EMS_EDGE_EXCU);

	parameter.varHi = EMS_VAR_HI;
	EMS_DP->sldVarH->value(EMS_VAR_HI);

	parameter.varLo = EMS_VAR_LO;
	EMS_DP->sldVarL->value(EMS_VAR_LO);

	parameter.brushSize = EMS_BRUSH;
	EMS_DP->valBrush->value(EMS_BRUSH);
	EMS_DP->valBrush2->value(EMS_BRUSH);

	parameter.passL = EMS_L_PASS;
	parameter.passH = EMS_H_PASS;

	parameter.transp = EMS_MSK_A;
//	EMS_DP->sldTrans->value(EMS_MSK_A);

	// clear keynote text
	EMS_DP->dspKeyNote->value("\0");

	// TEMPLATE panel
	EMS_DP->dspTempFile->value("\0");
	EMS_DP->dspTempN->value(0);
	EMS_DP->dspTempX->value(0);
	EMS_DP->dspTempY->value(0);

	refer.editFrame = -1;
	EMS_DP->valRefSelect->value(0);

	parameter.tmpX = 0;
	parameter.tmpY = 0;
	parameter.tmpN = 0;
	parameter.tmpSize = 0;
	parameter.lcfMaskNum = 0;
	parameter.scfMaskNum = 0;

	parameter.lcfMaskRad = EMS_LCF_MASKR;
	EMS_DP->sldLCFrad->value(EMS_LCF_MASKR);

	parameter.scfMaskRad = EMS_SCF_MASKR;
	EMS_DP->sldSCFrad->value(EMS_SCF_MASKR);

	parameter.scfMaskRadc = EMS_SCF_MASKC;
	EMS_DP->sldSCFradc->value(EMS_SCF_MASKC);

	parameter.rotaRange = EMS_ROTA_RANGE;
	EMS_DP->dspRotaRange->value(EMS_ROTA_RANGE);

	parameter.rotaDelta = EMS_ROTA_DELTA;
	EMS_DP->dspRotaDelta->value(EMS_ROTA_DELTA);

	parameter.refBrit = EMS_BRIGHTNESS;
	parameter.refCntr = EMS_CONTRAST;

	// PARTICLE panel
	parameter.lcfCount = 0;
	parameter.lcfAve = 0.0;
	parameter.lcfMax = 0.0;
	EMS_DP->cntLCF->value(0);
	EMS_DP->aveLCF->value(0.000);
	EMS_DP->maxLCF->value(0.000);
	parameter.lcfThres = EMS_LCF_THRES;
	EMS_DP->sldLCF->value(EMS_LCF_THRES);

	parameter.scfCount = 0;
	parameter.scfAve = 0.0;
	parameter.scfMax = 0.0;
	EMS_DP->cntSCF->value(0);
	EMS_DP->aveSCF->value(0.000);
	EMS_DP->maxSCF->value(0.000);
	parameter.scfThres = EMS_SCF_THRES;
	EMS_DP->sldSCF->value(EMS_SCF_THRES);

	parameter.distance = EMS_PRTK_DIST;
	EMS_DP->dspDist->value(EMS_PRTK_DIST);

	sqrFrameSize = EMS_FRAME_SIZE;
	EMS_DP->dspFrameSize->value(EMS_FRAME_SIZE);

	parameter.monBrit = EMS_BRIGHTNESS;
	EMS_DP->monBrit->value(EMS_BRIGHTNESS);

	parameter.monCntr = EMS_CONTRAST;
	EMS_DP->monCntr->value(EMS_CONTRAST);

	inspectID = inspectUD = -1;
	parameter.dcfCount = 0;
	parameter.dcfAve = 0.0;
	parameter.dcfMax = 0.0;
	EMS_DP->sldScore->minimum(0.000);
	EMS_DP->sldScore->maximum(1.000);
	EMS_DP->sldScore->value(0.000);

	// 2D IMAGE panel
	parameter.mon3Brit = EMS_BRIGHTNESS;
	EMS_DP->mon3Brit->value(EMS_BRIGHTNESS);

	parameter.mon3Cntr = EMS_CONTRAST;
	EMS_DP->mon3Cntr->value(EMS_CONTRAST);

	cropMode = EMS_FALSE;
	EMS_DP->btnOrtho->value(0);

	filaDiam = EMS_FILA_DIAM;
	EMS_DP->dspFilaDiam->value(EMS_FILA_DIAM);

	filaUnit = EMS_FILA_UNIT;
	EMS_DP->dspUnitLeng->value(EMS_FILA_UNIT);

	filaCrox = EMS_FILA_CROX;
	EMS_DP->dspUnitCrox->value(EMS_FILA_CROX);

	sysStatus.filaAve = EMS_FALSE;
	EMS_DP->btnFilaAve->value(0);

	cPanel->clearMonitor();
	cPanel->monitorFila(NULL, 0, 0);
	EMS_DP->imgDisplay->clearFilaBoxing();

	// MONITORs
	cPanel->setResz1(EMS_MON_RESZ1);
	EMS_DP->mon1Resz->value(EMS_MON_RESZ1);

	cPanel->setBlur1(EMS_MON_BLUR1);
	EMS_DP->mon1Blur->value(EMS_MON_BLUR1);

	cPanel->setResz2(EMS_MON_RESZ2);
	EMS_DP->mon2Resz->value(EMS_MON_RESZ2);

	cPanel->setBlur2(EMS_MON_BLUR2);
	EMS_DP->mon2Blur->value(EMS_MON_BLUR2);

	cPanel->setResz3(EMS_MON_RESZ3);
	EMS_DP->mon3Resz->value(EMS_MON_RESZ3);

	sysStatus.mon1X = EMS_TRUE;
	sysStatus.mon2X = EMS_TRUE;
	sysStatus.mon3X = EMS_TRUE;
	EMS_DP->btnCross1->value(1);
	EMS_DP->btnCross2->value(1);
	EMS_DP->btnCross3->value(1);

	// clear all display information
	cPanel->clearImgInfo();
	cPanel->clearMonitor();
	EMS_DP->imgDisplay->clearFocus();
	EMS_DP->imgDisplay->clearFilmMarker();

	// MESSAGE panel
	parameter.totCount = 0;
	EMS_DP->dspCount->value(0);
	msgReport("\0");
}
