//	James Z. Chen
//	January 1, 2005
//	All rights reserved

#ifndef __MAPS_H
#define __MAPS_H 1

#include "/usr/include/stdio.h"
#include "/usr/include/stdlib.h"
#include "/usr/include/string.h"
#include "/usr/include/math.h"
#include "parameter.h"


struct 	MRC_Header {
	int   nXYZ[3]; //nxyz[3];
	int   mode;
	int   nXYZstart[3];
	int   mXYZ[3];
	float cell[6];
	int	  mapCRS[3];
	float mapMin, mapMax, mapAve;
	int	  spaceGroup;
	int	  symmetry;
	int	  extra[25];
	float oXYZ[3];
	char  mapID[4];
	int   machine;
	float mapStd;
	int	  nLabel;
	char  label[800];
};


struct SPD_Header {
	float  nslice;
	float  nrow;
	float irec[2];
	float iform;
	float gap1[6];
	float nsam;
	float gap2[9];
	float hsize;
};


class MRC_MAP {
public:
	char  exist;					// flag indicating existance
	char  endian;					// flag for endin property (FALSE for no-swap)
	int   dimX, dimY, dimZ;			// image dimension
	int   samX, samY, samZ;			// image resampling
	int   mode, dataBytes;			// MRC format mode
	long  mapSize;					// number of pi(vo)xels
	float maxInt, minInt;			// intensity range
	float aveInt, stdInt;			// intensity statistics
	float pixelX, pixelY, pixelZ;	// pixel size
	int   binBox;					// image element binning
	int   hstGram[EMS_INFO_BIN];	// store histogram count

	// electron density data, normalized to N(0,1), always mode 2
	float *density;

	MRC_MAP();
	~MRC_MAP();

	// loading map data to density_original[]
	char input(char *map_file);
	char input_spd(char *map_file);
	char input_tif(char *map_file);

	// writing map data to file
	char output(char *map_file);

	// clear all map entries
	void clear(void);

	// MIN, MAX, AVE, STD statistics
	void statistics(void);

	// normalize map data to N(0.0, 1.0)
	void normalize(void);

	// build intensity histogram
	void histogram(void);
	void clipping(float valMin, float valMax);

	// invert intensity contrast
	void invert(void);

	// element binning
	void bin2D(void);

	// image flipping
	void flipH(void);
	void flipV(void);

	// image resampling
	char resample2D(void);

	// set map parameters for writing
	void setDimensn(int x, int y, int z, float px, float py, float pz);
	void setDensity(float *data);
};


class EMS_MAP {
public:
	char  exist;	// EMS_TRUE for data existance
	char  *dataC;	// char[] mask data
	float *dataF;	// float[] image data
	float *rel;		// real-part of FFT
	float *img;		// imag-part of FFT
	int dimX, dimY;	// image dimension
	int fftX, fftY;	// FT image dimension
	long mapSize;	// size of the map data: dimX*dimY
	long fftSize;	// size of the FFT data: fftX*fftY

	EMS_MAP();
	~EMS_MAP();

	// allocate memory and initialize data array
	void initC(char val, int x, int y);
	void initF(float val, int x, int y);

	// allocate memory and load data, calculate FT if (ftx,fty>0)
	void setup(char *map, int x, int y, int ftx, int fty);
	void setup(int x, int y, char *map, int ftx, int fty, float *fftR, float *fftI);
	void setup(float *map, int x, int y, int ftx, int fty);
	void setup(int x, int y, float *map, int ftx, int fty, float *fftR, float *fftI);

	// set data array element to constant value
	void resetC(char val);
	void resetF(float val);

	// clear all map entries
	inline void clear(void) {
		if ( dataC != NULL ) delete [] dataC;
		if ( dataF != NULL ) delete [] dataF;
		if ( rel != NULL ) delete [] rel;
		if ( img != NULL ) delete [] img;
		
		dataC = NULL;
		dataF = NULL;
		rel = img = NULL;
		dimX = dimY = 0; mapSize = 0;
		fftX = fftY = 0; fftSize = 0;
		exist = EMS_FALSE;
	}

	// compute FFT for FILM-sized image, swap quadrants
	void imageFFT(int ftx, int fty);

	// compute FFT for TEMPLATE-sized image, put to center
	void kernelFFT(int ftx, int fty);

	// swap image quadrants
	void swapQuad(void);
};


// filtering via FFT
void localCOR(EMS_MAP &kernel, EMS_MAP &image, EMS_MAP &stdv, EMS_MAP &result, float scaling);
void localSTD(EMS_MAP &kernel, EMS_MAP &image, EMS_MAP &result, float scaling);
void localAVE(EMS_MAP &kernel, EMS_MAP &image, EMS_MAP &result, float scaling);

// multiply EMS_MAP data
void multiplyMap(EMS_MAP &src1, EMS_MAP &src2, EMS_MAP &dst);

// divide EMS_MAP data
void divideMap(EMS_MAP &src1, EMS_MAP &src2, EMS_MAP &dst);

// copy EMS_MAP data
void copyMap(EMS_MAP &dst, EMS_MAP &src);

// smooth and interpolate a power spectrum image
void smoothSpectrum(EMS_MAP &src, EMS_MAP &dst, int box);

// smooth an EMS_MAP image
void smoothMap(EMS_MAP &src, int box);


#endif
