//	James Z. Chen
//	January 1, 2005
//	All rights reserved

#include "maps.h"
#include "misc.h"
#include "fftn.h"
#include "project.h"
#include "tiffio.h"


#define MRCH_OLD	1024	// header size of OLD MRC
#define MRCH_NEW	1104	// header size of NEW MRC
#define OLD2NEW		80		// 80 extra chars in NEW MRC header
#define RGB_CHANNEL	2		// 32-bit TIFF pixel: ABGR


MRC_MAP::MRC_MAP()
{
	mode = -1;
	dimX = dimY = dimZ = 0;
	samX = samY = samZ = 0;
	mapSize = 0;
	binBox = 1;
	dataBytes = 0;
	endian = EMS_FALSE;
	pixelX = pixelY = pixelZ = 0.0;
	maxInt = aveInt = minInt = stdInt = 0.0;
	density = NULL;
	exist = EMS_FALSE;

	for ( int index=0; index<EMS_INFO_BIN; index++ )
		hstGram[index] = 0;
}


MRC_MAP::~MRC_MAP()
{
	if ( density != NULL ) {
		delete [] density;
		density = NULL;
	}
}


// load MRC image
char MRC_MAP::input(char *map_file)
{
	long index, file_size;
	struct MRC_Header mapHeader;
	unsigned char *buffer, symstr[OLD2NEW];
	FILE *fp;
	
	// open map file to read
	fp = fopen(map_file, "rb");
	
	if ( fp == NULL ) {
		errReport("Cannot open this MRC image file!");
		return(EMS_FALSE);
	}
	
	// read map parameters from the header
	file_size = fileSize(fp);
	fread(&mapHeader, sizeof(struct MRC_Header), 1, fp);
	
	// byteswap header if necessary (normal modes take small ID#)
	if ( mapHeader.mode > 255 ) {
		endian = EMS_TRUE;
		byteSwap((char *)&mapHeader.mode, sizeof(int), 1);
		byteSwap((char *)&mapHeader.spaceGroup, sizeof(int), 1);
		byteSwap((char *)&mapHeader.symmetry, sizeof(int), 1);
		byteSwap((char *)&mapHeader.nLabel, sizeof(int), 1);
		byteSwap((char *)mapHeader.oXYZ, sizeof(float), 3);
		byteSwap((char *)mapHeader.nXYZ, sizeof(int), 3);
		byteSwap((char *)mapHeader.mXYZ, sizeof(int), 3);
		byteSwap((char *)mapHeader.mapCRS, sizeof(int), 3);
		byteSwap((char *)&mapHeader.mapMin, sizeof(float), 1);
		byteSwap((char *)&mapHeader.mapMax, sizeof(float), 1);
		byteSwap((char *)&mapHeader.mapAve, sizeof(float), 1);
		byteSwap((char *)&mapHeader.mapStd, sizeof(float), 1);
		byteSwap((char *)mapHeader.cell, sizeof(float), 6);
	}
	else
		endian = EMS_FALSE;

	// 3D data array SHOULD be in X-Y-Z order
	if ( mapHeader.mapCRS[0] != 1 )
		msgReport("Density map is not in X-Y-Z order. Proceed anyway ...");

	mode = mapHeader.mode;
	dimX = mapHeader.nXYZ[0];
	dimY = mapHeader.nXYZ[1];
	dimZ = mapHeader.nXYZ[2];

	// pixel size (A) has to be physically meaningful
	if ( (mapHeader.cell[0]>EMS_ZERO) && (mapHeader.mXYZ[0]>EMS_ZERO) )
		pixelX = mapHeader.cell[0] / mapHeader.mXYZ[0];
	if ( pixelX < 0.1 ) pixelX = EMS_PIXEL_SIZE;

	if ( (mapHeader.cell[1]>EMS_ZERO) && (mapHeader.mXYZ[1]>EMS_ZERO) )
		pixelY = mapHeader.cell[1] / mapHeader.mXYZ[1];
	if ( pixelY < 0.1 ) pixelY = EMS_PIXEL_SIZE;

	if ( (mapHeader.cell[2]>EMS_ZERO) && (mapHeader.mXYZ[2]>EMS_ZERO) )
		pixelZ = mapHeader.cell[2] / mapHeader.mXYZ[2];
	if ( pixelZ < 0.1 ) pixelZ = EMS_PIXEL_SIZE;

	// calculate number of bytes to read
	if ( mode == 0 )		// signed char
		dataBytes = sizeof(char);
	else if ( mode == 1 )	// short int (2-bytes)
		dataBytes = sizeof(short);
	else if ( mode == 2 )	// float (4-bytes)
		dataBytes = sizeof(float);
	else {
		errReport("Unrecognized image format!");
		return(EMS_FALSE);
	}
	
	mapSize = dimX * dimY * dimZ;

	// safe-guard data format
	if ( (file_size-MRCH_NEW) == (mapSize*dataBytes) ) {
		fread(symstr, OLD2NEW, 1, fp);
	}
	else if ( (file_size-MRCH_OLD) != (mapSize*dataBytes) ) {
		errReport("Incorrect map dimension or data type!");
		return(EMS_FALSE);
	}

	density = new float[mapSize];
	buffer = new unsigned char[mapSize * dataBytes];
	
	if ( (density==NULL) || (buffer==NULL) ) {
		errReport("Cannot allocate enough memory for this density map!");
		return(EMS_FALSE);
	}
	
	fread(buffer, mapSize, dataBytes, fp);
	fclose(fp);
	
	if ( endian == EMS_TRUE )
		byteSwap((char *)buffer, dataBytes, mapSize);

	for ( index=0; index<mapSize; index++ ) {
		switch ( mode ) {
		case 0:
			density[index] = buffer[index];
			break;
		case 1:
			density[index] = ((short *)buffer)[index];
			break;
		case 2:
			density[index] = ((float *)buffer)[index];
			break;
		}
	}

	delete [] buffer;

	// evaluate intensity range (cannot trust the original header)
	statistics();
	exist = EMS_TRUE;

	return(EMS_TRUE);
}


// load TIFF image
char MRC_MAP::input_tif(char *map_file)
{
	long x, y, index;
	uint32 *raster;
	unsigned char *pval;
	TIFF *tif;

    if ( (tif=TIFFOpen(map_file, "r")) != NULL ) {
		
		TIFFGetField(tif, TIFFTAG_IMAGEWIDTH, &dimX);
		TIFFGetField(tif, TIFFTAG_IMAGELENGTH, &dimY);
		
		dimZ = 1;
		mapSize = dimX * dimY * dimZ;
		pixelX = pixelY = pixelZ = EMS_PIXEL_SIZE;

		if ( mapSize > 0 ) {
			raster = (uint32 *) _TIFFmalloc(mapSize * sizeof (uint32));
			
			if ( raster != NULL ) {
				if ( TIFFReadRGBAImage(tif, dimX, dimY, raster, 0) ) {
					
					pval = (unsigned char *)raster;
					density = new float[mapSize];

					if ( density != NULL ) {
						// loading map intensity to density[]
						for ( y=0; y<dimY; y++ ) {
							for ( x=0; x<dimX; x++ ) {
								index = 4 * ((dimY-1-y) * dimX + x);
								density[y*dimX+x] = (float)(pval[index+RGB_CHANNEL]);
							}
						}
						
						statistics();
						exist = EMS_TRUE;
					}
					else
						errReport("Cannot allocate enough memory for this density map!");
				}
				
				_TIFFfree(raster);
			}
			else
				errReport("Cannot allocate enough memory to read this map!");
		}
		else {
			errReport("Unknown image format!");
		}
		
		TIFFClose(tif);	
    }
	else {
		errReport("Cannot open this TIFF image file!");
	}

	return(exist);
}


// load SPD image
char MRC_MAP::input_spd(char *map_file)
{
	long header_size, file_size;
	struct SPD_Header mapHeader;
	FILE *fp;
	
	// open map file to read
	fp = fopen(map_file, "rb");
	
	if ( fp == NULL ) {
		errReport("Cannot open this SPD image file!");
		return(EMS_FALSE);
	}

	// read map parameters from the header
	file_size = fileSize(fp);
	fread(&mapHeader, sizeof(struct SPD_Header), 1, fp);
	
	// byteswap header if necessary (normal modes take 1 or 3)
	if ( ((int)mapHeader.iform!=1) && ((int)mapHeader.iform!=3) ) {
		endian = EMS_TRUE;
		byteSwap((char *)&mapHeader.nrow, sizeof(float), 1);
		byteSwap((char *)&mapHeader.nsam, sizeof(float), 1);
		byteSwap((char *)&mapHeader.nslice, sizeof(float), 1);
		byteSwap((char *)&mapHeader.iform, sizeof(float), 1);
		byteSwap((char *)&mapHeader.hsize, sizeof(float), 1);
	}
	else
		endian = EMS_FALSE;

	mode = (int)(mapHeader.iform);
	dimX = (int)(mapHeader.nsam);
	dimY = (int)(mapHeader.nrow);
	dimZ = (int)(mapHeader.nslice);
	mapSize = dimX * dimY * dimZ;
	pixelX = pixelY = pixelZ = EMS_PIXEL_SIZE;
	header_size = (int)(mapHeader.hsize);

	// calculate number of bytes to read
	if ( (mode!=1) && (mode!=3) ) {
		errReport("Unrecognized image format!");
		return(EMS_FALSE);
	}
	else if ( dimZ != 1 ) {
		errReport("This is not a 2D image!");
		return(EMS_FALSE);
	}
	else if ( (file_size-(mapSize*sizeof(float)+header_size)) != 0 ) {
		errReport("Incorrect map dimension or data type!");
		return(EMS_FALSE);
	}

	density = new float[mapSize];
	
	if ( density == NULL ) {
		errReport("Cannot allocate enough memory for this density map!");
		return(EMS_FALSE);
	}

	fseek(fp, header_size, SEEK_SET);
	fread(density, mapSize, sizeof(float), fp);
	fclose(fp);

	if ( endian == EMS_TRUE )
		byteSwap((char *)density, sizeof(float), mapSize);

	// evaluate intensity range (cannot trust the original header)
	statistics();
	exist = EMS_TRUE;

	return(EMS_TRUE);
}


// write out an MRC map without symmetry record
char MRC_MAP::output(char *map_file)
{
	int index;
	FILE *fp;
	struct MRC_Header mapHeader;

	if ( density == NULL ) {
		errReport("There is no data to save!");
		return(EMS_FALSE);
	}

	// set map parameters
	// must call setDimensn() and setDensity() beforehand
	statistics();
	mapHeader.mode = mode;

	// set number of bytes to write
	if ( mode == 0 ) {       // 1-byte data
		dataBytes = sizeof(char);
	}
	else if ( mode == 2 ) {  // 4-byte reals
		dataBytes = sizeof(float);
	}
	else {
		errReport("Incorrect MRC map mode!");
		return(EMS_FALSE);
	}

	// setup map dimensions, etc.
	mapHeader.nXYZ[0] = dimX;
	mapHeader.nXYZ[1] = dimY;
	mapHeader.nXYZ[2] = dimZ;

	mapHeader.nXYZstart[0] = 0;
	mapHeader.nXYZstart[1] = 0;
	mapHeader.nXYZstart[2] = 0;

	mapHeader.mXYZ[0] = dimX;
	mapHeader.mXYZ[1] = dimY;
	mapHeader.mXYZ[2] = dimZ;

	mapHeader.oXYZ[0] = 0;
	mapHeader.oXYZ[1] = 0;
	mapHeader.oXYZ[2] = 0;

	mapHeader.mapCRS[0] = 1;
	mapHeader.mapCRS[1] = 2;
	mapHeader.mapCRS[2] = 3;

	mapHeader.cell[0] = pixelX * dimX;
	mapHeader.cell[1] = pixelY * dimY;
	mapHeader.cell[2] = pixelZ * dimZ;
	mapHeader.cell[3] = 90.0;
	mapHeader.cell[4] = 90.0;
	mapHeader.cell[5] = 90.0;

	mapHeader.mapMin = minInt;
	mapHeader.mapMax = maxInt;
	mapHeader.mapAve = aveInt;
	mapHeader.mapStd = stdInt;

	mapHeader.spaceGroup = 0;
	mapHeader.symmetry = 0;
	mapHeader.machine = 0;
	strncpy(mapHeader.mapID, "MAP ", 4);

	for ( index=0; index<25; index++ ) mapHeader.extra[index] = 0;
	for ( index=0; index<800; index++ ) mapHeader.label[index] = ' ';

	mapHeader.nLabel = 1;
	strcpy(mapHeader.label, "EMS particle selection");

	fp = fopen(map_file, "wb");

	if ( fp == NULL ) {
		sprintf(message, "Cannot write to file %s!", map_file);
		errReport(message);
		return(EMS_FALSE);
	}

	fwrite(&mapHeader, sizeof(struct MRC_Header), 1, fp);
	fwrite(density, mapSize, dataBytes, fp);
	fclose(fp);

	return(EMS_TRUE);
}


void MRC_MAP::clear(void)
{
	int index;

	if ( density != NULL ) {
		delete [] density;
		density = NULL;
	}

	mode = -1;
	dimX = dimY = dimZ = 0;
	mapSize = 0;
	dataBytes = 0;
	endian = EMS_FALSE;
	pixelX = pixelY = pixelZ = 0.0;
	maxInt = aveInt = minInt = stdInt = 0.0;
	exist = EMS_FALSE;

	for ( index=0; index<EMS_INFO_BIN; index++ )
		hstGram[index] = 0;
}


void MRC_MAP::statistics(void)
{
	long index;
	double std_t, ave_t;

	maxInt = -1.0E9;
	minInt = +1.0E9;
	ave_t = std_t = 0.0;

	for ( index=0; index<mapSize; index++ ) {
		ave_t += density[index];
		std_t += density[index] * density[index];

		if ( density[index] > maxInt )
			maxInt = density[index];

		if ( density[index] < minInt )
			minInt = density[index];
	}

	aveInt = (float)(ave_t / mapSize);
	stdInt = (float)(std_t / mapSize - aveInt * aveInt);

	if ( stdInt > 0.0 )
		stdInt = (float)(sqrt(stdInt));
	else
		stdInt = EMS_ZERO;
}


void MRC_MAP::normalize(void)
{
	long index;

	if ( exist == EMS_FALSE ) return;

	statistics();

	for ( index=0; index<mapSize; index++ )
		density[index] = (float)((density[index] - aveInt) / stdInt);

	maxInt = (float)((maxInt - aveInt) / stdInt);
	minInt = (float)((minInt - aveInt) / stdInt);
	aveInt = 0.0; stdInt = 1.0;
}


void MRC_MAP::histogram(void)
{
	int index, bin_id;
	float deltInt, minHst;

	if ( exist == EMS_FALSE ) return;

	for ( index=0; index<EMS_INFO_BIN; index++ ) {
		hstGram[index] = 0;
	}

	// in 2D case, all intensity values
	statistics();

	if ( dimZ == 1 ) {
		deltInt = (maxInt - minInt) / (EMS_INFO_BIN - 1) + EMS_ZERO;
		
		for ( index=0; index<mapSize; index++ ) {
			bin_id = (int)((density[index] - minInt) / deltInt);
			hstGram[bin_id] ++;
		}
	}

	// in 3D case, within EMS_HST3_RANGE s.t.d. only
	else {
		minHst = aveInt + EMS_HST3_BIN0 * stdInt;
		deltInt = (EMS_HST3_BIN1 - EMS_HST3_BIN0) * stdInt / (EMS_INFO_BIN - 1) + EMS_ZERO;

		for ( index=0; index<mapSize; index++ ) {
			bin_id = (int)((density[index] - minHst) / deltInt);
			if ( bin_id < 0 ) bin_id = 0;
			if ( bin_id >= EMS_INFO_BIN ) bin_id = EMS_INFO_BIN - 1;
			hstGram[bin_id] ++;
		}
	}
}


void MRC_MAP::clipping(float valMin, float valMax)
{
	long index;

	if ( exist == EMS_FALSE ) return;

	for ( index=0; index<mapSize; index++ ) {
		if ( density[index] < valMin )
			density[index] = valMin;
		else if ( density[index] > valMax )
			density[index] = valMax;
	}

	histogram();
}


void MRC_MAP::invert(void)
{
	long index;

	if ( exist == EMS_FALSE ) return;

	for ( index=0; index<mapSize; index++ )
		density[index] *= -1.0;

	statistics();
}


void MRC_MAP::bin2D(void)
{
	int xx, yy, binX, binY, binBox2;
	long x, y, z, index, binMapSize;
	float *binMap, binSum;

	binX = dimX / binBox;
	binY = dimY / binBox;
	binBox2 = binBox * binBox;

	binMapSize = binX * binY * dimZ;
	binMap = new float[binMapSize];

	for ( z=0; z<dimZ; z++ ) {
		for ( y=0; y<binY; y++ ) {
			for ( x=0; x<binX; x++ ) {
				binSum = 0.0;

				for ( yy=0; yy<binBox; yy++ ) {
					for ( xx=0; xx<binBox; xx++ ) {
						index = (z*dimY+y*binBox+yy)*dimX + x*binBox + xx;
						binSum += density[index];
					}
				}

				binMap[(z*binY+y)*binX+x] = binSum / binBox2;
			}
		}
	}

	delete [] density;

	dimX = binX; pixelX *= binBox;
	dimY = binY; pixelY *= binBox;
	mapSize = binMapSize;

	density = new float[mapSize];
	memcpy(density, binMap, sizeof(float)*mapSize);
	statistics();

	delete [] binMap;
}


void MRC_MAP::flipH(void)
{
	int x, y, z;
	long src, dst;

	for ( z=0; z<dimZ; z++ ) {
		for ( y=0; y<dimY; y++ ) {
			for ( x=0; x<(dimX/2); x++ ) {
				src = (z * dimY + y) * dimX + x;
				dst = (z * dimY + y) * dimX + dimX - 1 - x;
				swapValF(density[src], density[dst]);
			}
		}
	}
}


void MRC_MAP::flipV(void)
{
	int x, y, z;
	long src, dst;

	for ( z=0; z<dimZ; z++ ) {
		for ( y=0; y<(dimY/2); y++ ) {
			for ( x=0; x<dimX; x++ ) {
				src = (z * dimY + y) * dimX + x;
				dst = (z * dimY + dimY - 1 - y) * dimX + x;
				swapValF(density[src], density[dst]);
			}
		}
	}
}


char MRC_MAP::resample2D(void)
{
	int x, y, z, x0, y0, x1, y1;
	long samMapSize, page;
	float rateX, rateY, xf, yf, dA, dB, dC, dD, dE, dF;
	float *samMap;

	if ( (samX<3) || (samY<3) ) {
		errReport("Resampling rate is too low!");
		return(EMS_FALSE);
	}

	rateX = 1.0f * (dimX-1) / (samX-1);
	rateY = 1.0f * (dimY-1) / (samY-1);
	samMapSize = samX * samY * dimZ;
	samMap = new float[samMapSize];

	for ( z=0; z<dimZ; z++ ) {
		page = z * dimX * dimY;

		// four corners
		samMap[z*samY*samX] = density[z*dimY*dimX];
		samMap[(z*samY+1)*samX-1] = density[(z*dimY+1)*dimX-1];
		samMap[((z+1)*samY-1)*samX] = density[((z+1)*dimY-1)*dimX];
		samMap[(z+1)*samY*samX-1] = density[(z+1)*dimY*dimX-1];

		// on the boundary
		for ( x=1; x<(samX-1); x++ ) {
			xf = rateX * x;
			x0 = (int)floor(xf); x1 = x0 + 1;
			
			dA = density[page + x0];
			dB = density[page + x1];
			samMap[z*samY*samX+x] = (x1-xf) * dA + (xf-x0) * dB;

			dA = density[page + (dimY-1) * dimX + x0];
			dB = density[page + (dimY-1) * dimX + x1];
			samMap[((z+1)*samY-1)*samX+x] = (x1-xf) * dA + (xf-x0) * dB;
		}

		for ( y=1; y<(samY-1); y++ ) {
			yf = rateY * y;
			y0 = (int)floor(yf); y1 = y0 + 1;
			
			dA = density[page + y0 * dimX];
			dB = density[page + y1 * dimX];
			samMap[(z*samY+y)*samX] = (y1-yf) * dA + (yf-y0) * dB;

			dA = density[page + y0 * dimX + dimX - 1];
			dB = density[page + y1 * dimX + dimX - 1];
			samMap[(z*samY+y+1)*samX-1] = (x1-xf) * dA + (xf-x0) * dB;
		}

		// inside the frame
		for ( y=1; y<(samY-1); y++ ) {
			for ( x=1; x<(samX-1); x++ ) {

				xf = rateX * x; yf = rateY * y;
				x0 = (int)floor(xf); x1 = x0 + 1;
				y0 = (int)floor(yf); y1 = y0 + 1;
				
				dA = density[page + y0 * dimX + x0];
				dB = density[page + y0 * dimX + x1];
				dC = density[page + y1 * dimX + x0];
				dD = density[page + y1 * dimX + x1];
				dE = (x1-xf) * dA + (xf-x0) * dB;
				dF = (x1-xf) * dC + (xf-x0) * dD;

				samMap[(z*samY+y)*samX+x] = (y1-yf) * dE + (yf-y0) * dF;
			}
		}
	}

	delete [] density;

	dimX = samX; pixelX *= rateX;
	dimY = samY; pixelY *= rateY;
	mapSize = samMapSize;

	density = new float[mapSize];
	memcpy(density, samMap, sizeof(float)*mapSize);
	statistics();

	delete [] samMap;

	return(EMS_TRUE);
}


void MRC_MAP::setDimensn(int x, int y, int z, float px, float py, float pz)
{
	dimX = x;
	dimY = y;
	dimZ = z;

	pixelX = px;
	pixelY = py;
	pixelZ = pz;

	// MODE-2 MRC MAP ONLY
	mode = 2;
	mapSize = dimX * dimY * dimZ;
}


void MRC_MAP::setDensity(float *data)
{
	long index;

	if ( mapSize == 0 ) {
		errReport("Map dimension is undefined!");
		return;
	}

	density = new float[mapSize];
	memcpy(density, data, mapSize * sizeof(float));

	aveInt = 0.0; stdInt = 0.0;
	maxInt = -1.0E9; minInt = +1.0E9;

	for ( index=0; index<mapSize; index++ ) {
		aveInt += density[index];
		stdInt += density[index] * density[index];

		if ( density[index] > maxInt )
			maxInt = density[index];

		if ( density[index] < minInt )
			minInt = density[index];
	}

	aveInt /= mapSize;
	stdInt = stdInt / mapSize - aveInt * aveInt;

	if ( stdInt > 0.0 )
		stdInt = (float)(sqrt(stdInt));
	else
		stdInt = 0.0f;

	exist = EMS_TRUE;
}


EMS_MAP::EMS_MAP()
{
	dataC = NULL;
	dataF = NULL;
	rel = img = NULL;
	dimX = dimY = 0; mapSize = 0;
	fftX = fftY = 0; fftSize = 0;
	exist = EMS_FALSE;
}


EMS_MAP::~EMS_MAP()
{
	if ( dataC != NULL ) delete [] dataC;
	if ( dataF != NULL ) delete [] dataF;
	if ( rel != NULL ) delete [] rel;
	if ( img != NULL ) delete [] img;
}


void EMS_MAP::initC(char val, int x, int y)
{
	if ( x<=0 || y<=0 ) {
		exist = EMS_FALSE;
		return;
	}
	
	// allocate memory and copy data
	clear();
	dimX = x; dimY = y;
	mapSize = dimX * dimY;
	dataC = new char[mapSize];
	
	if ( dataC == NULL ) {
		exist = EMS_FALSE;
		return;
	}

	exist = EMS_TRUE;
	resetC(val);
}


void EMS_MAP::initF(float val, int x, int y)
{
	if ( x<=0 || y<=0 ) {
		exist = EMS_FALSE;
		return;
	}
	
	// allocate memory and copy data
	clear();
	dimX = x; dimY = y;
	mapSize = dimX * dimY;
	dataF = new float[mapSize];
	
	if ( dataF == NULL ) {
		exist = EMS_FALSE;
		return;
	}

	exist = EMS_TRUE;
	resetF(val);
}


void EMS_MAP::setup(char *map, int x, int y, int ftx, int fty)
{
	clear();

	if ( x<=0 || y<=0 ) {
		exist = EMS_FALSE;
		return;
	}
	
	// allocate memory and copy data
	dimX = x; dimY = y;
	mapSize = dimX * dimY;
	dataC = new char[mapSize];
	
	if ( dataC == NULL ) {
		exist = EMS_FALSE;
		return;
	}
	
	memcpy(dataC, map, mapSize * sizeof(char));
	exist = EMS_TRUE;
	
	// FFT only if required (ftx,fty>0)
	if ( ftx<=0 || fty<=0 )
		return;
	else if ( (ftx==x) && (fty==y) )
		imageFFT(ftx, fty);
	else if ( (ftx>x) && (fty>y) )
		kernelFFT(ftx, fty);
	else {
		errReport("Incorrect image data!");
		exist = EMS_FALSE;
		delete [] dataC;
		dataC = NULL;
	}
}


void EMS_MAP::setup(int x, int y, char *map, int ftx, int fty, float *fftR, float *fftI)
{
	clear();

	if ( x<=0 || y<=0 || ftx<0 || fty<0 ) {
		exist = EMS_FALSE;
		return;
	}

	// allocate memory and copy image data
	dimX = x; dimY = y;
	mapSize = dimX * dimY;
	dataC = new char[mapSize];
	memcpy(dataC, map, mapSize * sizeof(char));

	// allocate memory and copy FFT data
	fftX = ftx; fftY = fty;
	fftSize = fftX * fftY;
	rel = new float[fftSize];
	img = new float[fftSize];
	memcpy(rel, fftR, fftSize * sizeof(float));
	memcpy(img, fftI, fftSize * sizeof(float));

	exist = EMS_TRUE;
}


void EMS_MAP::setup(float *map, int x, int y, int ftx, int fty)
{
	clear();

	if ( x<=0 || y<=0 ) {
		exist = EMS_FALSE;
		return;
	}

	// allocate memory and copy data
	dimX = x; dimY = y;
	mapSize = dimX * dimY;
	dataF = new float[mapSize];
	
	if ( dataF == NULL ) {
		exist = EMS_FALSE;
		return;
	}
	
	memcpy(dataF, map, mapSize * sizeof(float));
	exist = EMS_TRUE;
	
	// FFT only if required (ftx,fty>0)
	if ( ftx<=0 || fty<=0 )
		return;
	else if ( (ftx==x) && (fty==y) )
		imageFFT(ftx, fty);
	else if ( (ftx>x) && (fty>y) )
		kernelFFT(ftx, fty);
	else {
		errReport("Incorrect image data!");
		exist = EMS_FALSE;
		delete [] dataF;
		dataF = NULL;
	}
}


void EMS_MAP::setup(int x, int y, float *map, int ftx, int fty, float *fftR, float *fftI)
{
	clear();

	if ( x<=0 || y<=0 || ftx<0 || fty<0 ) {
		exist = EMS_FALSE;
		return;
	}

	// allocate memory and copy image data
	dimX = x; dimY = y;
	mapSize = dimX * dimY;
	dataF = new float[mapSize];
	memcpy(dataF, map, mapSize * sizeof(float));

	// allocate memory and copy FFT data
	fftX = ftx; fftY = fty;
	fftSize = fftX * fftY;
	rel = new float[fftSize];
	img = new float[fftSize];
	memcpy(rel, fftR, fftSize * sizeof(float));
	memcpy(img, fftI, fftSize * sizeof(float));

	exist = EMS_TRUE;
}


void EMS_MAP::resetC(char val)
{
	long index;
	
	if ( exist == EMS_FALSE ) return;
	if ( rel != NULL ) delete [] rel;
	if ( img != NULL ) delete [] img;

	rel = img = NULL;
	fftX = fftY = 0; fftSize = 0;

	for ( index=0; index<mapSize; index++ )
		dataC[index] = val;
}


void EMS_MAP::resetF(float val)
{
	long index;
	
	if ( exist == EMS_FALSE ) return;
	if ( rel != NULL ) delete [] rel;
	if ( img != NULL ) delete [] img;

	rel = img = NULL;
	fftX = fftY = 0; fftSize = 0;

	for ( index=0; index<mapSize; index++ )
		dataF[index] = val;
}


void EMS_MAP::imageFFT(int ftx, int fty)
{
	int x, y, xx, yy, dimen[2];
	long index1, index2;
	
	if ( exist == EMS_FALSE ) {
		errReport("FFT: image data does not exist!");
		return;
	}
	else if ( fftSize > 0 ) {
		errReport("FFT: FT data already exists!");
		return;
	}
	else if ( (ftx!=dimX) || (fty!=dimY) ) {
		errReport("FFT: incompatible image data!");
		return;
	}
	
	fftX = dimen[0] = ftx;
	fftY = dimen[1] = fty;
	fftSize = fftX * fftY;

	rel = new float[fftSize];
	img = new float[fftSize];
	
	if ( dataC != NULL ) {
		for ( y=0; y<dimY; y++ ) {
			for ( x=0; x<dimX; x++ ) {
				// swap quadrants
				xx = (x + dimX/2) % dimX;
				yy = (y + dimY/2) % dimY;
				index1 = y * dimX + x;
				index2 = yy * dimX + xx;
				rel[index2] = (float)(dataC[index1]);
				img[index2] = 0.0f;
			}
		}
	}
	else if ( dataF != NULL ) {
		for ( y=0; y<dimY; y++ ) {
			for ( x=0; x<dimX; x++ ) {
				// swap quadrants
				xx = (x + dimX/2) % dimX;
				yy = (y + dimY/2) % dimY;
				index1 = y * dimX + x;
				index2 = yy * dimX + xx;
				rel[index2] = dataF[index1];
				img[index2] = 0.0f;
			}
		}
	}
	else {
		errReport("FFT: incorrect image data!");
		return;
	}

	fftwN(2, dimen, rel, img, FFTW_FORWARD);
}


void EMS_MAP::kernelFFT(int ftx, int fty)
{
	int x, y, dimen[2];
	long index;
	
	if ( exist == EMS_FALSE ) {
		errReport("FFT: image data does not exist!");
		return;
	}
	else if ( fftSize > 0 ) {
		errReport("FFT: FT data already exists!");
		return;
	}
	else if ( (dimX>ftx) || (dimY>fty) ) {
		errReport("FFT: kernel dimension is too large!");
		return;
	}
	
	fftX = dimen[0] = ftx;
	fftY = dimen[1] = fty;
	fftSize = fftX * fftY;

	rel = new float[fftSize];
	img = new float[fftSize];
	
	for ( index=0; index<fftSize; index++ ) {
		rel[index] = 0.0f;
		img[index] = 0.0f;
	}

	// put image data at the center of padded image
	if ( dataC != NULL ) {
		for ( y=0; y<dimY; y++ ) {
			for ( x=0; x<dimX; x++ ) {
				index = ((fftY-dimY)/2+y) * fftX + (fftX-dimX)/2 + x;
				rel[index] = (float)(dataC[y*dimX+x]);
			}
		}
	}
	else if ( dataF != NULL ) {
		for ( y=0; y<dimY; y++ ) {
			for ( x=0; x<dimX; x++ ) {
				index = ((fftY-dimY)/2+y) * fftX + (fftX-dimX)/2 + x;
				rel[index] = dataF[y*dimX+x];
			}
		}
	}
	else {
		errReport("FFT: incorrect image data!");
		return;
	}

	fftwN(2, dimen, rel, img, FFTW_FORWARD);
}


void EMS_MAP::swapQuad(void)
{
	int x, y, xx, yy;
	long index1, index2;
	char valC;
	float valF;

	if ( exist == EMS_FALSE ) {
		errReport("SWAP: image data does not exist!");
		return;
	}

	if ( (dimX%2)!=0 || (dimY%2)!=0 ) {
		errReport("SWAP: image size has to be even!");
		return;
	}

	// this function clears FFT component
	if ( rel != NULL ) {
		delete [] rel;
		rel = NULL;
	}

	if ( img != NULL ) {
		delete [] img;
		img = NULL;
	}

	fftX = fftY = 0;
	fftSize = 0;

	// swap image quadrants
	if ( dataC != NULL ) {
		for ( y=0; y<(dimY/2); y++ ) {
			for ( x=0; x<dimX; x++ ) {
				// swap quadrants
				xx = (x + dimX/2) % dimX;
				yy = (y + dimY/2) % dimY;
				index1 = y * dimX + x;
				index2 = yy * dimX + xx;
				valC = dataC[index1];
				dataC[index1] = dataC[index2];
				dataC[index2] = valC;
			}
		}
	}

	else if ( dataF != NULL ) {
		for ( y=0; y<(dimY/2); y++ ) {
			for ( x=0; x<dimX; x++ ) {
				// swap quadrants
				xx = (x + dimX/2) % dimX;
				yy = (y + dimY/2) % dimY;
				index1 = y * dimX + x;
				index2 = yy * dimX + xx;
				valF = dataF[index1];
				dataF[index1] = dataF[index2];
				dataF[index2] = valF;
			}
		}
	}

	else {
		errReport("SWAP: incorrect image data!");
		return;
	}
}


void localAVE(EMS_MAP &kernel, EMS_MAP &image, EMS_MAP &result, float scaling)
{
	int index, dimen[2];
	long dsize;
	float *cr, *ci, *dst, ww;

	dimen[0] = image.dimX;
	dimen[1] = image.dimY;

	// safeguarding data input
	if ( (kernel.exist==EMS_FALSE) || (image.exist==EMS_FALSE) ) {
		errReport("AVE: incomplete image data input!");
		return;
	}

	if ( image.fftSize <= 0 ) {
		image.imageFFT(dimen[0], dimen[1]);
	}

	if ( kernel.fftSize <= 0 ) {
		kernel.kernelFFT(dimen[0], dimen[1]);
	}

	if ( (kernel.fftX!=image.fftX) || (kernel.fftY!=image.fftY) ) {
		errReport("AVE: incompatible map dimensions!");
		return;
	}

	dimen[0] = image.fftX;
	dimen[1] = image.fftY;
	dsize = dimen[0] * dimen[1];
	ww = (float)(sqrt(dsize));

	cr = new float[dsize];	// real part of []x[]
	ci = new float[dsize];	// imag part of []x[]
	dst = new float[dsize];	// result array

	for ( index=0; index<dsize; index++ ) {
		cr[index] = ww * (kernel.rel[index] * image.rel[index] + kernel.img[index] * image.img[index]);
		ci[index] = ww * (kernel.rel[index] * image.img[index] - kernel.img[index] * image.rel[index]);
	}

	fftwN(2, dimen, cr, ci, FFTW_BACKWARD);

	for ( index=0; index<dsize; index++ )
		dst[index] = scaling * cr[index];

	result.setup(dst, dimen[0], dimen[1], 0, 0);

	delete [] cr;
	delete [] ci;
	delete [] dst;
}


void localSTD(EMS_MAP &kernel, EMS_MAP &image, EMS_MAP &result, float scaling)
{
	int index, dimen[2];
	long dsize;
	float variance, *dst;
	EMS_MAP ave_t, img2_t, var_t;

	dimen[0] = image.dimX;
	dimen[1] = image.dimY;

	// safeguarding data input
	if ( (kernel.exist==EMS_FALSE) || (image.exist==EMS_FALSE) ) {
		errReport("STD: incomplete image data input!");
		return;
	}

	if ( image.fftSize <= 0 ) {
		image.imageFFT(dimen[0], dimen[1]);
	}

	if ( kernel.fftSize <= 0 ) {
		kernel.kernelFFT(dimen[0], dimen[1]);
	}

	if ( (kernel.fftX!=image.fftX) || (kernel.fftY!=image.fftY) ) {
		errReport("STD: incompatible map dimensions!");
		return;
	}

	dimen[0] = image.fftX;
	dimen[1] = image.fftY;
	dsize = dimen[0] * dimen[1];

	dst = new float[dsize];

	localAVE(kernel, image, ave_t, scaling);
	multiplyMap(image, image, img2_t);
	localAVE(kernel, img2_t, var_t, scaling);

	for ( index=0; index<dsize; index++ ) {
		variance = var_t.dataF[index] - ave_t.dataF[index] * ave_t.dataF[index];

		if ( variance > 0.0 )
			dst[index] = (float)(sqrt(variance));
		else
			dst[index] = 0.0f;
	}

	result.setup(dst, dimen[0], dimen[1], 0, 0);

	delete [] dst;
}


void localCOR(EMS_MAP &kernel, EMS_MAP &image, EMS_MAP &stdv, EMS_MAP &result, float scaling)
{
	int index, dimen[2];
	long dsize;
	float *cr, *ci, *dst, ww;

	dimen[0] = image.dimX;
	dimen[1] = image.dimY;

	// safeguarding data input
	if ( (kernel.exist==EMS_FALSE) || (image.exist==EMS_FALSE) || (stdv.exist==EMS_FALSE) ) {
		errReport("COR: incomplete image data input!");
		return;
	}

	if ( image.fftSize <= 0 ) {
		image.imageFFT(dimen[0], dimen[1]);
	}

	if ( kernel.fftSize <= 0 ) {
		kernel.kernelFFT(dimen[0], dimen[1]);
	}

	if ( (kernel.fftX!=image.fftX) || (kernel.fftY!=image.fftY) ||
		 (stdv.dimX!=image.fftX) || (stdv.dimY!=image.fftY) ) {
		errReport("COR: incompatible map dimensions!");
		return;
	}

	dimen[0] = image.fftX;
	dimen[1] = image.fftY;
	dsize = dimen[0] * dimen[1];
	ww = (float)(sqrt(dsize));

	cr = new float[dsize];	// real part of []x[]
	ci = new float[dsize];	// imag part of []x[]
	dst = new float[dsize];	// result array

	for ( index=0; index<dsize; index++ ) {
		cr[index] = ww * (kernel.rel[index] * image.rel[index] + kernel.img[index] * image.img[index]);
		ci[index] = ww * (kernel.rel[index] * image.img[index] - kernel.img[index] * image.rel[index]);
	}

	fftwN(2, dimen, cr, ci, FFTW_BACKWARD);

	for ( index=0; index<dsize; index++ ) {
		if ( fabs(stdv.dataF[index]) >= EMS_ZERO )
			dst[index] = scaling * cr[index] / stdv.dataF[index];
		else
			dst[index] = 0.0;
	}

	result.setup(dst, dimen[0], dimen[1], 0, 0);

	delete [] cr;
	delete [] ci;
	delete [] dst;
}


void multiplyMap(EMS_MAP &src1, EMS_MAP &src2, EMS_MAP &dst)
{
	long index;
	float *ff;

	if ( src1.exist == EMS_FALSE ) return;
	if ( src2.exist == EMS_FALSE ) return;
	if ( src1.dataF == NULL ) return;
	if ( src2.dataF == NULL ) return;

	if ( src1.mapSize != src2.mapSize ) {
		errReport("Incompatible map size!");
		return;
	}

	ff = new float[src1.mapSize];

	for ( index=0; index<src1.mapSize; index++ )
		ff[index] = src1.dataF[index] * src2.dataF[index];

	dst.setup(ff, src1.dimX, src1.dimY, 0, 0);

	delete [] ff;
}


void divideMap(EMS_MAP &src1, EMS_MAP &src2, EMS_MAP &dst)
{
	long index;
	float *ff;

	if ( src1.exist == EMS_FALSE ) return;
	if ( src2.exist == EMS_FALSE ) return;
	if ( src1.dataF == NULL ) return;
	if ( src2.dataF == NULL ) return;

	if ( src1.mapSize != src2.mapSize ) {
		errReport("Incompatible map size!");
		return;
	}

	ff = new float[src1.mapSize];

	for ( index=0; index<src1.mapSize; index++ ) {
		if ( fabs(src2.dataF[index]) >= EMS_ZERO )
			ff[index] = src1.dataF[index] / src2.dataF[index];
		else
			ff[index] = 0.0;
	}

	dst.setup(ff, src1.dimX, src1.dimY, 0, 0);

	delete [] ff;
}


void smoothSpectrum(EMS_MAP &src, EMS_MAP &dst, int box)
{
	float scaling, valSwap;
	long index, index_t;
	int x, y, x0, y0, x1, y1, xt, yt, boxSize;
	double xf, yf, dA, dB, dC, dD, dE, dF;
	EMS_MAP kernel, aveImg;

	boxSize = 3 * box;
	scaling = 1.0f / (box * box);
	kernel.initC(EMS_FALSE, boxSize, boxSize);

	// set up the filter kernel
	for ( y=box; y<(2*box); y++ ) {
		for ( x=box; x<(2*box); x++ ) {
			kernel.dataC[y*boxSize+x] = EMS_TRUE;
		}
	}

	// swapping spectrum quadrants
	for ( y=0; y<(src.dimY/2); y++ ) {
		yt = (y + src.dimY/2) % src.dimY;
		for ( x=0; x<src.dimX; x++ ) {
			xt = (x + src.dimX/2) % src.dimX;

			index = y * src.dimX + x;
			index_t = yt * src.dimX + xt;

			valSwap = src.dataF[index];
			src.dataF[index] = src.dataF[index_t];
			src.dataF[index_t] = valSwap;
		}
	}

	localAVE(kernel, src, aveImg, scaling);

	// interpolate image data
	for ( y=0; y<dst.dimY; y++ ) {
		yf = 1.0 * y * (src.dimY-1) / (dst.dimY-1);
		y0 = (int)floor(yf); y1 = y0 + 1;
		if ( y1 >= src.dimY ) y1 = src.dimY - 1;
		if ( y0 == y1 ) y0 = src.dimY - 2;
		
		for ( x=0; x<dst.dimX; x++ ) {
			xf = 1.0 * x * (src.dimX-1) / (dst.dimX-1);
			x0 = (int)floor(xf); x1 = x0 + 1;
			if ( x1 >= src.dimX ) x1 = src.dimX - 1;
			if ( x0 == x1 ) x0 = src.dimX - 2;
			
			dA = aveImg.dataF[y0*src.dimX+x0];
			dB = aveImg.dataF[y0*src.dimX+x1];
			dC = aveImg.dataF[y1*src.dimX+x0];
			dD = aveImg.dataF[y1*src.dimX+x1];
			dE = (x1-xf) * dA + (xf-x0) * dB;
			dF = (x1-xf) * dC + (xf-x0) * dD;
			dst.dataF[y*dst.dimX+x] = float((y1-yf) * dE + (yf-y0) * dF);
			
		}
	}
}


void smoothMap(EMS_MAP &src, int box)
{
	float scaling;
	int x, y, boxSize;
	EMS_MAP kernel, aveImg;

	boxSize = 3 * box;
	scaling = 1.0f / (box * box);
	kernel.initC(EMS_FALSE, boxSize, boxSize);

	// set up the filter kernel
	for ( y=box; y<(2*box); y++ ) {
		for ( x=box; x<(2*box); x++ ) {
			kernel.dataC[y*boxSize+x] = EMS_TRUE;
		}
	}

	localAVE(kernel, src, aveImg, scaling);
	copyMap(src, aveImg);
}


void copyMap(EMS_MAP &dst, EMS_MAP &src)
{
	if ( src.exist == EMS_FALSE ) return;

	if ( src.dataC != NULL )
		dst.setup(src.dataC, src.dimX, src.dimY, 0, 0);
	else if ( src.dataF != NULL )
		dst.setup(src.dataF, src.dimX, src.dimY, 0, 0);
/*
	if ( (src.fftSize>0) && (src.fftSize==dst.fftSize) ) {
		memcpy(dst.rel, src.rel, src.fftSize * sizeof(float));
		memcpy(dst.img, src.img, src.fftSize * sizeof(float));
	}
*/
}
