//	James Z. Chen
//	January 1, 2005
//	All rights reserved

#include "interact.h"
#include "../signature/include/FL/Fl_File_Chooser.H"
#include "../signature/include/FL/Fl_Color_Chooser.H"
#include "../signature/include/FL/fl_draw.H"
#include "../signature/include/FL/fl_ask.H"
#include "panel.h"
#include "images.h"
#include "misc.h"
#include "gui.h"
#include "logo.h"


extern DisplayPanel_UI *EMS_DP;
extern StartUp_UI *LoginWindow;


// Login panel and CP modes

void EMS_Login_CB(char flagLogin)
{
	LoginWindow->hide();
	delete LoginWindow;

	EMS_DP->show();
	project->resetParam();
	functImage->resetParam();
	EMS_DP->indicator->deactivate();
	msgReport("Welcome to SIGNATURE");
}


void setDPMode_CB(int mode)
{
	msgReport("\0");

	if ( project->sysStatus.emsdpMode == mode ) {
		EMS_DP->imgDisplay->redraw();
		return;
	}

	systemStatus(EMS_BUSY);
	project->sysStatus.emsdpMode = mode;

	// update the film mask when changing DP mode
	if ( project->sysStatus.drawTools != EMS_FALSE ) {
		EMS_DP->btnLines->value(0);
		EMS_DP->btnCircle->value(0);
		EMS_DP->btnPaint->value(0);
		EMS_DP->btnErase->value(0);
		project->sysStatus.syncMask = EMS_FALSE;
		project->sysStatus.drawTools = EMS_FALSE;
		EMS_DP->imgDisplay->setMaskTexture(project->dispMask);
	}

	EMS_DP->btnMcrgraph->value(0);
	EMS_DP->btnTemplate->value(0);
	EMS_DP->btnParticle->value(0);

	EMS_DP->grpMcrgraph->hide();
	EMS_DP->grpTemplate->hide();
	EMS_DP->grpParticle->hide();
	EMS_DP->grp2DImage->hide();
	EMS_DP->functImages->hide();
	EMS_DP->functDigitGel->hide();

	// ToolBar buttons works only at appropriate page(s)
	EMS_DP->btnShowMask->value(0);
	EMS_DP->btnShowPrtk->value(0);
	EMS_DP->btnShowZoom->value(0);
	project->sysStatus.showZoom = EMS_FALSE;

	switch ( mode ) {

	case DP_MCRGRAPH_MODE:
		EMS_DP->btnMcrgraph->value(1);
		EMS_DP->grpMcrgraph->show();
		EMS_DP->imgDisplay->clearFocus();
		EMS_DP->imgDisplay->resetTransform(EMS_FALSE);
		EMS_DP->sldBrit->value(project->parameter.flmBrit);
		EMS_DP->sldCntr->value(project->parameter.flmCntr);

		if ( project->sysStatus.showMask == EMS_TRUE )
			EMS_DP->btnShowMask->value(1);

		EMS_DP->imgDisplay->setGraphicsMode(EMS_GRAPHX_2D);
		break;

	case DP_TEMPLATE_MODE:
		EMS_DP->btnTemplate->value(1);
		EMS_DP->grpTemplate->show();
		EMS_DP->imgDisplay->resetTransform(EMS_TRUE);
		EMS_DP->sldBrit->value(project->parameter.refBrit);
		EMS_DP->sldCntr->value(project->parameter.refCntr);

		if ( project->sysStatus.showMask == EMS_TRUE )
			EMS_DP->btnShowMask->value(1);
		if ( project->sysStatus.showPrtk == EMS_TRUE )
			EMS_DP->btnShowPrtk->value(1);

		EMS_DP->imgDisplay->setGraphicsMode(EMS_GRAPHX_2D);
		break;

	case DP_PARTICLE_MODE:
		EMS_DP->btnParticle->value(1);
		EMS_DP->grpParticle->show();
		EMS_DP->imgDisplay->clearFocus();
		EMS_DP->imgDisplay->resetTransform(EMS_FALSE);
		EMS_DP->sldBrit->value(project->parameter.flmBrit);
		EMS_DP->sldCntr->value(project->parameter.flmCntr);

		if ( project->sysStatus.showMask == EMS_TRUE )
			EMS_DP->btnShowMask->value(1);
		if ( project->sysStatus.showPrtk == EMS_TRUE )
			EMS_DP->btnShowPrtk->value(1);

		EMS_DP->imgDisplay->setGraphicsMode(EMS_GRAPHX_2D);
		break;

	case DP_2DMIMAGE_MODE:
		EMS_DP->grp2DImage->show();
		EMS_DP->imgDisplay->resetTransform(EMS_FALSE);
		EMS_DP->sldBrit->value(project->parameter.flmBrit);
		EMS_DP->sldCntr->value(project->parameter.flmCntr);

		if ( project->sysStatus.showMask == EMS_TRUE )
			EMS_DP->btnShowMask->value(1);
		if ( project->sysStatus.showPrtk == EMS_TRUE )
			EMS_DP->btnShowPrtk->value(1);

		EMS_DP->imgDisplay->setGraphicsMode(EMS_GRAPHX_2D);
		break;

	case DP_FUNCIMAG_MODE:
		EMS_DP->functImages->show();
		EMS_DP->imgDisplay->clearFocus();
		EMS_DP->imgDisplay->resetTransform(EMS_TRUE);
		EMS_DP->sldBrit->value(functImage->imgBrit);
		EMS_DP->sldCntr->value(functImage->imgCntr);

		EMS_DP->imgDisplay->setGraphicsMode(EMS_GRAPHX_2D);
		break;

	case DP_FUNCDGEL_MODE:
		EMS_DP->functDigitGel->show();
		EMS_DP->imgDisplay->resetTransform(EMS_TRUE);

		EMS_DP->imgDisplay->setGraphicsMode(EMS_GRAPHX_2D);
		break;
	}

	EMS_DP->imgDisplay->redraw();
	systemStatus(EMS_FREE);
}

// menubar functions

void EMS_Exit_CB(void)
{
	askSaveProj();
	project->clearProject();
	closeImageStack_CB();
	EMS_DP->DisplayWindow->hide();

	if ( EMS_DP != NULL ) delete EMS_DP;
	if ( cPanel != NULL ) delete cPanel;
	if ( project != NULL ) delete project;
	if ( functImage != NULL ) delete functImage;

	exit(0);
}


void newProjSpace_CB(void)
{
	char *newProj, projName[NAMESIZE];

	clearProject_CB();
	msgReport("Setup a new project space");

	newProj = fl_file_chooser("New project space ...", "*.ems", NULL, 0);
	if ( newProj == NULL ) return;

	strcpy(projName, newProj);
	project->newProjSpace(projName);

	EMS_DP->dspProj->value(project->projName);
	EMS_DP->dspFilm->value("\0");
//	project->projSaveStatus = EMS_FALSE;
}


void loadProjSpace_CB(void)
{
	char *newProj, projName[NAMESIZE];

	clearProject_CB();
	msgReport("Open a project space");

	newProj = fl_file_chooser("Open project space ...", "*.sys", NULL, 0);
	if ( newProj == NULL ) return;

	strcpy(projName, newProj);
	project->loadProjSpace(projName);

	EMS_DP->dspProj->value(project->projName);
	EMS_DP->dspFilm->value("\0");
}


void loadProject_CB(void)
{
	char *fileName;

	msgReport("Load a screening project");
	askSaveProj();

	fileName = fl_file_chooser("Loading screening project ...", "*.prj", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_TRUE);
	project->clearProject();

	if ( project->loadProject(fileName) ) {
		cPanel->displayParam();
		EMS_DP->dspProj->value(project->projName);
		EMS_DP->dspFilm->value(project->filmName);
		
		setDPMode_CB(DP_MCRGRAPH_MODE);
		project->projSaveStatus = EMS_TRUE;
		msgReport("Screening project has been loaded");
	}

	systemStatus(EMS_FREE);
}


void saveProject_CB(void)
{
	msgReport("Save a screening project");

	if ( (project->micrograph.exist==EMS_FALSE) || (project->templates.exist==EMS_FALSE) ) {
		errReport("Please specify micrograph and template images!");
		return;
	}

	systemStatus(EMS_BUSY);

	if ( strcmp(project->projName, EMS_PRJ_FN) ) {
		project->saveProject(project->filmName);
		project->projSaveStatus = EMS_TRUE;
		msgReport("Screening project has been saved");
	}
	else
		saveProjectAs_CB();

	systemStatus(EMS_FREE);
}


void saveProjectAs_CB(void)
{
	char *fileName, projName[NAMESIZE];
	int nameSize;

	msgReport("Save a screening project");
	if ( (project->micrograph.exist==EMS_FALSE) || (project->templates.exist==EMS_FALSE) ) {
		errReport("Please specify micrograph and template data first!");
		return;
	}

	fileName = fl_file_chooser("Saving project ...", "*.prj", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);

	// attach ".prj" extension if necessary
	nameSize = strlen(fileName);
	if ( strncmp(fileName+nameSize-4, ".prj", 4) )
		sprintf(projName, "%s.prj", fileName);
	else
		sprintf(projName, "%s", fileName);

	project->saveProject(projName);
	project->projSaveStatus = EMS_TRUE;

	systemStatus(EMS_FREE);
}


void clearProject_CB(void)
{
	askSaveProj();
	project->clearProject();
	closeImageStack_CB();
	setDPMode_CB(DP_MCRGRAPH_MODE);
	msgReport("Ready for a new project");
}


void saveParticlePTK_CB(void)
{
	char fileName[NAMESIZE];

	msgReport("Save particle coordinates in PTK format");

	if ( project->parameter.totCount <= 0 ) {
		errReport("There is no particle data to save!");
		return;
	}
	else {
		sprintf(fileName, "%s/%s.ptk", project->dirParticle, project->filmName);
	}

	systemStatus(EMS_BUSY);
	project->saveParticlePTK(fileName);
	msgReport("Particle selection has been saved in PTK format");
	systemStatus(EMS_FREE);
}


void saveParticleMRC_CB(void)
{
	char *fileName;

	msgReport("Save particle images to an MRC stack");

	if ( project->parameter.totCount <= 0 ) {
		errReport("There is no particle data to save!");
		return;
	}

	systemStatus(EMS_BUSY);

	fileName = fl_file_chooser("Saving particle stack as ...", "*.mrc", NULL, 0);
	if ( fileName == NULL ) return;

	if ( project->saveParticleMRC(fileName) )
		msgReport("Particle images have been saved to an MRC stack");

	systemStatus(EMS_FREE);
}


void saveParticlePLT_CB(void)
{
	char fileName[NAMESIZE];

	msgReport("Save particle coordinates in PLT format");

	if ( project->parameter.totCount <= 0 ) {
		errReport("There is no particle data to save!");
		return;
	}
	else {
		sprintf(fileName, "%s/%s.plt", project->dirParticle, project->filmName);
	}

	systemStatus(EMS_BUSY);
	project->saveParticlePLT(fileName);
	msgReport("Particle selection has been saved in PLT format");
	systemStatus(EMS_FREE);
}


void saveParticleSPD_CB(void)
{
	char fileName[NAMESIZE];

	msgReport("Save particle coordinates in SPD format");

	if ( project->parameter.totCount <= 0 ) {
		errReport("There is no particle data to save!");
		return;
	}
	else {
		sprintf(fileName, "%s/%s.spd", project->dirParticle, project->filmName);
	}

	systemStatus(EMS_BUSY);
	project->saveParticleSPD(fileName);
	msgReport("Particle selection has been saved in SPD format");
	systemStatus(EMS_FREE);
}


void clearParticle_CB(void)
{
	msgReport("Clear particle markers");
	if ( project->parameter.totCount <= 0 ) return;

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Clear all particle selections?") == 0 ) return;

	project->clearParticle();
	setDPMode_CB(DP_PARTICLE_MODE);

	project->projSaveStatus = EMS_FALSE;
}


void loadParticlePTK_CB(void)
{
	char *fileName;

	msgReport("Load PTK particle coordinates");

	if ( project->micrograph.exist == EMS_FALSE ) {
		errReport("Please load EM micrograph first!");
		return;
	}

	fileName = fl_file_chooser("Loading particle selection ...", "*.ptk", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);

	if ( project->loadParticlePTK(fileName) ) {
		EMS_DP->dspCount->value(project->parameter.totCount);
		setDPMode_CB(DP_PARTICLE_MODE);
		msgReport("PTK particle selection has been successfully loaded");
	}

	systemStatus(EMS_FREE);
}


void loadParticlePLT_CB(void)
{
	char *fileName;

	msgReport("Load PLT particle coordinates");

	if ( project->micrograph.exist == EMS_FALSE ) {
		errReport("Please load EM micrograph first!");
		return;
	}

	fileName = fl_file_chooser("Loading particle selection ...", "*.plt", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);

	if ( project->loadParticlePLT(fileName) ) {
		EMS_DP->dspCount->value(project->parameter.totCount);
		setDPMode_CB(DP_PARTICLE_MODE);
		msgReport("PLT particle selection has been successfully loaded");
	}

	systemStatus(EMS_FREE);
}


void loadParticleSPD_CB(void)
{
	char *fileName;

	msgReport("Load SPD particle coordinates");

	if ( project->micrograph.exist == EMS_FALSE ) {
		errReport("Please load EM micrograph first!");
		return;
	}

	fileName = fl_file_chooser("Loading particle selection ...", "*.spd", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);

	if ( project->loadParticleSPD(fileName) ) {
		EMS_DP->dspCount->value(project->parameter.totCount);
		setDPMode_CB(DP_PARTICLE_MODE);
		msgReport("SPD particle selection has been successfully loaded");
	}

	systemStatus(EMS_FREE);
}


void loadMcrgraph_CB(void)
{
	char *fileName;

	msgReport("Load MRC micrograph image");

	fileName = fl_file_chooser("Loading micrograph data ...", "*.mrc", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);
	project->clearMicrograph();

	if ( project->loadMicrograph(fileName) ) {
		simpleFilePath(fileName);
		EMS_DP->dspFilmFile->value(fileName);
		EMS_DP->dspFilmX->value(project->micrograph.dimX);
		EMS_DP->dspFilmY->value(project->micrograph.dimY);
		EMS_DP->dspPixSize->value(project->parameter.pixelSize);
		EMS_DP->dspFilm->value(project->filmName);
		
		setDPMode_CB(DP_MCRGRAPH_MODE);
		project->projSaveStatus = EMS_FALSE;
	}

	// reset keynote
	EMS_DP->dspKeyNote->value("\0");

	systemStatus(EMS_FREE);
}


void loadMcrgraphTIF_CB(void)
{
	char *fileName;

	msgReport("Load TIFF micrograph image");

	fileName = fl_file_chooser("Loading micrograph data ...", "*.tif", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);
	project->clearMicrograph();

	if ( project->loadMcrgraphTIF(fileName) ) {
		simpleFilePath(fileName);
		EMS_DP->dspFilmFile->value(fileName);
		EMS_DP->dspFilmX->value(project->micrograph.dimX);
		EMS_DP->dspFilmY->value(project->micrograph.dimY);
		EMS_DP->dspPixSize->value(project->parameter.pixelSize);
		EMS_DP->dspFilm->value(project->filmName);
		
		setDPMode_CB(DP_MCRGRAPH_MODE);
		project->projSaveStatus = EMS_FALSE;
	}

	// reset keynote
	EMS_DP->dspKeyNote->value("\0");

	systemStatus(EMS_FREE);
}


void loadMcrgraphSPD_CB(void)
{
	char *fileName;

	msgReport("Load SPD micrograph image");

	fileName = fl_file_chooser("Loading micrograph data ...", "*.spd", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);
	project->clearMicrograph();

	if ( project->loadMcrgraphSPD(fileName) ) {
		simpleFilePath(fileName);
		EMS_DP->dspFilmFile->value(fileName);
		EMS_DP->dspFilmX->value(project->micrograph.dimX);
		EMS_DP->dspFilmY->value(project->micrograph.dimY);
		EMS_DP->dspPixSize->value(project->parameter.pixelSize);
		EMS_DP->dspFilm->value(project->filmName);
		
		setDPMode_CB(DP_MCRGRAPH_MODE);
		project->projSaveStatus = EMS_FALSE;
	}

	// reset keynote
	EMS_DP->dspKeyNote->value("\0");

	systemStatus(EMS_FREE);
}


void saveMcrgraph_CB(void)
{
	char *fileName;

	msgReport("Save an MRC 2D image");

	if ( project->micrograph.exist == EMS_FALSE ) {
		errReport("There is no image data to save!");
		return;
	}

	fileName = fl_file_chooser("Saving MRC image ...", "*.mrc", NULL, 0);
	if ( fileName == NULL ) return;

	if ( project->saveMicrograph(fileName) ) {
		simpleFilePath(fileName);
		EMS_DP->dspFilmFile->value(fileName);
		EMS_DP->dspFilmX->value(project->micrograph.dimX);
		EMS_DP->dspFilmY->value(project->micrograph.dimY);
		EMS_DP->dspPixSize->value(project->parameter.pixelSize);
		EMS_DP->dspFilm->value(project->filmName);
	}

	project->projSaveStatus = EMS_FALSE;
}


void clearMcrgraph_CB(void)
{
	msgReport("\0");

	project->clearMicrograph();
	EMS_DP->dspFilmFile->value("\0");
	EMS_DP->dspFilmX->value(0);
	EMS_DP->dspFilmY->value(0);
	EMS_DP->dspPixSize->value(EMS_PIXEL_SIZE);
	setDPMode_CB(DP_MCRGRAPH_MODE);
}


void loadTemplate_CB(void)
{
	char *fileName;

	msgReport("Load MRC template images");

	fileName = fl_file_chooser("Loading template data ...", "*.mrc", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);

	project->clearTemplate();
	project->loadTemplate(fileName);

	simpleFilePath(fileName);
	EMS_DP->dspTempFile->value(fileName);
	EMS_DP->dspTempN->value(project->templates.dimZ);
	EMS_DP->dspTempX->value(project->templates.dimX);
	EMS_DP->dspTempY->value(project->templates.dimY);

	// reset Inc/Exc flag
	EMS_DP->btnTempInc->value(1);
	EMS_DP->btnTempInc->label("inc");

	setDPMode_CB(DP_TEMPLATE_MODE);
	project->projSaveStatus = EMS_FALSE;

	systemStatus(EMS_FREE);
}


void clearTemplate_CB(void)
{
	msgReport("\0");

	project->clearTemplate();
	EMS_DP->dspTempFile->value("\0");
	EMS_DP->dspTempN->value(0);
	EMS_DP->dspTempX->value(0);
	EMS_DP->dspTempY->value(0);
	setDPMode_CB(DP_TEMPLATE_MODE);
}


void saveImageCrop_CB(void)
{
	char *fileName;

	msgReport("Save an image crop (MRC)");

	if ( project->imgCrop.exist == EMS_FALSE ) {
		errReport("There is no image data to be saved!");
		return;
	}

	fileName = fl_file_chooser("Saving image (MRC) ...", "*.mrc", NULL, 0);

	if ( fileName != NULL )
		project->writeMap2D(fileName, project->imgCrop);
}


void EMS_Docum_CB(void)
{
	msgReport("\0");

	sprintf(message, "%s%s",
		"The user manual is online at \n\n",
		"       www.brandeis.edu/~jzchen/Signature");

	fl_message(message);
}


void EMS_About_CB(void)
{
	msgReport("\0");

	sprintf(message, "%s%s%s%s",
		"SIGNATURE is a particle analysis system for \n",
		"molecular EM. Questions & comments? \n\n",
		"               James  Z.  Chen\n",
		"       james.z.chen  at  gmail.com");

	fl_message(message);
}

// MICROGRAPH page

void setPrtkSize_CB(double value)
{
	project->parameter.prtkSize = (float)value;
//	project->sysStatus.mapRestore = EMS_FALSE;
	project->projSaveStatus = EMS_FALSE;
}


void setPixSize_CB(double value)
{
	project->parameter.pixelSize = (float)value;
//	project->sysStatus.mapRestore = EMS_FALSE;
	project->projSaveStatus = EMS_FALSE;
}


void setDownSize_CB(int value)
{
	if ( project->micrograph.exist == EMS_FALSE ) {
		project->parameter.downSize = EMS_DOWNSIZE;
		EMS_DP->dspDownSize->value(EMS_DOWNSIZE);
		return;
	}
	
	project->parameter.downSize = value;
	EMS_DP->dspDownSize->value(value);
	
	project->parameter.totCount = 0;
	EMS_DP->dspCount->value(0);
	
	project->sysStatus.newScale = EMS_TRUE;
	project->sysStatus.mapRestore = EMS_FALSE;
	project->projSaveStatus = EMS_FALSE;
}


void setVariaBox_CB(int value)
{
	project->parameter.variaBox = value;
	project->stdMap.exist = EMS_FALSE;
}


void setImgBright_CB(float value)
{
	msgReport("Adjust image brightness");
	systemStatus(EMS_BUSY);

	if ( (project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE) ||
		 (project->sysStatus.emsdpMode == DP_PARTICLE_MODE) ||
		 (project->sysStatus.emsdpMode == DP_2DMIMAGE_MODE) ) {
		
		project->parameter.flmBrit = value;
		if ( project->micrograph.exist == EMS_TRUE )
			EMS_DP->imgDisplay->setFilmTexture(project->dispFilm);
	}

	else if ( project->sysStatus.emsdpMode == DP_TEMPLATE_MODE ) {
		project->parameter.refBrit = value;
		if ( project->templates.exist == EMS_TRUE ) {
			if ( project->sysStatus.editTemp == EMS_TRUE )
				EMS_DP->imgDisplay->setRefrTexture();
			else
				EMS_DP->imgDisplay->setTempTexture();
		}
	}

	else if ( project->sysStatus.emsdpMode == DP_FUNCIMAG_MODE ) {
		functImage->imgBrit = value;
		if ( functImage->imgStack.exist == EMS_TRUE )
			EMS_DP->imgDisplay->setStackTexture();
	}

	project->projSaveStatus = EMS_FALSE;
	systemStatus(EMS_FREE);
}


void setImgContrast_CB(float value)
{
	msgReport("Adjust image contrast");
	systemStatus(EMS_BUSY);

	if ( (project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE) ||
		 (project->sysStatus.emsdpMode == DP_PARTICLE_MODE) ||
		 (project->sysStatus.emsdpMode == DP_2DMIMAGE_MODE) ) {

		project->parameter.flmCntr = value;
		if ( project->micrograph.exist == EMS_TRUE )
			EMS_DP->imgDisplay->setFilmTexture(project->dispFilm);
	}

	else if ( project->sysStatus.emsdpMode == DP_TEMPLATE_MODE ) {
		project->parameter.refCntr = value;
		if ( project->templates.exist == EMS_TRUE ) {
			if ( project->sysStatus.editTemp == EMS_TRUE )
				EMS_DP->imgDisplay->setRefrTexture();
			else
				EMS_DP->imgDisplay->setTempTexture();
		}
	}

	else if ( project->sysStatus.emsdpMode == DP_FUNCIMAG_MODE ) {
		functImage->imgCntr = value;
		if ( functImage->imgStack.exist == EMS_TRUE )
			EMS_DP->imgDisplay->setStackTexture();
	}

	project->projSaveStatus = EMS_FALSE;
	systemStatus(EMS_FREE);
}


void setSpecBright_CB(float value)
{
	msgReport("\0");
	project->parameter.spcBrit = value;

	systemStatus(EMS_BUSY);

	if ( project->micrograph.exist == EMS_TRUE )
		cPanel->drawImgInfo(0);

	systemStatus(EMS_FREE);
}


void setSpecContrast_CB(float value)
{
	msgReport("\0");
	project->parameter.spcCntr = value;

	systemStatus(EMS_BUSY);

	if ( project->micrograph.exist == EMS_TRUE )
		cPanel->drawImgInfo(0);

	systemStatus(EMS_FREE);
}


void setImgInfoMode_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnSpecMode->value() ) {
		EMS_DP->imgDisplay->clearFocus();
		project->sysStatus.imgInform = EMS_SPEC_MODE;
	}
	else
		project->sysStatus.imgInform = EMS_HSTG_MODE;

	systemStatus(EMS_BUSY);

	if ( project->micrograph.exist == EMS_TRUE ) {
		cPanel->drawImgInfo(0);
		EMS_DP->imgDisplay->redraw();
	}

	systemStatus(EMS_FREE);
}


void imageRestore_CB(void)
{
	msgReport("\0");

	if ( !strcmp(project->dirParticle, EMS_PRJ_DR) ) {
		errReport("Please set up a project space first!");
		return;
	}

	if ( project->micrograph.exist == EMS_FALSE ) {
		errReport("Please load micrograph data first!");
		return;
	}

	systemStatus(EMS_BUSY);

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Restore the original micrograph image?") ) {

		project->restoreOriginal();
		project->setDispFilm();
		EMS_DP->imgDisplay->setFilmTexture(project->dispFilm);
		cPanel->drawImgInfo(0);

		project->sysStatus.newScale = EMS_TRUE;
		project->sysStatus.mapRestore = EMS_FALSE;
		project->projSaveStatus = EMS_FALSE;
	}

	systemStatus(EMS_FREE);
}

// TEMPLATE page

void focusTemplate_CB(int tid)
{
	msgReport("\0");

	if ( project->templates.exist == EMS_FALSE ) {
		errReport("Please load template data first!");
		return;
	}

	if ( tid < 1 ) {
		tid = 1;
		errReport("Template ID is out of range! Reset to 1.");
	}
	else if ( tid > project->parameter.tmpN ) {
		tid = project->parameter.tmpN;
		sprintf(message, "Template ID is out of range! reset to %d.", tid);
		errReport(message);
	}

	project->focusTemplate(tid);
	EMS_DP->imgDisplay->redraw();
}


void setLCFmask_CB(float value)
{
	msgReport("Set LCF masking radius");

	project->parameter.lcfMaskRad = value;
	project->sysStatus.mapRestore = EMS_FALSE;
	EMS_DP->sldLCFrad->value(value);
	project->projSaveStatus = EMS_FALSE;
}


void setSCFmask_CB(float value)
{
	msgReport("Set SCF masking radius");

	project->parameter.scfMaskRad = value;
	project->sysStatus.mapRestore = EMS_FALSE;
	EMS_DP->sldSCFrad->value(value);
	project->projSaveStatus = EMS_FALSE;
}


void setSCFcent_CB(float value)
{
	msgReport("Set SCF center blocking radius");

	project->parameter.scfMaskRadc = value;
	project->sysStatus.mapRestore = EMS_FALSE;
	EMS_DP->sldSCFradc->value(value);
	project->projSaveStatus = EMS_FALSE;
}


void appProfileMask_CB(void)
{
	msgReport("\0");
	systemStatus(EMS_BUSY);

	if ( project->templates.exist == EMS_TRUE ) {
		project->profileMasking(project->refer.maskID);
		EMS_DP->imgDisplay->setTempTexture();
	}

	systemStatus(EMS_FREE);
}


void setMaskLevel_CB(char value)
{
	msgReport("Select masking level (L1 / L2)");
	systemStatus(EMS_BUSY);

	project->refer.maskID = value;
	EMS_DP->imgDisplay->setTempTexture();

	systemStatus(EMS_FREE);
}


void setL2Mask_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnL2Mask->value() )
		project->sysStatus.activeL2 = EMS_TRUE;
	else
		project->sysStatus.activeL2 = EMS_FALSE;
}


void setTempFrame_CB(double value)
{
	msgReport("\0");

	if ( project->templates.exist == EMS_FALSE ) {
		EMS_DP->btnEditFrame->value(0);
		return;
	}

	// convert from base-1 to base-0
	project->refer.editFrame = (int)(EMS_DP->valRefSelect->value()) - 1;

	if ( (project->refer.editFrame<0) || (project->refer.editFrame>=project->parameter.tmpN) ) {
		project->refer.editFrame = -1;
		errReport("The specified frame does not exist!");
		EMS_DP->btnEditFrame->value(0);
		EMS_DP->btnTempInc->value(1);
		EMS_DP->btnTempInc->label("inc");
		return;
	}

	if ( project->refer.select[project->refer.editFrame] ) {
		EMS_DP->btnTempInc->value(1);
		EMS_DP->btnTempInc->label("inc");
	}
	else {
		EMS_DP->btnTempInc->value(0);
		EMS_DP->btnTempInc->label("exc");
	}
}


void setTempInclude_CB(void)
{
	msgReport("\0");

	if ( project->templates.exist == EMS_FALSE ) {
		EMS_DP->btnTempInc->value(1);
		EMS_DP->btnTempInc->label("inc");
		return;
	}

	if ( (project->refer.editFrame<0) || (project->refer.editFrame>=project->parameter.tmpN) ) {
		project->refer.editFrame = -1;
		errReport("The specified frame does not exist!");
//		EMS_DP->btnEditFrame->value(0);
//		EMS_DP->btnTempInc->value(1);
//		EMS_DP->btnTempInc->label("inc");
		return;
	}

	if ( EMS_DP->btnTempInc->value() ) {
		project->refer.select[project->refer.editFrame] = EMS_TRUE;
		EMS_DP->btnTempInc->label("inc");
	}
	else {
		project->refer.select[project->refer.editFrame] = EMS_FALSE;
		EMS_DP->btnTempInc->label("exc");
	}
}


void setIncExcAll_CB(char status)
{
	msgReport("Set Inc/Exc flag for all template images");

	if ( project->templates.exist == EMS_FALSE ) {
		EMS_DP->btnTempInc->value(1);
		EMS_DP->btnTempInc->label("inc");
		return;
	}

	project->setIncExcState(status);

	if ( status == EMS_TRUE ) {
		EMS_DP->btnTempInc->value(1);
		EMS_DP->btnTempInc->label("inc");
	}
	else {
		EMS_DP->btnTempInc->value(0);
		EMS_DP->btnTempInc->label("exc");
	}
}


void editTempMask_CB(char status)
{
	msgReport("\0");

	if ( status == 1 ) {
		project->sysStatus.editTemp = EMS_TRUE;
		EMS_DP->imgDisplay->setRefrTexture();
	}
	else {
		EMS_DP->btnLines->value(0);
		EMS_DP->btnCircle->value(0);
		EMS_DP->btnPaint->value(0);
		EMS_DP->btnErase->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		EMS_DP->imgDisplay->setTempTexture();
		project->sysStatus.editTemp = EMS_FALSE;
		EMS_DP->valRefSelect->value(project->refer.editFrame+1);
	}

	EMS_DP->imgDisplay->redraw();
}


void syncTempMask_CB(void)
{
	msgReport("\0");

	if ( project->templates.exist == EMS_FALSE ) return;
	if ( project->sysStatus.editTemp == EMS_TRUE ) return;

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Synchronize template masking?") == 0 ) return;

	project->syncTempMask();
	EMS_DP->imgDisplay->setTempTexture();
	EMS_DP->imgDisplay->redraw();
}


void clearTempMask_CB(void)
{
	msgReport("\0");

	if ( project->templates.exist == EMS_FALSE ) return;
	if ( project->sysStatus.editTemp == EMS_TRUE ) return;

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Clear template masking?") == 0 ) return;

	project->clearTempMask();
	EMS_DP->imgDisplay->setTempTexture();
	EMS_DP->imgDisplay->redraw();
}


void setRotatFlag_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnRotate->value() )
		project->sysStatus.rotaScreen = EMS_TRUE;
	else
		project->sysStatus.rotaScreen = EMS_FALSE;
}


void setRotaRange_CB(float value)
{
	project->parameter.rotaRange = value;
	project->sysStatus.mapRestore = EMS_FALSE;
	project->projSaveStatus = EMS_FALSE;
}


void setRotaDelta_CB(float value)
{
	project->parameter.rotaDelta = value;
	project->sysStatus.mapRestore = EMS_FALSE;
	project->projSaveStatus = EMS_FALSE;
}


// MASKING page

void setEdgeExcu_CB(int value)
{
	project->parameter.edgeExcu = value;
	project->projSaveStatus = EMS_FALSE;
}


void setSpecSize_CB(int value)
{
	project->parameter.specSize = value;
	project->projSaveStatus = EMS_FALSE;
}


void setFilmMarker_CB(void)
{
	EMS_DP->imgDisplay->setFilmMarker();
}


void setVarHi_CB(float value)
{
	project->parameter.varHi = value;
	project->projSaveStatus = EMS_FALSE;
}


void setVarLo_CB(float value)
{
	project->parameter.varLo = value;
	project->projSaveStatus = EMS_FALSE;
}


void autoMasking_CB(void)
{
	msgReport("Apply automatic masking");
	systemStatus(EMS_BUSY);

	project->buildFilmMask();
	project->sysStatus.syncMask = EMS_FALSE;
	project->projSaveStatus = EMS_FALSE;

	systemStatus(EMS_FREE);
}


void setBrushSize_CB(int value)
{
	project->parameter.brushSize = value;
	EMS_DP->valBrush->value(value);
	EMS_DP->valBrush2->value(value);

	project->projSaveStatus = EMS_FALSE;
}


void circleMask_CB(void)
{
	char active_flag;

	msgReport("\0");

	// active only under MICROGRAPH / TEMPLATE / PARTICLE / FILAMENT modes
	switch ( project->sysStatus.emsdpMode ) {
	case DP_MCRGRAPH_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_TEMPLATE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_PARTICLE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_2DMIMAGE_MODE:
		active_flag = EMS_TRUE;
		break;
	default:
		active_flag = EMS_FALSE;
	}

	if ( active_flag == EMS_FALSE ) {
		EMS_DP->btnCircle->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	// exclusive with showZoom button
	if ( project->sysStatus.showZoom == EMS_TRUE ) {
		EMS_DP->btnCircle->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	// safe-guard data availability
	if ( project->dispMask.exist == EMS_FALSE ) {
		EMS_DP->btnCircle->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	if ( (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) && (project->sysStatus.editTemp==EMS_FALSE) ) {
		EMS_DP->btnCircle->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	systemStatus(EMS_BUSY);

	if ( EMS_DP->btnCircle->value() ) {
		project->sysStatus.drawTools = EMS_CIRCLE;
		EMS_DP->btnShowMask->value(1);
		project->sysStatus.showMask = EMS_TRUE;
		EMS_DP->imgDisplay->manuCircle(0, 0, 0);
		EMS_DP->imgDisplay->redraw();
	}
	else {
		EMS_DP->imgDisplay->maskCircle();
		project->sysStatus.drawTools = EMS_FALSE;
		project->sysStatus.syncMask = EMS_FALSE;
		project->projSaveStatus = EMS_FALSE;
	}

	systemStatus(EMS_FREE);
}


void linesMask_CB(void)
{
	char active_flag;

	msgReport("\0");

	// active only under MICROGRAPH / TEMPLATE / PARTICLE / FILAMENT modes
	switch ( project->sysStatus.emsdpMode ) {
	case DP_MCRGRAPH_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_TEMPLATE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_PARTICLE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_2DMIMAGE_MODE:
		active_flag = EMS_TRUE;
		break;
	default:
		active_flag = EMS_FALSE;
	}

	if ( active_flag == EMS_FALSE ) {
		EMS_DP->btnLines->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	// exclusive with showZoom button
	if ( project->sysStatus.showZoom == EMS_TRUE ) {
		EMS_DP->btnLines->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	// safe-guard data availability
	if ( project->dispMask.exist == EMS_FALSE ) {
		EMS_DP->btnLines->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	if ( (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) && (project->sysStatus.editTemp==EMS_FALSE) ) {
		EMS_DP->btnLines->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	systemStatus(EMS_BUSY);

	if ( EMS_DP->btnLines->value() ) {
		project->sysStatus.drawTools = EMS_LINES;
		project->parameter.brushSize = EMS_PENCL;
		EMS_DP->valBrush->value(EMS_PENCL);
		EMS_DP->valBrush2->value(EMS_PENCL);
		EMS_DP->btnShowMask->value(1);
		project->sysStatus.showMask = EMS_TRUE;
		EMS_DP->imgDisplay->redraw();
	}
	else {
		EMS_DP->imgDisplay->setMaskTexture(project->dispMask);
		project->sysStatus.drawTools = EMS_FALSE;
		project->parameter.brushSize = EMS_BRUSH;
		EMS_DP->valBrush->value(EMS_BRUSH);
		EMS_DP->valBrush2->value(EMS_BRUSH);
		project->sysStatus.syncMask = EMS_FALSE;
		project->projSaveStatus = EMS_FALSE;
	}

	systemStatus(EMS_FREE);
}


void paintMask_CB(void)
{
	char active_flag;

	msgReport("\0");

	// active only under MICROGRAPH / TEMPLATE / PARTICLE / FILAMENT modes
	switch ( project->sysStatus.emsdpMode ) {
	case DP_MCRGRAPH_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_TEMPLATE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_PARTICLE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_2DMIMAGE_MODE:
		active_flag = EMS_TRUE;
		break;
	default:
		active_flag = EMS_FALSE;
	}

	if ( active_flag == EMS_FALSE ) {
		EMS_DP->btnPaint->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	// exclusive with showZoom button
	if ( project->sysStatus.showZoom == EMS_TRUE ) {
		EMS_DP->btnPaint->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	// safe-guard data availability
	if ( project->dispMask.exist == EMS_FALSE ) {
		EMS_DP->btnPaint->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	if ( (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) && (project->sysStatus.editTemp==EMS_FALSE) ) {
		EMS_DP->btnPaint->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	systemStatus(EMS_BUSY);

	if ( EMS_DP->btnPaint->value() ) {
		project->sysStatus.drawTools = EMS_PAINT;
		EMS_DP->btnShowMask->value(1);
		project->sysStatus.showMask = EMS_TRUE;
		EMS_DP->imgDisplay->redraw();
	}
	else {
		EMS_DP->imgDisplay->setMaskTexture(project->dispMask);
		project->sysStatus.drawTools = EMS_FALSE;
		project->sysStatus.syncMask = EMS_FALSE;
		project->projSaveStatus = EMS_FALSE;
	}

	systemStatus(EMS_FREE);
}


void eraseMask_CB(void)
{
	char active_flag;

	msgReport("\0");

	// active only under MICROGRAPH / TEMPLATE / PARTICLE / FILAMENT modes
	switch ( project->sysStatus.emsdpMode ) {
	case DP_MCRGRAPH_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_TEMPLATE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_PARTICLE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_2DMIMAGE_MODE:
		active_flag = EMS_TRUE;
		break;
	default:
		active_flag = EMS_FALSE;
	}

	if ( active_flag == EMS_FALSE ) {
		EMS_DP->btnErase->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	// exclusive with showZoom button
	if ( project->sysStatus.showZoom == EMS_TRUE ) {
		EMS_DP->btnErase->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	// safe-guard data availability
	if ( project->dispMask.exist == EMS_FALSE ) {
		EMS_DP->btnErase->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	if ( (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) && (project->sysStatus.editTemp==EMS_FALSE) ) {
		EMS_DP->btnErase->value(0);
		project->sysStatus.drawTools = EMS_FALSE;
		return;
	}

	systemStatus(EMS_BUSY);

	if ( EMS_DP->btnErase->value() ) {
		project->sysStatus.drawTools = EMS_ERASE;
		EMS_DP->btnShowMask->value(1);
		project->sysStatus.showMask = EMS_TRUE;
		EMS_DP->imgDisplay->redraw();
	}
	else {
		EMS_DP->imgDisplay->setMaskTexture(project->dispMask);
		project->sysStatus.drawTools = EMS_FALSE;
		project->sysStatus.syncMask = EMS_FALSE;
		project->projSaveStatus = EMS_FALSE;
	}

	systemStatus(EMS_FREE);
}


void setMaskTrans_CB(unsigned char value)
{
	project->parameter.transp = value;
	project->projSaveStatus = EMS_FALSE;

	if ( project->dispMask.exist == EMS_TRUE )
		EMS_DP->imgDisplay->setMaskTexture(project->dispMask);
}


void clearMasking_CB(void)
{
	msgReport("\0");

	systemStatus(EMS_BUSY);
	project->clearFilmMask();
	systemStatus(EMS_FREE);
	project->projSaveStatus = EMS_FALSE;
}


void setMaskStatus_CB(void)
{
	char active_flag;

	msgReport("Toggle mask display");

	// active only under MICROGRAPH / TEMPLATE / PARTICLE / FILAMENT modes
	switch ( project->sysStatus.emsdpMode ) {
	case DP_MCRGRAPH_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_TEMPLATE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_PARTICLE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_2DMIMAGE_MODE:
		active_flag = EMS_TRUE;
		break;
	default:
		active_flag = EMS_FALSE;
	}

	if ( active_flag == EMS_FALSE ) {
		EMS_DP->btnShowMask->value(0);
		return;
	}

	// safe-guard data availability
//	if ( (project->dispMask.exist==EMS_FALSE) || (project->templates.exist==EMS_FALSE) ) {
	if ( project->dispMask.exist == EMS_FALSE ) {
		EMS_DP->btnShowMask->value(0);
		project->sysStatus.showMask = EMS_FALSE;
		return;
	}

	// action
	if ( EMS_DP->btnShowMask->value() )
		project->sysStatus.showMask = EMS_TRUE;
	else
		project->sysStatus.showMask = EMS_FALSE;

	EMS_DP->imgDisplay->redraw();
}


void setZoomStatus_CB(void)
{
	char active_flag;

	msgReport("Toggle floating magnifier");

	// active only under MCRGRAPH / PARTICLE / FILAMENT modes
	switch ( project->sysStatus.emsdpMode ) {
	case DP_MCRGRAPH_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_PARTICLE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_2DMIMAGE_MODE:
		active_flag = EMS_TRUE;
		break;
	default:
		active_flag = EMS_FALSE;
	}

	if ( active_flag == EMS_FALSE ) {
		EMS_DP->btnShowZoom->value(0);
		return;
	}

	// safe-guard data availability
	if ( project->dispFilm.exist == EMS_FALSE ) {
		EMS_DP->btnShowZoom->value(0);
		project->sysStatus.showZoom = EMS_FALSE;
		return;
	}

	// action
	if ( EMS_DP->btnShowZoom->value() ) {
		if ( project->sysStatus.drawTools == EMS_FALSE ) {
			EMS_DP->imgDisplay->resetZoomer();
			project->sysStatus.showZoom = EMS_TRUE;
		}
		else {
			EMS_DP->btnShowZoom->value(0);
			project->sysStatus.showZoom = EMS_FALSE;
			return;
		}
	}
	else
		project->sysStatus.showZoom = EMS_FALSE;

	EMS_DP->imgDisplay->redraw();
}

// PARTICLES page

void testSignature_CB(void)
{
	msgReport("\0");
	if ( project->micrograph.exist == EMS_FALSE ) return;

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Apply TEST screening?") == 0 ) return;

	systemStatus(EMS_BUSY);

	if ( project->sysStatus.fullScreen == EMS_TRUE )
		project->sysStatus.mapRestore = EMS_FALSE;

	project->sysStatus.testScreen = EMS_TRUE;
	project->sysStatus.fullScreen = EMS_FALSE;
	project->testScreen();
	EMS_DP->imgDisplay->redraw();

	systemStatus(EMS_FREE);
}


void calcSignature_CB(void)
{
	msgReport("\0");
	if ( project->micrograph.exist == EMS_FALSE ) return;

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Apply COMPLETE screening?") == 0 ) return;

	systemStatus(EMS_BUSY);

	if ( project->sysStatus.testScreen == EMS_TRUE )
		project->sysStatus.mapRestore = EMS_FALSE;

	project->sysStatus.testScreen = EMS_FALSE;
	project->sysStatus.fullScreen = EMS_TRUE;
	project->fullScreen();
	EMS_DP->imgDisplay->redraw();

	systemStatus(EMS_FREE);
	project->projSaveStatus = EMS_FALSE;
}


void gotoStackEditor_CB(void)
{
	char fileName[NAMESIZE];

	msgReport("Build and display particle image stack");

	if ( project->micrograph.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}

	if ( !strcmp(project->dirParticle, EMS_PRJ_DR) ) {
		errReport("Please set up a project space first!");
		return;
	}

	if ( project->parameter.totCount <= 0 ) {
		errReport("There is no particle data to save!");
		return;
	}

	systemStatus(EMS_BUSY);

	// save particle frames to stack
	sprintf(fileName, "%s/%s.mrc", project->dirParticle, project->filmName);
	
	if ( project->saveParticleMRC(fileName) ) {
		// switch to Image Stack Editor
		functImage->loadImageStack(fileName);
		functImage->paramDisplay();
		EMS_DP->imgDisplay->setStackTexture();
		setDPMode_CB(DP_FUNCIMAG_MODE);
	}

	systemStatus(EMS_FREE);
}


void setLCFthres_CB(double value)
{
	project->parameter.lcfThres = (float)(value - EMS_ZERO);
	project->projSaveStatus = EMS_FALSE;

	systemStatus(EMS_BUSY);
	project->screenLCF(project->parameter.lcfThres);
	project->screenSCF(project->parameter.scfThres);
	EMS_DP->imgDisplay->redraw();
	systemStatus(EMS_FREE);
}


void setSCFthres_CB(double value)
{
	project->parameter.scfThres = (float)(value - EMS_ZERO);
	project->projSaveStatus = EMS_FALSE;

	systemStatus(EMS_BUSY);
	project->screenSCF(project->parameter.scfThres);
	project->screenDCF(-1);
	EMS_DP->imgDisplay->redraw();
	systemStatus(EMS_FREE);
}


void setDCFthres_CB(double value)
{
	project->parameter.dcfThres = (float)(value - EMS_ZERO);
	project->projSaveStatus = EMS_FALSE;
}


void setDSTthres_CB(double value)
{
	project->parameter.distance = (float)(value - EMS_ZERO);
	project->projSaveStatus = EMS_FALSE;
/*
	systemStatus(EMS_BUSY);
	project->screenDST(project->parameter.distance);
	EMS_DP->imgDisplay->redraw();
	systemStatus(EMS_FREE);
*/
}


void setPrtkStatus_CB(void)
{
	char active_flag;

	msgReport("Toggle particle marker display");

	// active only under TEMPLATE / PARTICLE / FILAMENT modes
	switch ( project->sysStatus.emsdpMode ) {
	case DP_TEMPLATE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_PARTICLE_MODE:
		active_flag = EMS_TRUE;
		break;
	case DP_2DMIMAGE_MODE:
		active_flag = EMS_TRUE;
		break;
	default:
		active_flag = EMS_FALSE;
	}

	if ( active_flag == EMS_FALSE ) {
		EMS_DP->btnShowPrtk->value(0);
		project->sysStatus.showPrtk = EMS_FALSE;
		return;
	}

	// safe-guard data availability
//	if ( (project->dispFilm.exist==EMS_FALSE) || (project->templates.exist==EMS_FALSE) ) {
	if ( project->dispFilm.exist == EMS_FALSE ) {
		EMS_DP->btnShowPrtk->value(0);
		project->sysStatus.showPrtk = EMS_FALSE;
		return;
	}

	// action
	if ( EMS_DP->btnShowPrtk->value() )
		project->sysStatus.showPrtk = EMS_TRUE;
	else
		project->sysStatus.showPrtk = EMS_FALSE;

	EMS_DP->imgDisplay->redraw();
}


void inspectAccept_CB(void)
{
	msgReport("Accept a particle selection");

	if ( project->film.exist == EMS_TRUE ) {
		project->inspectAccept();
		project->projSaveStatus = EMS_FALSE;
	}
}


void inspectReject_CB(void)
{
	msgReport("Reject a particle selection");

	if ( project->film.exist == EMS_TRUE ) {
		project->inspectReject();
		project->projSaveStatus = EMS_FALSE;
	}
}


void inspectUndo_CB(void)
{
	msgReport("Undo the last decision");

	if ( project->film.exist == EMS_TRUE ) {
		project->inspectUndo();
		project->projSaveStatus = EMS_FALSE;
	}
}


void setFrameSize_CB(int value)
{
	project->sqrFrameSize = value;
}

// MONITOR page

void setMon1Resz_CB(int value)
{
	msgReport("\0");

	if ( value <=0 ) {
		errReport("Scaling factor should be a natural integer! Reset to 1.");
		value = 1;
	}
	else if ( value > EMS_RESZ_MAX ) {
		sprintf(message, "Scaling factor is too large! Reset to %d.", EMS_RESZ_MAX);
		errReport(message);
		value = EMS_RESZ_MAX;
	}

	EMS_DP->mon1Resz->value(value);
	cPanel->setResz1(value);

	if ( project->sysStatus.showSpec == EMS_TRUE )
		cPanel->redrawSpec();
	else
		cPanel->redrawFocus();

	project->projSaveStatus = EMS_FALSE;
}


void setMon2Resz_CB(int value)
{
	msgReport("\0");

	if ( value <=0 ) {
		errReport("Scaling factor should be a natural integer! Reset to 1.");
		value = 1;
	}
	else if ( value > EMS_RESZ_MAX ) {
		sprintf(message, "Scaling factor is too large! Reset to %d.", EMS_RESZ_MAX);
		errReport(message);
		value = EMS_RESZ_MAX;
	}

	EMS_DP->mon2Resz->value(value);
	cPanel->setResz2(value);

	if ( project->sysStatus.showRefr == EMS_TRUE )
		cPanel->redrawTemp();
	else
		cPanel->redrawFocus();

	project->projSaveStatus = EMS_FALSE;
}


void setMon1Blur_CB(int value)
{
	msgReport("\0");

	if ( value <0 ) {
		errReport("Blurring factor should be a natural integer! Reset to 0.");
		value = 0;
	}
	else if ( value > EMS_BLUR_MAX ) {
		sprintf(message, "Blurring factor is too large! Reset to %d.", EMS_BLUR_MAX);
		errReport(message);
		value = EMS_BLUR_MAX;
	}
	
	EMS_DP->mon1Blur->value(value);
	cPanel->setBlur1(value);

	if ( project->sysStatus.showSpec == EMS_FALSE )
		cPanel->redrawFocus();

	project->projSaveStatus = EMS_FALSE;
}


void setMon2Blur_CB(int value)
{
	msgReport("\0");

	if ( value <0 ) {
		errReport("Blurring factor should be a natural integer! Reset to 0.");
		value = 0;
	}
	else if ( value > EMS_BLUR_MAX ) {
		sprintf(message, "Blurring factor is too large! Reset to %d.", EMS_BLUR_MAX);
		errReport(message);
		value = EMS_BLUR_MAX;
	}
	
	EMS_DP->mon2Blur->value(value);
	cPanel->setBlur2(value);

	if ( project->sysStatus.showRefr == EMS_FALSE )
		cPanel->redrawFocus();

	project->projSaveStatus = EMS_FALSE;
}


void setMon1Cross_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnCross1->value() )
		project->sysStatus.mon1X = EMS_TRUE;
	else
		project->sysStatus.mon1X = EMS_FALSE;

	cPanel->monitorPrtk();

	if ( project->sysStatus.showZoom == EMS_TRUE )
		EMS_DP->imgDisplay->redraw();
}


void setMon2Cross_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnCross2->value() )
		project->sysStatus.mon2X = EMS_TRUE;
	else
		project->sysStatus.mon2X = EMS_FALSE;

	cPanel->monitorPrtk();
}


void setSpecView_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnSpec->value() ) {
		project->sysStatus.showSpec = EMS_TRUE;
		cPanel->redrawSpec();

		if ( project->sysStatus.showRefr == EMS_FALSE ) {
			EMS_DP->monBrit->value(project->parameter.spcBrit);
			EMS_DP->monCntr->value(project->parameter.spcCntr);
		}
	}
	else {
		project->sysStatus.showSpec = EMS_FALSE;
		cPanel->redrawFocus();

		if ( project->sysStatus.showRefr == EMS_TRUE ) {
			EMS_DP->monBrit->value(project->parameter.refBrit);
			EMS_DP->monCntr->value(project->parameter.refCntr);
		}
		else {
			EMS_DP->monBrit->value(project->parameter.monBrit);
			EMS_DP->monCntr->value(project->parameter.monCntr);
		}
	}
}


void setRefrView_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnRefr->value() ) {
		project->sysStatus.showRefr = EMS_TRUE;
		cPanel->redrawTemp();

		EMS_DP->monBrit->value(project->parameter.refBrit);
		EMS_DP->monCntr->value(project->parameter.refCntr);
	}
	else {
		project->sysStatus.showRefr = EMS_FALSE;
		cPanel->redrawFocus();

		if ( project->sysStatus.showSpec == EMS_TRUE ) {
			EMS_DP->monBrit->value(project->parameter.spcBrit);
			EMS_DP->monCntr->value(project->parameter.spcCntr);
		}
		else {
			EMS_DP->monBrit->value(project->parameter.monBrit);
			EMS_DP->monCntr->value(project->parameter.monCntr);
		}
	}
}


void setMonBright_CB(double value)
{
	msgReport("\0");
	systemStatus(EMS_BUSY);
	
	if ( project->sysStatus.showRefr == EMS_TRUE ) {
		project->parameter.refBrit = (float)value;
		if ( project->templates.exist == EMS_TRUE )
			cPanel->redrawTemp();
	}
	else if ( project->sysStatus.showSpec == EMS_TRUE ) {
		project->parameter.spcBrit = (float)value;
		if ( project->micrograph.exist == EMS_TRUE )
			cPanel->redrawSpec();
	}
	else {
		project->parameter.monBrit = (float)value;
		if ( project->micrograph.exist == EMS_TRUE )
			cPanel->redrawFocus();
	}

	project->projSaveStatus = EMS_FALSE;
	systemStatus(EMS_FREE);
}


void setMonContrast_CB(double value)
{
	msgReport("\0");
	systemStatus(EMS_BUSY);
	
	if ( project->sysStatus.showRefr == EMS_TRUE ) {
		project->parameter.refCntr = (float)value;
		if ( project->templates.exist == EMS_TRUE )
			cPanel->redrawTemp();
	}
	else if ( project->sysStatus.showSpec == EMS_TRUE ) {
		project->parameter.spcCntr = (float)value;
		if ( project->micrograph.exist == EMS_TRUE )
			cPanel->redrawSpec();
	}
	else {
		project->parameter.monCntr = (float)value;
		if ( project->micrograph.exist == EMS_TRUE )
			cPanel->redrawFocus();
	}

	project->projSaveStatus = EMS_FALSE;
	systemStatus(EMS_FREE);
}


void setFilaBright_CB(double value)
{
	msgReport("\0");
	systemStatus(EMS_BUSY);
	
	project->parameter.mon3Brit = (float)value;

	if ( project->imgCrop.exist == EMS_TRUE )
		cPanel->monitorFila(NULL, 0, 0);
	
	project->projSaveStatus = EMS_FALSE;
	systemStatus(EMS_FREE);
}


void setFilaContrast_CB(double value)
{
	msgReport("\0");
	systemStatus(EMS_BUSY);

	project->parameter.mon3Cntr = (float)value;

	if ( project->imgCrop.exist == EMS_TRUE )
		cPanel->monitorFila(NULL, 0, 0);

	project->projSaveStatus = EMS_FALSE;
	systemStatus(EMS_FREE);
}


void findSegment_CB(void)
{
	msgReport("\0");
	systemStatus(EMS_BUSY);

	project->findSegment();
	EMS_DP->imgDisplay->redraw();

	systemStatus(EMS_FREE);
}


void traceFilament_CB(void)
{
	msgReport("\0");
	systemStatus(EMS_BUSY);

	project->traceFilament();
	EMS_DP->imgDisplay->redraw();

	systemStatus(EMS_FREE);
}


void setFilaBridge_CB(double value)
{
	project->parameter.filaBrdg = (float)value;
	project->projSaveStatus = EMS_FALSE;
}


void setFilaBend_CB(double value)
{
	project->parameter.filaBend = (float)value;
	project->projSaveStatus = EMS_FALSE;
}


void setMon3Resz_CB(int value)
{
	msgReport("\0");

	if ( value <=0 ) {
		errReport("Scaling factor should be a natural integer! Reset to 1.");
		value = 1;
	}
	else if ( value > EMS_RESZ_MAX ) {
		sprintf(message, "Scaling factor is too large! Reset to %d.", EMS_RESZ_MAX);
		errReport(message);
		value = EMS_RESZ_MAX;
	}

	EMS_DP->mon3Resz->value(value);
	cPanel->setResz3(value);
	cPanel->monitorFila(NULL, 0, 0);

	project->projSaveStatus = EMS_FALSE;
}


void setMon3Cross_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnCross3->value() )
		project->sysStatus.mon3X = EMS_TRUE;
	else
		project->sysStatus.mon3X = EMS_FALSE;

	cPanel->monitorFila(NULL, 0, 0);
}


void setCropMode_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnOrtho->value() )
		project->cropMode = EMS_TRUE;
	else
		project->cropMode = EMS_FALSE;
}


void setFilaDiameter_CB(double value)
{
	project->filaDiam = value;
}


void setSegmtLeng_CB(double value)
{
	project->filaUnit = value;
}


void setSegmtCrox_CB(double value)
{
	project->filaCrox = value;
}


void setFilaShiftX_CB(double value)
{
	msgReport("\0");

	project->filaDX = value;
	project->setFilaModel(0, 0, 0, 0, 0);
}


void setFilaShiftY_CB(double value)
{
	msgReport("\0");

	project->filaDY = value;
	project->setFilaModel(0, 0, 0, 0, 0);
}


void setFilaExtnX_CB(double value)
{
	msgReport("\0");

	project->filaDL = value;
	project->setFilaModel(0, 0, 0, 0, 0);
}


void setFilaExtnY_CB(double value)
{
	msgReport("\0");

	project->filaDW = value;
	project->setFilaModel(0, 0, 0, 0, 0);
}


void setFilaRotate_CB(double value)
{
	msgReport("\0");

	project->filaDR = value;
	project->setFilaModel(0, 0, 0, 0, 0);
}


void setFilaAve_CB(void)
{
	msgReport("\0");

	if ( EMS_DP->btnFilaAve->value() )
		project->sysStatus.filaAve = EMS_TRUE;
	else
		project->sysStatus.filaAve = EMS_FALSE;

	cPanel->monitorFila(NULL, 0, 0);
}


void clearFilaTemp_CB(void)
{
	msgReport("\0");
	if ( project->imgCrop.exist == EMS_FALSE ) return;

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Clear image cropping?") == 0 ) return;

	project->imgCrop.clear();
	cPanel->monitorFila(NULL, 0, 0);
	EMS_DP->imgDisplay->clearFilaBoxing();
}


void applyImageProc_CB(int opt)
{
	msgReport("\0");

	static int opCode=EMS_OPT_NORMAL;
	char status=EMS_FALSE;

	// set operation mode
	if ( opt > 0 ) {
		opCode = opt;
		return;
	}

	if ( project->micrograph.exist == EMS_FALSE ) return;

	// carry out the operation
	systemStatus(EMS_BUSY);

	switch ( opCode ) {
	case EMS_OPT_NORMAL:
		project->normalizeImage();
		project->setDispFilm();
		status = EMS_TRUE;

		break;

	case EMS_OPT_FILTER:
		if ( !strcmp(project->dirParticle, EMS_PRJ_DR) ) {
			errReport("Please set up a project space first!");
			return;
		}
		
		project->parameter.passL = (float)(EMS_DP->valParam4->value());
		project->parameter.passH = (float)(EMS_DP->valParam5->value());
		project->bandFiltering();
		project->setDispFilm();
		status = EMS_TRUE;

		break;

	case EMS_OPT_INVERT:
		project->invContrast();
		status = EMS_TRUE;

		break;

	case EMS_OPT_FLATEN:
		if ( !strcmp(project->dirParticle, EMS_PRJ_DR) ) {
			errReport("Please set up a project space first!");
			return;
		}
		
		project->imageFlattening();
		project->setDispFilm();
		status = EMS_TRUE;

		break;

	case EMS_OPT_HFLIP:
		project->micrograph.flipH();
		project->resizeMcrgraph();
		project->setDispFilm();
		status = EMS_TRUE;

		break;

	case EMS_OPT_VFLIP:
		project->micrograph.flipV();
		project->resizeMcrgraph();
		project->setDispFilm();
		status = EMS_TRUE;

		break;

	case EMS_OPT_PIXBIN:
		project->micrograph.binBox = (int)(EMS_DP->valParam4->value());

		if ( project->micrograph.binBox >= 2 ) {
			project->pixelBinning();
			project->setDispFilm();
			status = EMS_TRUE;
			
			EMS_DP->dspFilmX->value(project->micrograph.dimX);
			EMS_DP->dspFilmY->value(project->micrograph.dimY);
			EMS_DP->dspPixSize->value(project->parameter.pixelSize);
		}
		else {
			errReport("Incorrect binning size!");
			status = EMS_FALSE;
		}

		break;
	}

	if ( status ) {
		project->sysStatus.newScale = EMS_TRUE;
		project->sysStatus.mapRestore = EMS_FALSE;
		project->projSaveStatus = EMS_FALSE;

		EMS_DP->imgDisplay->setFilmTexture(project->dispFilm);
		cPanel->drawImgInfo(0);
	}

	systemStatus(EMS_FREE);
}

// FUNCTION: IMAGE STACK

// load MRC image stack
void loadImageStack_CB(void)
{
	char *fileName;

	msgReport("Load a new MRC stack");

	// save the current particle stack if necessary
	if ( functImage->flagSaved == EMS_FALSE ) {
		if ( fl_ask("Save the current particle stack?") ) {
			fileName = fl_file_chooser("Saving MRC stack ...", "*.mrc", NULL, 0);
			if ( fileName != NULL ) functImage->saveImageStack(fileName);
		}
	}

	// choose the file to be opened
	fileName = fl_file_chooser("Loading MRC stack ...", "*.mrc", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);

	if ( functImage->loadImageStack(fileName) ) {
		functImage->paramDisplay();
		EMS_DP->imgDisplay->setStackTexture();
		EMS_DP->imgDisplay->redraw();
	}

	systemStatus(EMS_FREE);
}


// save MRC image stack
void saveImageStack_CB(void)
{
	char *fileName;

	msgReport("Save images to an MRC stack");
	if ( functImage->imgStack.exist == EMS_FALSE ) return;

	fileName = fl_file_chooser("Saving MRC stack ...", "*.mrc", NULL, 0);
	if ( fileName == NULL ) return;

	if ( functImage->saveImageStack(fileName) ) {
		simpleFilePath(fileName);
		EMS_DP->dspImageFile->value(fileName);
	}
}


void setDisplayFrame_CB(int f0)
{
	functImage->setDispFrames(f0);
}


void dispImageStack_CB(void)
{
	msgReport("\0");

	systemStatus(EMS_BUSY);
	EMS_DP->imgDisplay->setStackTexture();
	EMS_DP->imgDisplay->redraw();
	EMS_DP->valFrame0->value(functImage->frame0+1);
	systemStatus(EMS_FREE);
}


void setDispQuality_CB(int quality)
{
	switch ( quality ) {
	case EMS_QUAL_NORM:
		functImage->textSize = EMS_TEX_NORM;
		break;
	case EMS_QUAL_GOOD:
		functImage->textSize = EMS_TEX_GOOD;
		break;
	case EMS_QUAL_HIGH:
		functImage->textSize = EMS_TEX_HIGH;
		break;
	default:
		functImage->textSize = EMS_TEX_NORM;
	}
}


void setFrameStart_CB(void)
{
	msgReport("\0");

	functImage->frame0 = 0;
	dispImageStack_CB();
}


void setFrameBack_CB(void)
{
	msgReport("\0");

	functImage->browseFrames(-1);
	dispImageStack_CB();
}


void setFrameFore_CB(void)
{
	msgReport("\0");

	functImage->browseFrames(+1);
	dispImageStack_CB();
}


void setFrameEnd_CB(void)
{
	msgReport("\0");

	functImage->browseFrames(0);
	dispImageStack_CB();
}


void setLabelDisplay_CB(void)
{
	msgReport("\0");

	if ( functImage->labelDisp == EMS_TRUE )
		functImage->labelDisp = EMS_FALSE;
	else
		functImage->labelDisp = EMS_TRUE;

	EMS_DP->imgDisplay->redraw();
}


void setFrameScale_CB(int val)
{
	msgReport("\0");

	if ( val < 1 ) {
		val = 1;
		EMS_DP->valDispScale->value(1);
	}

	functImage->imgScale = val;
}


void clearImgSelect_CB(void)
{
	msgReport("\0");
	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Clear the current frame selection?") == 0 ) return;

	functImage->clearSelect();
	EMS_DP->imgDisplay->redraw();
}


void extractImgSelect_CB(void)
{
	char flagSync=EMS_FALSE;
	char pName[NAMESIZE], stakName[NAMESIZE];

	if ( functImage->selCount == 0 ) {
		msgReport("No image frame has been selected");
		return;
	}
	else
		msgReport("Extract selected particle images");

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Extract the selected image frames?") == 0 ) return;

	systemStatus(EMS_BUSY);

	if ( functImage->extractFrames(flagSync) ) {
		functImage->paramDisplay();
		dispImageStack_CB();
	}

	if ( flagSync && strcmp(project->dirParticle, EMS_PRJ_DR) ) {
		
		// assign image stack file name
		getFilePath(functImage->stackName, pName, stakName);
		removeExt(stakName, ".mrc");
		
		// attempt to synchronize with micrograph display
		fl_beep(FL_BEEP_MESSAGE);
		if ( fl_ask("Update particle selection in micrograph?") ) {
			
			if ( project->syncPrtkSelect(pName, stakName) ) {
				EMS_DP->dspCount->value(project->parameter.totCount);
				setDPMode_CB(DP_PARTICLE_MODE);
			}
		}
	}

	systemStatus(EMS_FREE);
}


void deleteImgSelect_CB(void)
{
	char flagSync=EMS_FALSE;
	char pName[NAMESIZE], stakName[NAMESIZE];

	if ( functImage->selCount == 0 ) {
		msgReport("No image frame has been selected");
		return;
	}
	else
		msgReport("Delete selected particle images");

	fl_beep(FL_BEEP_MESSAGE);
	if ( fl_ask("Delete the selected image frames?") == 0 ) return;

	systemStatus(EMS_BUSY);

	if ( functImage->deleteFrames(flagSync) ) {
		functImage->paramDisplay();
		dispImageStack_CB();
	}

	if ( flagSync && strcmp(project->dirParticle, EMS_PRJ_DR) ) {
		
		// assign image stack file name
		getFilePath(functImage->stackName, pName, stakName);
		removeExt(stakName, ".mrc");
		
		// attempt to synchronize with micrograph display
		fl_beep(FL_BEEP_MESSAGE);
		if ( fl_ask("Update particle selection in micrograph?") ) {
			
			if ( project->syncPrtkSelect(pName, stakName) ) {
				EMS_DP->dspCount->value(project->parameter.totCount);
				setDPMode_CB(DP_PARTICLE_MODE);
			}
		}
	}

	systemStatus(EMS_FREE);
}


void closeImageStack_CB(void)
{
	char *fileName;

	// save the current particle stack if necessary
	if ( functImage->imgStack.exist == EMS_FALSE ) return;
	setDPMode_CB(DP_FUNCIMAG_MODE);

	if ( functImage->flagSaved == EMS_FALSE ) {
		if ( fl_ask("Save the current particle stack?") ) {
			fileName = fl_file_chooser("Saving MRC stack ...", "*.mrc", NULL, 0);
			if ( fileName != NULL ) functImage->saveImageStack(fileName);
		}
	}

	functImage->clearImageStack();
	functImage->paramDisplay();
	EMS_DP->imgDisplay->redraw();
}


void appendImageStack_CB(void)
{
	char *fileName;

	msgReport("\0");

	fileName = fl_file_chooser("Appending MRC stack ...", "*.mrc", NULL, 0);
	if ( fileName == NULL ) return;

	systemStatus(EMS_BUSY);

	if ( functImage->appendImageStack(fileName) ) {
		functImage->paramDisplay();
		EMS_DP->imgDisplay->setStackTexture();
		EMS_DP->imgDisplay->redraw();
	}

	systemStatus(EMS_FREE);
}


void applyStackProc_CB(int opt)
{
	static int opCode=EMS_OPT_NORMAL;
	char status=EMS_FALSE;

	msgReport("\0");

	// set operation code
	if ( opt > 0 ) {
		opCode = opt;
		return;
	}

	if ( functImage->imgStack.exist == EMS_FALSE ) return;

	// carry out the operation
	systemStatus(EMS_BUSY);

	switch ( opCode ) {
	case EMS_OPT_NORMAL:
		status = functImage->normImageStack();
		break;

	case EMS_OPT_FILTER:
		functImage->lowPass = EMS_DP->valParam1->value();
		functImage->highPass = EMS_DP->valParam2->value();
		status = functImage->filtImageStack();
		break;

	case EMS_OPT_ISCALE:
		functImage->frameMult = EMS_DP->valParam1->value();
		functImage->frameAddt = EMS_DP->valParam2->value();
		status = functImage->scalImageStack();
		break;

	case EMS_OPT_HSCLIP:
		functImage->clipMin = (float)(EMS_DP->valParam1->value());
		functImage->clipMax = (float)(EMS_DP->valParam2->value());
		status = functImage->clipImageStack();
		break;

	case EMS_OPT_RESIZE:
		functImage->cropX = (int)(EMS_DP->valParam1->value());
		functImage->cropY = (int)(EMS_DP->valParam2->value());
		status = functImage->cropImageStack();
		break;

	case EMS_OPT_SAMPLE:
		functImage->imgStack.samX = (int)(EMS_DP->valParam1->value());
		functImage->imgStack.samY = (int)(EMS_DP->valParam2->value());
		status = functImage->imgResample();
		break;

	case EMS_OPT_PIXBIN:
		functImage->imgStack.binBox = (int)(EMS_DP->valParam1->value());
		status = functImage->imgPixBin();
		break;

	case EMS_OPT_ROTAVE:
		status = functImage->imgRotaAve();
		break;

	case EMS_OPT_ROTSUM:
		status = functImage->imgRotaSum();
		break;

	case EMS_OPT_SUMALL:
		status = functImage->imgSumAll();
		break;
	}

	if ( status ) {
		functImage->paramDisplay();
		EMS_DP->imgDisplay->setStackTexture();
		EMS_DP->imgDisplay->redraw();
	}

	systemStatus(EMS_FREE);
}

// FUNCTION: DIGITAL GEL FILTRATION

void gelMassStat_CB(void)
{
	msgReport("Generate density-mass histogram");
	systemStatus(EMS_BUSY);

	if ( functImage->calcGelMass() ) {
		EMS_DP->gelMassMin->value(functImage->gelMassMin);
		EMS_DP->gelMassMax->value(functImage->gelMassMax);
		EMS_DP->gelMassAve->value(functImage->gelMassAve);
		EMS_DP->gelMassVar->value(functImage->gelMassStd);
		EMS_DP->gelMassLow->value(functImage->gelLmass);
		EMS_DP->gelMassHigh->value(functImage->gelHmass);
		EMS_DP->imgDisplay->redraw();
	}

	systemStatus(EMS_FREE);
}


void setGelMassRad_CB(double val)
{
	functImage->gelMassRad = (float)val;
}


void gelFiltrate_CB(void)
{
	msgReport("Apply digital gel filtration");
	systemStatus(EMS_BUSY);

	if ( functImage->gelFiltration() ) {
		EMS_DP->gelPrtkTotal->value(functImage->param.imgN);
		EMS_DP->gelPrtkSelect->value(functImage->gelCount);
		EMS_DP->gelMassLow->value(functImage->gelLmass);
		EMS_DP->gelMassHigh->value(functImage->gelHmass);
		EMS_DP->imgDisplay->redraw();
	}

	systemStatus(EMS_FREE);
}


void setGelLmass_CB(double mass)
{
	functImage->setGelLmass(mass);
}


void setGelHmass_CB(double mass)
{
	functImage->setGelHmass(mass);
}


void gelCommit_CB(void)
{
	msgReport("\0");

	if ( functImage->gelCommit() )
		setDPMode_CB(DP_FUNCIMAG_MODE);
}
