//	James Z. Chen
//	January 1, 2005
//	All rights reserved

#include "display.h"
#include "project.h"
#include "images.h"
#include "panel.h"
#include "misc.h"
#include "gui.h"
#include "gltext.h"

// lighting parameters
#define GRX_LIT_AMBIENT		0.25f
#define GRX_LIT_DIFFUSE		0.35f

// scaling factors for mouse interaction
#define R_SCALE	0.4		// rotation
#define T_SCALE	0.5		// translation
#define Z_SCALE 0.005	// zooming


extern DisplayPanel_UI *EMS_DP;

GLfloat glMatShininess = 50.0;
GLfloat glLightSpecular[] = { 0.8f, 0.8f, 0.8f, 1.0f };
GLfloat glLightPosition[] = { 1.0f, 1.0f, 1.0f, 0.0f };
GLfloat glLightLevelset[] = { 0.0f, 0.0f, 1.0f, 0.0f };
GLfloat glLightAmbient[] = { GRX_LIT_AMBIENT, GRX_LIT_AMBIENT, GRX_LIT_AMBIENT, 1.0f };
GLfloat glLightDiffuse[] = { GRX_LIT_DIFFUSE, GRX_LIT_DIFFUSE, GRX_LIT_DIFFUSE, 1.0f };

GLfloat glMatAmbient[] = { 0.2f, 0.2f, 0.2f, 1.0f };
GLfloat glMatDiffuse[] = { 0.4f, 0.4f, 0.4f, 1.0f };
GLfloat glMatSpecular[] = { 0.8f, 0.8f, 0.8f, 1.0f };
GLfloat glModelAmbient[] = { 0.2f, 0.2f, 0.2f, 1.0f };
GLfloat glLinesAmbient[] = { 1.0f, 1.0f, 1.0f, 1.0f };
GLfloat glLit2DAmbient[] = { 1.0f, 1.0f, 1.0f, 1.0f };


EMS_Display::EMS_Display(int x, int y, int w, int h, const char *l) : Fl_Gl_Window(x, y, w, h, l)
{
	grxBkgR = EMS_BKG_R;
	grxBkgG = EMS_BKG_G;
	grxBkgB = EMS_BKG_B;
	grxMode = EMS_GRAPHX_2D;

	zoom = zoom3D = 1.0;
	tranX = tranY = 0.0;
	tranX3 = deltaX = 0.0;
	tranY3 = deltaY = 0.0;
	rotaH3 = rotaV3 = 0.0;
	nearPlane = 100;
	viewDepth = 100;
	setIdentity(rotaMatrix, 4);

	filmX = filmY = 0;

	// allocate memory for FILM and MASK texture
	texFilmName = texMaskName = 0;
	texFilm = new GLubyte[EMS_FILM_TEX * EMS_FILM_TEX * 3];
	texMask = new GLubyte[EMS_FILM_TEX * EMS_FILM_TEX * 4];

	// memory for TEMPLATE texture will be allocated dynamically
	texTempName = NULL;
	texTemp = NULL;

	// allocate memory for reference editing texture
	texRefr = new GLubyte[EMS_RFED_TEX * EMS_RFED_TEX * 3];

	// memory for image stack texture will be allocated dynamically
	texStackName = NULL;
	texStack = NULL;

	// initialize focus point
	focusX = focusY = -1;

	// initialize line drawing parameters
	lineX0 = lineY0 = lineX1 = lineY1 = -1;

	// initialize filament boxing parameters
	filaX0 = filaY0 = filaX1 = filaY1 = -1;

	// initialize circle drawing parameters
	circX = circY = circR = -1;

	// initialize magnifier parameters
	magX = magY = -1;

	// initialize film marker counter
	markNum = 0;

	// initialize model color display
	model_R = EMS_DENSITY_R;
	model_G = EMS_DENSITY_G;
	model_B = EMS_DENSITY_B;
	model_A = EMS_DENSITY_A;
}


EMS_Display::~EMS_Display()
{
	if ( texFilm != NULL ) delete [] texFilm;
	if ( texMask != NULL ) delete [] texMask;
	if ( texTemp != NULL ) delete [] texTemp;
	if ( texRefr != NULL ) delete [] texRefr;
	if ( texStack != NULL ) delete [] texStack;
	if ( texTempName != NULL ) delete [] texTempName;
	if ( texStackName != NULL ) delete [] texStackName;
}


void EMS_Display::initDisplay(int dimX, int dimY)
{
	filmX = dimX;
	filmY = dimY;

	zoom = 1.0;
	tranX = tranY = 0.0;
}


void EMS_Display::draw(void)
{
    if ( !valid() ) {
		winX = w();
		winY = h();
        glViewport(0, 0, winX, winY);
	}

	// 2D image display
	if ( grxMode == EMS_GRAPHX_2D ) {
		
		glMatrixMode( GL_PROJECTION );
		glLoadIdentity();
		glOrtho(-winX/2.0, winX/2.0, -winY/2.0, winY/2.0, -1, EMS_INF_DEPTH);
		
		glMatrixMode( GL_MODELVIEW );
		glLoadIdentity();
		gluLookAt(0.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, -1.0, 0.0);
		
		glEnable( GL_LIGHTING );
		glEnable( GL_LIGHT0 );
		glEnable( GL_COLOR_MATERIAL );
		glEnable( GL_DEPTH_TEST );
		
		glLightfv( GL_LIGHT0, GL_AMBIENT, glLit2DAmbient );
		
		glClearColor(grxBkgR, grxBkgG, grxBkgB, 0.0);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		
		glMatrixMode( GL_MODELVIEW );
		glScaled(zoom, zoom, 1.0);
		glViewport(0, 0, winX, winY);
		
		// 2D display
		drawFilm();
		drawTemp();
		drawMask();
		drawPeak();
		drawFocus();
		drawStack();
		drawZoomer();
		drawDigitGel();
		drawCharts();
		drawFilmMarker();
		drawFilaBoxing();
		drawFilaTrace();
	}
}


int EMS_Display::handle(int event)
{
	int x, y;
	static int x0, y0;
	static double zoom0, x0_ang, y0_ang, x0_tran, y0_tran;
	static double x0_zoom, y0_clip;
	
	x = Fl::event_x();
	y = Fl::event_y();

	rotaH3 = rotaV3 = 0.0;
	deltaX = deltaY = 0.0;

	switch ( event ) {
	case FL_PUSH:
		x0_tran = x0_ang = x0_zoom = x0 = x;
		y0_tran = y0_ang = y0_clip = y0 = y;
		zoom0 = zoom3D;

		if ( Fl::event_button() == FL_LEFT_MOUSE ) {

			// for filament boxing
			if ( Fl::event_ctrl() && (project->sysStatus.emsdpMode==DP_2DMIMAGE_MODE) )
				manuBoxing(x, y, 0);

			// for image frame selection
			else if ( Fl::event_ctrl() && (project->sysStatus.emsdpMode==DP_FUNCIMAG_MODE) )
				setFocusXY(x, y);

			// for template alignment center
			else if ( Fl::event_ctrl() && (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) )
				setAlignCenter(x, y);

			// for digital gel thresholding
			else if ( project->sysStatus.emsdpMode == DP_FUNCDGEL_MODE )
				gelThreshold(x0, y0, 0);

			// for image zooming
			else if ( project->sysStatus.showZoom == EMS_TRUE )
				zoomCenter(x, y);

			// for line drawing, record the starting point
			else if ( project->sysStatus.drawTools == EMS_LINES )
				manuLines(x, y, 0);

			// for manual paint mask
			else if ( project->sysStatus.drawTools == EMS_PAINT )
				manuPaint(x, y);

			// for manual erase mask
			else if ( project->sysStatus.drawTools == EMS_ERASE )
				manuErase(x, y);

			// clear film focus
			else
				clearFocus();
		}

		redraw();
		return(1);
		
	case FL_DRAG:

		// CTRL+LEFT_BTN generates 251 after holding for a while (PC Windows)
		if ( (Fl::event_button() == FL_LEFT_MOUSE) || (Fl::event_button() == 251) ) {
			
			// for filament boxing
			if ( Fl::event_ctrl() && (project->sysStatus.emsdpMode==DP_2DMIMAGE_MODE) )
				manuBoxing(x, y, 1);

			// for digital gel thresholding
			else if ( project->sysStatus.emsdpMode == DP_FUNCDGEL_MODE )
				gelThreshold(x, y, 1);

			// for image zooming
			else if ( project->sysStatus.showZoom == EMS_TRUE )
				zoomCenter(x, y);
			
			// for manual mask painting
			else if ( project->sysStatus.drawTools == EMS_PAINT )
				manuPaint(x, y);
			
			// for line drawing
			else if ( project->sysStatus.drawTools == EMS_LINES )
				manuLines(x, y, 1);
			
			// for circle drawing
			else if ( project->sysStatus.drawTools == EMS_CIRCLE )
				manuCircle(x-x0, y-y0, 1);
			
			// for manual mask erasing
			else if ( project->sysStatus.drawTools == EMS_ERASE )
				manuErase(x, y);

			// for image translation
			else
				setTrans(x-x0, y-y0);
		}
		
		else if ( Fl::event_button() == FL_RIGHT_MOUSE ) {
			
			// for circle drawing
			if ( project->sysStatus.drawTools == EMS_CIRCLE )
				manuCircle(x-x0, y-y0, 2);
			
			// for image zooming
			else if ( (project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE) ||
				      (project->sysStatus.emsdpMode == DP_PARTICLE_MODE) ||
					  (project->sysStatus.emsdpMode == DP_2DMIMAGE_MODE) )
				setZoom(x+y-x0-y0);
		}
		
		x0 = x; y0 = y;
		redraw();
		return(1);
		
	case FL_RELEASE:
		x0 = x; y0 = y;
		
		if ( Fl::event_button() == FL_LEFT_MOUSE ) {
			
			// selection with "CTRL + LEFT_BUTTON"
			if ( Fl::event_ctrl() ) {
				// for filament boxing
				if ( project->sysStatus.emsdpMode == DP_2DMIMAGE_MODE )
					manuBoxing(x, y, -1);
				// for particle selection
				else if ( project->sysStatus.emsdpMode == DP_PARTICLE_MODE )
					manuSelect(x, y);
				// for spectrum region selection
				else if ( project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE ) {
					if ( project->sysStatus.imgInform == EMS_SPEC_MODE )
						manuSelect(x, y);
				}
			}

			// semi-automatic selection with "ALT + LEFT_BUTTON"
			else if ( Fl::event_alt() || Fl::event_state(FL_META) ) {
				if ( project->sysStatus.emsdpMode == DP_PARTICLE_MODE )
					coopSelect(x, y);
			}

			// particle deletion with "SHIFT + LEFT_BUTTON"
			else if ( Fl::event_shift() ) {
				if ( project->sysStatus.emsdpMode == DP_PARTICLE_MODE )
					manuDelete(x, y);
			}

			// for digital gel thresholding
			else if ( project->sysStatus.emsdpMode == DP_FUNCDGEL_MODE )
				gelThreshold(x, y, -1);

			// for mask line drawing with width
			else if ( project->sysStatus.drawTools == EMS_LINES )
				manuLines(x, y, -1);
		}

		else if ( Fl::event_button() == FL_RIGHT_MOUSE ) {

			// display current focus on monitors
			if ( Fl::event_ctrl() )
				setInspectFocus(x, y, 1);
		}

		redraw();
		return(1);
		
	default:
		return(Fl_Gl_Window::handle(event));
	}

	return(Fl_Gl_Window::handle(event));
}


void EMS_Display::setFilmTexture(EMS_MAP &map)
{
	GLubyte *scaled, *filmc;
	double ave, std, var, delInt, val;
	long index, mapSize, texSize = EMS_FILM_TEX * EMS_FILM_TEX;

	if ( project->dispFilm.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}

	mapSize = filmX * filmY;
	scaled = new GLubyte[texSize];
	filmc = new GLubyte[mapSize];
	
	ave = var = std = delInt = 0.0;

	for ( index=0; index<mapSize; index++ ) {
		ave += map.dataF[index];
		var += map.dataF[index] * map.dataF[index];
	}
	
	ave /= mapSize;
	var /= mapSize;
	std = var - ave * ave;

	if ( std > 0.0 )
		std = sqrt(std);
	else
		std = 0.0;

	delInt = (1.0 - project->parameter.flmCntr) * std * EMS_HSTG_SIGMA + EMS_ZERO;

	for ( index=0; index<mapSize; index++ ) {
		val = 255 * ((map.dataF[index] - ave) / delInt + project->parameter.flmBrit);

		if ( val < 0 )
			filmc[index] = 0;
		else if ( val > 255 )
			filmc[index] = 255;
		else
			filmc[index] = (GLubyte)val;
	}

	gluScaleImage(GL_RED, filmX, filmY, GL_UNSIGNED_BYTE, filmc, EMS_FILM_TEX, EMS_FILM_TEX, GL_UNSIGNED_BYTE, scaled);
	
	for ( index=0; index<texSize; index++ ) {
		texFilm[index*3+0] = scaled[index];
		texFilm[index*3+1] = scaled[index];
		texFilm[index*3+2] = scaled[index];
	}
	
	delete [] scaled;
	delete [] filmc;
	
	// prepare texture map for micrograph image display
	if ( glIsTexture(texFilmName) )
		glDeleteTextures(1, &texFilmName);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(1, &texFilmName);
	glBindTexture(GL_TEXTURE_2D, texFilmName);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, EMS_FILM_TEX, EMS_FILM_TEX, 0, GL_RGB, GL_UNSIGNED_BYTE, texFilm);
	
	redraw();
}


void EMS_Display::setMaskTexture(EMS_MAP &map)
{
	GLubyte *maskc, *scaled;
	long index, mapSize, texSize = EMS_FILM_TEX * EMS_FILM_TEX;

	if ( project->dispMask.exist == EMS_FALSE ) {
		errReport("EM micrograph has not been loaded yet!");
		return;
	}

	mapSize = filmX * filmY;
	maskc = new GLubyte[mapSize];
	scaled = new GLubyte[texSize];

	for ( index=0; index<mapSize; index++ ) {
		if ( map.dataC[index] == EMS_TRUE )
			maskc[index] = 255;
		else
			maskc[index] = 0;
	}

	gluScaleImage(GL_RED, filmX, filmY, GL_UNSIGNED_BYTE, maskc, EMS_FILM_TEX, EMS_FILM_TEX, GL_UNSIGNED_BYTE, scaled);
	
	for ( index=0; index<texSize; index++ ) {
		if ( scaled[index] > 128 ) {
			texMask[index*4+0] = EMS_MSK_R;
			texMask[index*4+1] = EMS_MSK_G;
			texMask[index*4+2] = EMS_MSK_B;
		}
		else {
			texMask[index*4+0] = EMS_MSK_BGD;
			texMask[index*4+1] = EMS_MSK_BGD;
			texMask[index*4+2] = EMS_MSK_BGD;
		}
		
		texMask[index*4+3] = project->parameter.transp;
	}
	
	delete [] maskc;
	delete [] scaled;
	
	// prepare texture map for film mask display
	if ( glIsTexture(texMaskName) )
		glDeleteTextures(1, &texMaskName);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(1, &texMaskName);
	glBindTexture(GL_TEXTURE_2D, texMaskName);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, EMS_FILM_TEX, EMS_FILM_TEX, 0, GL_RGBA, GL_UNSIGNED_BYTE, texMask);
	
	redraw();
}


void EMS_Display::setTempTexture(void)
{
	int dimX, dimY, dimZ, x, y, z, dim;
	long page, index, index2, mapSize, frameSize;
	long texSize = EMS_TEMP_TEX * EMS_TEMP_TEX;
	GLubyte *ptk_c, *pwr_c;
	float *ptk, *pwr;
	double ptk_ave, ptk_std, ptk_var, ptk_del, ptk_val;
	double pwr_ave, pwr_std, pwr_var, pwr_del, pwr_val;
	double xc, yc, r2, lcf_rd, lcf_r2;
	double scf_rd, scf_r2, scf_rcd, scf_rc2;
	static int dimZ_previous=0;

	dimX = (project->templates.dimX / 4) * 4;
	dimY = (project->templates.dimY / 4) * 4;
	dimZ = project->templates.dimZ;
	frameSize = project->templates.dimX * project->templates.dimY;

	mapSize = dimX * dimY;
	dim = (dimX <= dimY)? dimX : dimY;

	xc = (double)(dimX / 2);
	yc = (double)(dimY / 2);
	
	lcf_rd = 0.5 * dim * project->parameter.lcfMaskRad;
	scf_rd = 0.5 * dim * project->parameter.scfMaskRad;
	scf_rcd = 0.5 * dim * project->parameter.scfMaskRadc;
	
	lcf_r2 = lcf_rd * lcf_rd;
	scf_r2 = scf_rd * scf_rd;
	scf_rc2 = scf_rcd * scf_rcd;

	// storing one masked template in RGB
	ptk_c = new GLubyte[mapSize * 3];
	pwr_c = new GLubyte[mapSize * 3];
	
	// prepare texture map for template image display
	// ptk[] and pwr[] are interleavingly stored, (2*dimZ) in total
	if ( (dimZ_previous>0) && glIsTexture(texTempName[0]) ) {
		glDeleteTextures(dimZ_previous * 2, texTempName);
		delete [] texTempName;
	}

	texTempName = new GLuint[dimZ * 2];
	dimZ_previous = dimZ;

	if ( texTemp != NULL ) delete [] texTemp;
	texTemp = new GLubyte[texSize * 6 * dimZ];

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(dimZ * 2, texTempName);

	for ( z=0; z<dimZ; z++ ) {
		page = z * frameSize;
		ptk = project->refer.ptkStack + page;
		pwr = project->refer.pwrStack + page;
		
		ptk_ave = ptk_var = ptk_std = 0.0;
		pwr_ave = pwr_var = pwr_std = 0.0;

		for ( index=0; index<frameSize; index++ ) {
			ptk_val = ptk[index];
			ptk_ave += ptk_val;
			ptk_var += ptk_val * ptk_val;

			pwr_val = pwr[index];
			pwr_ave += pwr_val;
			pwr_var += pwr_val * pwr_val;
		}
		
		ptk_ave /= frameSize;
		ptk_var /= frameSize;
		ptk_std = ptk_var - ptk_ave * ptk_ave;

		if ( ptk_std > 0.0 )
			ptk_std = sqrt(ptk_std);
		else
			ptk_std = 0.0;

		ptk_del = (1.0 - project->parameter.refCntr) * ptk_std * EMS_HSTG_SIGMA + EMS_ZERO;
		
		pwr_ave /= frameSize;
		pwr_var /= frameSize;
		pwr_std = pwr_var - pwr_ave * pwr_ave;

		if ( pwr_std > 0.0 )
			pwr_std = sqrt(pwr_std);
		else
			pwr_std = 0.0;

		pwr_del = (1.0 - project->parameter.refCntr) * pwr_std * EMS_HSTG_SIGMA + EMS_ZERO;

		for ( y=0; y<dimY; y++ ) {
			for ( x=0; x<dimX; x++ ) {
				
				index = y * dimX + x;
				index2 = y * project->templates.dimX + x;

				ptk_val = 255 * ((ptk[index2] - ptk_ave) / ptk_del + project->parameter.refBrit);
				pwr_val = 255 * ((pwr[index2] - pwr_ave) / pwr_del + project->parameter.refBrit);

				// for particle images, arbitrary shape
				if ( ptk_val < 0 ) {
					ptk_c[3*index+0] = 0;
					ptk_c[3*index+1] = 0;
					ptk_c[3*index+2] = 0;
				}
				else if ( ptk_val > 255 ) {
					ptk_c[3*index+0] = 255;
					ptk_c[3*index+1] = 255;
					ptk_c[3*index+2] = 255;
				}
				else {
					ptk_c[3*index+0] = (GLubyte)ptk_val;
					ptk_c[3*index+1] = (GLubyte)ptk_val;
					ptk_c[3*index+2] = (GLubyte)ptk_val;
				}

				// capable displaying either L1 or L2 mask
				if ( (project->refer.maskID==EMS_MASK_L1) && (project->refer.L1Mask[page+index2]==EMS_FALSE) ) {
					ptk_c[3*index+0] = EMS_TEMP_MSKR;
					ptk_c[3*index+1] = EMS_TEMP_MSKG;
					ptk_c[3*index+2] = EMS_TEMP_MSKB;
				}
				else if ( (project->refer.maskID==EMS_MASK_L2) && (project->refer.L2Mask[page+index2]==EMS_FALSE) ) {
					ptk_c[3*index+0] = EMS_TEMP_MSKR;
					ptk_c[3*index+1] = EMS_TEMP_MSKG;
					ptk_c[3*index+2] = EMS_TEMP_MSKB;
				}

				// for auto-correlation images, circular only
				r2 = (x - xc) * (x - xc) + (y - yc) * (y - yc);

				if ( (r2 < scf_rc2) || (r2 > scf_r2) ) {
					pwr_c[3*index+0] = EMS_TEMP_MSKR;
					pwr_c[3*index+1] = EMS_TEMP_MSKG;
					pwr_c[3*index+2] = EMS_TEMP_MSKB;
				}
				else if ( pwr_val < 0 ) {
					pwr_c[3*index+0] = 0;
					pwr_c[3*index+1] = 0;
					pwr_c[3*index+2] = 0;
				}
				else if ( pwr_val > 255 ) {
					pwr_c[3*index+0] = 255;
					pwr_c[3*index+1] = 255;
					pwr_c[3*index+2] = 255;
				}
				else {
					pwr_c[3*index+0] = (GLubyte)pwr_val;
					pwr_c[3*index+1] = (GLubyte)pwr_val;
					pwr_c[3*index+2] = (GLubyte)pwr_val;
				}
			}
		}

		// setup ptk[] texture
		page = texSize * z * 6;
		gluScaleImage(GL_RGB, dimX, dimY, GL_UNSIGNED_BYTE, ptk_c, EMS_TEMP_TEX, EMS_TEMP_TEX, GL_UNSIGNED_BYTE, texTemp+page);
		
		glBindTexture(GL_TEXTURE_2D, texTempName[2 * z]);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, EMS_TEMP_TEX, EMS_TEMP_TEX, 0, GL_RGB, GL_UNSIGNED_BYTE, texTemp+page);

		// setup pwr[] texture
		page = texSize * z * 6 + texSize * 3;
		gluScaleImage(GL_RGB, dimX, dimY, GL_UNSIGNED_BYTE, pwr_c, EMS_TEMP_TEX, EMS_TEMP_TEX, GL_UNSIGNED_BYTE, texTemp+page);
		
		glBindTexture(GL_TEXTURE_2D, texTempName[2 * z + 1]);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, EMS_TEMP_TEX, EMS_TEMP_TEX, 0, GL_RGB, GL_UNSIGNED_BYTE, texTemp+page);
	}
	
	delete [] ptk_c;
	delete [] pwr_c;

	redraw();
}


void EMS_Display::setRefrTexture(void)
{
	int x, y, dimX, dimY, dim;
	long index, mapSize, frameSize;
	// "ptk" names used here are inherited from "template" function
	// "frame" label might be more intuitive
	GLubyte *ptk_c;
	float *ptk;
	double ptk_ave, ptk_std, ptk_var, ptk_del, ptk_val;

	if ( (project->refer.editFrame<0) || (project->refer.editFrame>=project->parameter.tmpN) ) {
		project->refer.editFrame = -1;
		errReport("The specified frame does not exist!");
		EMS_DP->btnEditFrame->value(0);
		return;
	}
	
	// convert to 4x size to satisfy texture mapping
	dimX = (project->templates.dimX / 4) * 4;
	dimY = (project->templates.dimY / 4) * 4;
	frameSize = project->templates.dimX * project->templates.dimY;
	dim = (dimX <= dimY)? dimX : dimY;
	mapSize = dimX * dimY;
	
	// storing one masked template in RGB
	ptk_c = new GLubyte[mapSize * 3];
	
	ptk = project->refer.ptkStack + project->refer.editFrame * frameSize;
	ptk_ave = ptk_var = ptk_std = 0.0;
	
	for ( index=0; index<frameSize; index++ ) {
		ptk_val = ptk[index];
		ptk_ave += ptk_val;
		ptk_var += ptk_val * ptk_val;
	}
	
	ptk_ave /= frameSize;
	ptk_var /= frameSize;
	ptk_std = ptk_var - ptk_ave * ptk_ave;

	if ( ptk_std > 0.0 )
		ptk_std = sqrt(ptk_std);
	else
		ptk_std = 0.0;

	ptk_del = (1.0 - project->parameter.refCntr) * ptk_std * EMS_HSTG_SIGMA + EMS_ZERO;
	
	for ( y=0; y<dimY; y++ ) {
		for ( x=0; x<dimX; x++ ) {
			
			index = y * dimX + x;
			ptk_val = 255 * ((ptk[y*project->templates.dimX+x] - ptk_ave) / ptk_del + project->parameter.refBrit);
			
			if ( ptk_val < 0 ) {
				ptk_c[3*index+0] = 0;
				ptk_c[3*index+1] = 0;
				ptk_c[3*index+2] = 0;
			}
			else if ( ptk_val > 255 ) {
				ptk_c[3*index+0] = 255;
				ptk_c[3*index+1] = 255;
				ptk_c[3*index+2] = 255;
			}
			else {
				ptk_c[3*index+0] = (GLubyte)ptk_val;
				ptk_c[3*index+1] = (GLubyte)ptk_val;
				ptk_c[3*index+2] = (GLubyte)ptk_val;
			}
		}
	}

	// setup texture map
	gluScaleImage(GL_RGB, dimX, dimY, GL_UNSIGNED_BYTE, ptk_c, EMS_RFED_TEX, EMS_RFED_TEX, GL_UNSIGNED_BYTE, texRefr);

	if ( glIsTexture(texRefrName) )
		glDeleteTextures(1, &texRefrName);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(1, &texRefrName);
	glBindTexture(GL_TEXTURE_2D, texRefrName);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, EMS_RFED_TEX, EMS_RFED_TEX, 0, GL_RGB, GL_UNSIGNED_BYTE, texRefr);

	delete [] ptk_c;

	// setup display scaling
	project->refer.scale = (float)(1.0 * EMS_RFED_TEX / dim);

	redraw();
}


void EMS_Display::setStackTexture(void)
{
	int dimX, dimY, dimZ, x, y, z;
	int frameTex, frame0, frame1, maxFrame;
	long page, index, mapSize, frameSize, texSize;
	// "ptk" names used here are inherited from "template" function
	// "frame" label might be more intuitive
	GLubyte *ptk_c;
	float *ptk;
	double ptk_ave, ptk_std, ptk_del, ptk_val;
	static int dimZ_previous=0;

	dimX = (functImage->param.imgX / 4) * 4;
	dimY = (functImage->param.imgY / 4) * 4;
	frameSize = functImage->frameSize;

	frame0 = functImage->frame0;
	frame1 = functImage->param.imgN - 1;
	frameTex = functImage->textSize;
	texSize = frameTex * frameTex;

	// display one page at the most
	maxFrame = (winX / (dimX/functImage->imgScale+EMS_STACK_GAP)) * (winY / (dimY/functImage->imgScale+EMS_STACK_GAP));
	if ( maxFrame < 1 ) maxFrame = 1;
	functImage->pageSize = maxFrame;

	if ( (frame1-frame0+1) > maxFrame ) frame1 = frame0 + maxFrame - 1;
	functImage->frame1 = frame1;
	dimZ = frame1 - frame0 + 1;

	// storing image frame in RGB[]
	mapSize = dimX * dimY;
	ptk_c = new GLubyte[mapSize * 3];
	
	// prepare texture map for image stack display
	if ( (dimZ_previous>0) && glIsTexture(texStackName[0]) ) {
		glDeleteTextures(dimZ_previous, texStackName);
		delete [] texStackName;
	}

	texStackName = new GLuint[dimZ];
	dimZ_previous = dimZ;

	if ( texStack != NULL ) delete [] texStack;
	texStack = new GLubyte[texSize * 3 * dimZ];

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(dimZ, texStackName);

	for ( z=frame0; z<=frame1; z++ ) {
		page = z * frameSize;
		ptk_ave = functImage->imgStack.aveInt;
		ptk_std = functImage->imgStack.stdInt;
		ptk_del = (1.0 - functImage->imgCntr) * ptk_std * EMS_HSTG_SIGMA + EMS_ZERO;
		ptk = functImage->imgStack.density + page;
		
		for ( y=0; y<dimY; y++ ) {
			for ( x=0; x<dimX; x++ ) {
				
				index = y * dimX + x;
				ptk_val = 255 * ((ptk[y*functImage->param.imgX+x] - ptk_ave) / ptk_del + functImage->imgBrit);

				// for particle images
				if ( ptk_val < 0 ) {
					ptk_c[3*index+0] = 0;
					ptk_c[3*index+1] = 0;
					ptk_c[3*index+2] = 0;
				}
				else if ( ptk_val > 255 ) {
					ptk_c[3*index+0] = 255;
					ptk_c[3*index+1] = 255;
					ptk_c[3*index+2] = 255;
				}
				else {
					ptk_c[3*index+0] = (GLubyte)ptk_val;
					ptk_c[3*index+1] = (GLubyte)ptk_val;
					ptk_c[3*index+2] = (GLubyte)ptk_val;
				}
			}
		}

		// setup frame texture
		page = texSize * (z-frame0) * 3;
		gluScaleImage(GL_RGB, dimX, dimY, GL_UNSIGNED_BYTE, ptk_c, frameTex, frameTex, GL_UNSIGNED_BYTE, texStack+page);
		
		glBindTexture(GL_TEXTURE_2D, texStackName[z-frame0]);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, frameTex, frameTex, 0, GL_RGB, GL_UNSIGNED_BYTE, texStack+page);
	}
	
	delete [] ptk_c;
	
	redraw();
}


void EMS_Display::setInspectFocus(int x, int y, char flagConvert)
{
	double scaling;

	// (x, y) is in film[] coordinates
	if ( flagConvert == 0 ) {
		focusX = x;
		focusY = y;
	}

	// (x, y) is in dispFilm[] coordinates
	else {
		scaling = 1.0 * project->parameter.dispScal / project->parameter.downSize;
		focusX = (int)floor(((x - winX/2) / zoom - tranX + filmX/2) * scaling + 0.5);
		focusY = (int)floor(((y - winY/2) / zoom - tranY + filmY/2) * scaling + 0.5);
	}

	// display in monitors
	if ( (x>=0) && (y>=0) && (project->sysStatus.emsdpMode==DP_PARTICLE_MODE) ) {
		cPanel->monitorPrtk();
	}

	redraw();
}


inline void EMS_Display::drawFilm(void)
{
	double x0, x1, y0, y1;

	// draw FILM only in MICROGRAPH / PARTICLE / FILAMENT modes
	if ( project->dispFilm.exist == EMS_FALSE ) return;
	if ( ( project->sysStatus.emsdpMode != DP_MCRGRAPH_MODE ) &&
		 ( project->sysStatus.emsdpMode != DP_PARTICLE_MODE ) &&
		 ( project->sysStatus.emsdpMode != DP_2DMIMAGE_MODE ) ) return;

	x0 = tranX - filmX/2;
	x1 = tranX + filmX/2;
	y0 = tranY - filmY/2;
	y1 = tranY + filmY/2;

	glEnable(GL_TEXTURE_2D);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
	glBindTexture(GL_TEXTURE_2D, texFilmName);
	
	glBegin(GL_QUADS);
		glTexCoord2d(0.0, 0.0); glVertex3d(x0, y0, EMS_FILM_GLD);
		glTexCoord2d(0.0, 1.0); glVertex3d(x0, y1, EMS_FILM_GLD);
		glTexCoord2d(1.0, 1.0); glVertex3d(x1, y1, EMS_FILM_GLD);
		glTexCoord2d(1.0, 0.0); glVertex3d(x1, y0, EMS_FILM_GLD);
	glEnd();
	
	glDisable(GL_TEXTURE_2D);
}


inline void EMS_Display::drawTemp(void)
{
	int dimX, dimY, dimZ, x, y, z;
	long index, frameSize;
	double x0, x1, y0, y1, xt, yt;
	double lnx0, lny0, lnx1, lny1, ang;
	char highlit, *msk;

	// draw templates and auto-corrs only in the TEMPLATE mode
	if ( project->sysStatus.emsdpMode != DP_TEMPLATE_MODE ) return;
	if ( project->templates.exist == EMS_FALSE ) return;
	if ( texTempName == NULL ) return;

//	dimX = project->templates.dimX;
//	dimY = project->templates.dimY;
	dimX = (project->templates.dimX / 4) * 4;
	dimY = (project->templates.dimY / 4) * 4;
	dimZ = project->templates.dimZ;
	frameSize = project->templates.dimX*project->templates.dimY;

	if ( dimX >= (winX-EMS_TEMP_GAP) ) {
		errReport("Template is too large to be displayed!");
		return;
	}

	// draw the selected frame only under editing mode
	if ( (project->sysStatus.editTemp) && (project->refer.editFrame>=0) ) {

		x0 = -0.5 * dimX * project->refer.scale;
		y0 = -0.5 * dimY * project->refer.scale;
		x1 = +0.5 * dimX * project->refer.scale;
		y1 = +0.5 * dimY * project->refer.scale;

		// drawing template image
		glEnable(GL_TEXTURE_2D);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
		glBindTexture(GL_TEXTURE_2D, texRefrName);

		glBegin(GL_QUADS);
			glTexCoord2d(0.0, 0.0); glVertex3d(x0, y0, EMS_TEMP_GLD);
			glTexCoord2d(0.0, 1.0); glVertex3d(x0, y1, EMS_TEMP_GLD);
			glTexCoord2d(1.0, 1.0); glVertex3d(x1, y1, EMS_TEMP_GLD);
			glTexCoord2d(1.0, 0.0); glVertex3d(x1, y0, EMS_TEMP_GLD);
		glEnd();

		glDisable(GL_TEXTURE_2D);

		// drawing template mask
		if ( project->refer.maskID == EMS_MASK_L1 )
			msk = project->refer.L1Mask;
		else if ( project->refer.maskID == EMS_MASK_L2 )
			msk = project->refer.L2Mask;

		if ( project->sysStatus.showMask ) {
			
			glEnable(GL_BLEND);
			glBlendFunc(GL_SRC_ALPHA, GL_SRC_ALPHA);
			glColor4d(EMS_TEMP_MSKR, EMS_TEMP_MSKG, EMS_TEMP_MSKB, project->parameter.transp);
			
			for ( yt=y0; yt<y1; yt+=1.0 ) {
				y = (int)(yt/project->refer.scale + dimY/2);
				for ( xt=x0; xt<x1; xt+=1.0 ) {
					x = (int)(xt/project->refer.scale + dimX/2);
					index = project->refer.editFrame*frameSize + y*project->templates.dimX + x;
					if ( msk[index] == EMS_TRUE ) continue;
					
					glBegin(GL_POINTS);
						glVertex3d(xt, yt, EMS_MASK_GLD);
					glEnd();
				}
			}
			
			glDisable(GL_BLEND);
			
			// line drawing if necessary
			if ( project->sysStatus.drawTools == EMS_LINES ) {
				
				if ( (lineX0>=0) && (lineY0>=0) && (lineX1>=0) && (lineY1>=0) ) {
					lnx0 = lineX0 - winX/2.0;
					lny0 = lineY0 - winY/2.0;
					lnx1 = lineX1 - winX/2.0;
					lny1 = lineY1 - winY/2.0;
					
					glColor3d(EMS_MSKLN_R, EMS_MSKLN_G, EMS_MSKLN_B);
					glLineWidth((float)1.0);
					
					glBegin(GL_LINES);
						glVertex3d(lnx0, lny0, EMS_LINE_GLD);
						glVertex3d(lnx1, lny1, EMS_LINE_GLD);
					glEnd();
				}
			}
			
			// circle drawing if necessary
			else if ( project->sysStatus.drawTools == EMS_CIRCLE ) {
				
				glColor3d(EMS_MSKLN_R, EMS_MSKLN_G, EMS_MSKLN_B);
				glLineWidth((float)2.0);
				
				x0 = circR + circX - winX/2.0;
				y0 = circY - winY/2.0;
				
				for ( ang=0.01; ang<6.30; ang+=0.05 ) {
					x1 = circR * cos(ang) + circX - winX/2.0;
					y1 = circR * sin(ang) + circY - winY/2.0;
					
					glBegin(GL_LINES);
						glVertex3d(x0, y0, EMS_LINE_GLD);
						glVertex3d(x1, y1, EMS_LINE_GLD);
					glEnd();
					
					x0 = x1; y0 = y1;
				}
			}
		}
		
		// overlay marker of alignment center
		if ( project->sysStatus.showPrtk ) {
			xt = project->refer.xc[project->refer.editFrame] - project->templates.dimX/2;
			yt = project->refer.yc[project->refer.editFrame] - project->templates.dimY/2;
			xt *= project->refer.scale;
			yt *= project->refer.scale;
			
			glLineWidth((float)EMS_MARK_LINE);
			glColor3d(EMS_MKOUI_R, EMS_MKOUI_G, EMS_MKOUI_B);
			
			glBegin(GL_LINES);
				glVertex3d(xt-EMS_MARK_SIZE*2, yt, EMS_PEAK_GLD);
				glVertex3d(xt+EMS_MARK_SIZE*2, yt, EMS_PEAK_GLD);
			glEnd();
			
			glBegin(GL_LINES);
				glVertex3d(xt, yt-EMS_MARK_SIZE*2, EMS_PEAK_GLD);
				glVertex3d(xt, yt+EMS_MARK_SIZE*2, EMS_PEAK_GLD);
			glEnd();
		}

		return;
	}

	// otherwise, draw all reference frame as a montage
	x0 = -winX/2 + EMS_TEMP_GAP;
	y0 = tranY - winY/2 + EMS_TEMP_GAP;
	highlit = EMS_FALSE;

	glEnable(GL_TEXTURE_2D);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

	// draw template image frame by frame
	for ( z=0; z<dimZ; z++ ) {

		if ( (x0+dimX) > winX/2 ) {
			x0 = -winX/2 + EMS_TEMP_GAP;
			y0 += 2 * dimY + EMS_TEMP_GAP + 1;
		}

		if ( z == project->refer.editFrame ) {
			xt = x0;
			yt = y0;
			highlit = EMS_TRUE;
		}

		x1 = x0 + dimX;
		y1 = y0 + dimY;
//		x1 = x0 + dimX - 1;
//		y1 = y0 + dimY - 1;

		// texture of ptk[]
		glBindTexture(GL_TEXTURE_2D, texTempName[2*z]);
		glBegin(GL_QUADS);
			glTexCoord2d(0.0, 0.0); glVertex3d(x0, y0, EMS_TEMP_GLD);
			glTexCoord2d(0.0, 1.0); glVertex3d(x0, y1, EMS_TEMP_GLD);
			glTexCoord2d(1.0, 1.0); glVertex3d(x1, y1, EMS_TEMP_GLD);
			glTexCoord2d(1.0, 0.0); glVertex3d(x1, y0, EMS_TEMP_GLD);
		glEnd();

		// texture of pwr[]
		glBindTexture(GL_TEXTURE_2D, texTempName[2*z+1]);
		glBegin(GL_QUADS);
			glTexCoord2d(0.0, 0.0); glVertex3d(x0, y0+dimY+1, EMS_TEMP_GLD);
			glTexCoord2d(0.0, 1.0); glVertex3d(x0, y1+dimY+1, EMS_TEMP_GLD);
			glTexCoord2d(1.0, 1.0); glVertex3d(x1, y1+dimY+1, EMS_TEMP_GLD);
			glTexCoord2d(1.0, 0.0); glVertex3d(x1, y0+dimY+1, EMS_TEMP_GLD);
		glEnd();

		x0 += dimX + EMS_TEMP_GAP;
	}
	
	glDisable(GL_TEXTURE_2D);

	// highlight recently edited frame
	if ( highlit == EMS_TRUE ) {
		glLineWidth((float)EMS_FOCUS_LINE);
		glColor3d(0.0, 1.0, 1.0);

		glBegin(GL_LINES);
			glVertex3d(xt, yt, EMS_SELC_GLD);
			glVertex3d(xt+dimX, yt, EMS_SELC_GLD);
		glEnd();

		glBegin(GL_LINES);
			glVertex3d(xt+dimX, yt, EMS_SELC_GLD);
			glVertex3d(xt+dimX, yt+2*dimY+1, EMS_SELC_GLD);
		glEnd();

		glBegin(GL_LINES);
			glVertex3d(xt+dimX, yt+2*dimY+1, EMS_SELC_GLD);
			glVertex3d(xt, yt+2*dimY+1, EMS_SELC_GLD);
		glEnd();

		glBegin(GL_LINES);
			glVertex3d(xt, yt+2*dimY+1, EMS_SELC_GLD);
			glVertex3d(xt, yt, EMS_SELC_GLD);
		glEnd();
	}

	glColor3d(1.0, 1.0, 0.0);
	x0 = -winX/2 + EMS_TEMP_GAP;
	y0 = tranY - winY/2 + EMS_TEMP_GAP;

	for ( z=0; z<dimZ; z++ ) {

		if ( (x0+dimX) > winX/2 ) {
			x0 = -winX/2 + EMS_TEMP_GAP;
			y0 += 2 * dimY + EMS_TEMP_GAP;
		}

		x1 = x0 + winX/2 + 2;
		y1 = winY/2 - y0 - dimY - 4;
		x0 += dimX + EMS_TEMP_GAP;

		sprintf(message, "%d", z+1);
		setLabel((int)x1, (int)y1, message);

	}
}


inline void EMS_Display::drawStack(void)
{
	int dimX, dimY, frame0, frame1, z;
	double x0, x1, y0, y1, xt, yt;

	// draw templates and auto-corrs only in the TEMPLATE mode
	if ( project->sysStatus.emsdpMode != DP_FUNCIMAG_MODE ) return;
	if ( functImage->imgStack.exist == EMS_FALSE ) return;
	if ( texStackName == NULL ) return;

	dimX = functImage->param.imgX / functImage->imgScale;
	dimY = functImage->param.imgY / functImage->imgScale;

	frame0 = functImage->frame0;
	frame1 = functImage->frame1;
	x0 = EMS_STACK_GAP - winX/2;
	y0 = EMS_STACK_GAP - winY/2;

	glEnable(GL_TEXTURE_2D);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

	for ( z=frame0; z<=frame1; z++ ) {

		if ( (x0+dimX) > winX/2 ) {
			x0 = EMS_STACK_GAP - winX/2;
			y0 += EMS_STACK_GAP + dimY;

			if ( y0 > (winY/2) ) {
				frame1 = z - 1;
				functImage->frame1 = frame1;
				break;
			}
		}

		x1 = x0 + dimX;
		y1 = y0 + dimY;
//		x1 = x0 + dimX - 1;
//		y1 = y0 + dimY - 1;

		// texture of image frame
		glBindTexture(GL_TEXTURE_2D, texStackName[z-frame0]);
		glBegin(GL_QUADS);
			glTexCoord2d(0.0, 0.0); glVertex3d(x0, y0, EMS_TEMP_GLD);
			glTexCoord2d(0.0, 1.0); glVertex3d(x0, y1, EMS_TEMP_GLD);
			glTexCoord2d(1.0, 1.0); glVertex3d(x1, y1, EMS_TEMP_GLD);
			glTexCoord2d(1.0, 0.0); glVertex3d(x1, y0, EMS_TEMP_GLD);
		glEnd();

		x0 += dimX + EMS_STACK_GAP;
	}
	
	glDisable(GL_TEXTURE_2D);

	// highlight selected image frames
	x0 = EMS_STACK_GAP - winX/2;
	y0 = EMS_STACK_GAP - winY/2;

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_SRC_ALPHA);
	glColor4d(0.0, 0.0, 1.0, EMS_FRAME_SELC);

	for ( z=frame0; z<=frame1; z++ ) {

		if ( (x0+dimX) > winX/2 ) {
			x0 = EMS_STACK_GAP - winX/2;
			y0 += EMS_STACK_GAP + dimY;
		}

		// detect frame selection
		if ( (focusX>=0) && (focusY>=0) ) {
			xt = focusX - x0 - winX/2;
			yt = focusY - y0 - winY/2;

			if ( (xt>=0) && (xt<=dimX) && (yt>=0) && (yt<=dimY) ) {
				if ( functImage->attrib[z].select == EMS_TRUE ) {
					functImage->attrib[z].select = EMS_FALSE;
					functImage->selCount --;
				}
				else {
					functImage->attrib[z].select = EMS_TRUE;
					functImage->selCount ++;
				}
			}
		}

		if ( functImage->attrib[z].select == EMS_TRUE ) {
			x1 = x0 + dimX;
			y1 = y0 + dimY;

			glBegin(GL_QUADS);
				glVertex3d(x0, y0, EMS_SELC_GLD);
				glVertex3d(x0, y1, EMS_SELC_GLD);
				glVertex3d(x1, y1, EMS_SELC_GLD);
				glVertex3d(x1, y0, EMS_SELC_GLD);
			glEnd();
		}

		x0 += dimX + EMS_STACK_GAP;
	}

	glDisable(GL_BLEND);

	// sequential ID labelling
	if ( functImage->labelDisp == EMS_TRUE ) {

		glColor3d(0.0, 1.0, 1.0);
		x0 = EMS_STACK_GAP - winX/2;
		y0 = EMS_STACK_GAP - winY/2;
		
		for ( z=frame0; z<=frame1; z++ ) {
			
			if ( (x0+dimX) > winX/2 ) {
				x0 = EMS_STACK_GAP - winX/2;
				y0 += EMS_STACK_GAP + dimY;
			}
			
			x1 = x0 + winX/2 + 2;
			y1 = winY/2 - y0 - dimY;
			x0 += dimX + EMS_STACK_GAP;
			
			sprintf(message, "%d", z+1);
			setLabel((int)x1, (int)y1, message);
		}
	}

	// clear frame selection focus
	focusX = focusY = -1;
}


inline void EMS_Display::drawDigitGel(void)
{
	int index;
	double x0, y0, x1, y1, xt, yt;

	// draw templates and auto-corrs only in the TEMPLATE mode
	if ( project->sysStatus.emsdpMode != DP_FUNCDGEL_MODE ) return;

	// draw mass histogram
	if ( functImage->gelBinMax > 0 ) {
		glLineWidth(1.0f);
		glColor3d(0.0, 1.0, 0.0);
		
		yt = EMS_GEL_VERTIC / 2.0;
		x0 = -EMS_GEL_RANGE * EMS_GEL_BINPIX;
		y0 = yt - 1.0 * EMS_GEL_VERTIC * functImage->gelHist[0] / functImage->gelBinMax;
		
		for ( index=1; index<EMS_GEL_BIN; index++ ) {
			x1 = (0.5 * index - EMS_GEL_RANGE) * EMS_GEL_BINPIX;
			y1 = yt - 1.0 * EMS_GEL_VERTIC * functImage->gelHist[index] / functImage->gelBinMax;
			
			glBegin(GL_LINES);
				glVertex3d(x0, y0, EMS_DGEL_GLD);
				glVertex3d(x1, y1, EMS_DGEL_GLD);
			glEnd();

			x0 = x1; y0 = y1;
		}
	}

	// draw L & H mass thresholds
	x0 = EMS_GEL_BINPIX * (functImage->gelLmass - functImage->gelMassAve) / functImage->gelMassStd;
	x1 = EMS_GEL_BINPIX * (functImage->gelHmass - functImage->gelMassAve) / functImage->gelMassStd;
	y0 = +EMS_GEL_VERTIC / 2.0;
	y1 = -EMS_GEL_VERTIC / 2.0 - 16;

	glLineWidth(2.0f);
	glColor3d(1.0, 0.0, 0.0);
	glBegin(GL_LINES);
		glVertex3d(x0, y0, EMS_DGEL_GLD);
		glVertex3d(x0, y1, EMS_DGEL_GLD);
	glEnd();

	glColor3d(0.0, 0.0, 1.0);
	glBegin(GL_LINES);
		glVertex3d(x1, y0, EMS_DGEL_GLD);
		glVertex3d(x1, y1, EMS_DGEL_GLD);
	glEnd();

	// draw horizontal axis
	glLineWidth(2.0f);
	glColor3d(1.0, 1.0, 0.0);

	x0 = -(EMS_GEL_RANGE+0.5) * EMS_GEL_BINPIX;
	x1 = +(EMS_GEL_RANGE+0.5) * EMS_GEL_BINPIX;
	y0 = y1 = EMS_GEL_VERTIC / 2;

	glBegin(GL_LINES);
		glVertex3d(x0, y0, EMS_DGEL_GLD);
		glVertex3d(x1, y1, EMS_DGEL_GLD);
	glEnd();

	// draw short bars for s.t.d. label
	y0 = EMS_GEL_VERTIC / 2 - 3;
	y1 = EMS_GEL_VERTIC / 2 + 3;
	yt = winY/2 - (int)y1 - 12;
	glLineWidth(1.0f);

	for ( index=(-EMS_GEL_RANGE); index<=(+EMS_GEL_RANGE); index++ ) {
		x0 = x1 = index * EMS_GEL_BINPIX;

		glBegin(GL_LINES);
			glVertex3d(x0, y0, EMS_DGEL_GLD);
			glVertex3d(x1, y1, EMS_DGEL_GLD);
		glEnd();

		if ( index < 0 ) {
			xt = (int)x1 + winX/2 - 8;
			sprintf(message, "%d", index);
		}
		else if ( index > 0 ) {
			xt = (int)x1 + winX/2 - 8;
			sprintf(message, "+%d", index);
		}
		else {
			xt = (int)x1 + winX/2 - 3;
			sprintf(message, "%d", index);
		}

		setLabel((int)xt, (int)yt, message);
	}

	xt = winX/2 + (1+EMS_GEL_RANGE) * EMS_GEL_BINPIX - 8;
	yt = (winY - EMS_GEL_VERTIC) / 2 - 3;

	glColor3d(1.0, 1.0, 0.0);
	setLabel((int)xt, (int)yt, "s.t.d.");
}


inline void EMS_Display::drawCharts(void)
{
	int index;
	double tick, phase_t;
	double x0, y0, x1, y1, xt, yt;

	// draw templates and auto-corrs only in the TEMPLATE mode
	if ( project->sysStatus.emsdpMode != DP_FUNCALGN_MODE ) return;

	// draw ALIGNMENT-CORRELATION histogram
	if ( functImage->ccBinMax > 0 ) {
		glLineWidth(1.0f);
		glColor3d(0.0, 1.0, 0.0);
		
		xt = fabs(EMS_CC_BASE_X0 - EMS_CC_BASE_X1);
		yt = fabs(EMS_CC_BASE_Y0 - EMS_CC_BASE_Y1);
		x0 = EMS_CC_BASE_X0;
		y0 = EMS_CC_BASE_Y0;
		
		for ( index=0; index<EMS_CC_BIN; index++ ) {
			x1 = EMS_CC_BASE_X0 + (1.0 * index / (EMS_CC_BIN-1)) * xt;
			y1 = EMS_CC_BASE_Y0 - yt * functImage->ccHist[index] / functImage->ccBinMax;
			
			glBegin(GL_LINES);
				glVertex3d(x0, y0, EMS_CHRT_GLD);
				glVertex3d(x1, y1, EMS_CHRT_GLD);
			glEnd();

			x0 = x1; y0 = y1;
		}
	}

	// draw ALIGNMENT-CORRELATION threshold
	if ( functImage->flagFrealign ) {
		phase_t = (functImage->phaseThres - EMS_PHASE_LOW) / (90.0 - EMS_PHASE_LOW);
		x0 = EMS_CC_BASE_X0 + 10 * phase_t * EMS_CC_TICK;
	}
	else
		x0 = EMS_CC_BASE_X0 + 10 * functImage->corrThres * EMS_CC_TICK;

	y0 = EMS_CC_BASE_Y0;
	y1 = EMS_CC_BASE_Y1;

	glLineWidth(2.0f);
	glColor3d(1.0, 0.0, 0.0);
	glBegin(GL_LINES);
		glVertex3d(x0, y0, EMS_CHRT_GLD);
		glVertex3d(x0, y1, EMS_CHRT_GLD);
	glEnd();

	// draw horizontal axis
	glLineWidth(2.0f);
	glColor3d(1.0, 1.0, 0.0);

	glBegin(GL_LINES);
		glVertex3d(EMS_CC_BASE_X0-20, EMS_CC_BASE_Y0, EMS_CHRT_GLD);
		glVertex3d(EMS_CC_BASE_X1+20, EMS_CC_BASE_Y0, EMS_CHRT_GLD);
	glEnd();

	// draw tick bars
	y0 = EMS_CC_BASE_Y0 - 3;
	y1 = EMS_CC_BASE_Y0 + 3;
	yt = winY/2 - EMS_CC_BASE_Y0 - 16;
	glLineWidth(1.0f);

	// correlation coefficient distribution
	if ( functImage->flagFrealign == EMS_FALSE ) {
		for ( tick=0.0; tick<1.1; tick+=0.2 ) {
			x0 = x1 = EMS_CC_BASE_X0 + 10 * tick * EMS_CC_TICK;
			xt = winX/2 + x0 - 10;
			
			glBegin(GL_LINES);
			glVertex3d(x0, y0, EMS_CHRT_GLD);
			glVertex3d(x1, y1, EMS_CHRT_GLD);
			glEnd();
			
			sprintf(message, "%3.1lf", tick);
			setLabel((int)xt, (int)yt, message);
		}
		
		xt = winX/2 + (EMS_CC_BASE_X0+EMS_CC_BASE_X1)/2 - 90;
		yt = winY/2 - EMS_CC_BASE_Y1 + 30;
		setLabel((int)xt, (int)yt, "Correlation Coefficient");
	}

	// phase residual distribution
	else {
		for ( tick=0.0; tick<1.1; tick+=0.2 ) {
			x0 = x1 = EMS_CC_BASE_X0 + 10 * tick * EMS_CC_TICK;
			xt = winX/2 + x0 - 15;

			glBegin(GL_LINES);
			glVertex3d(x0, y0, EMS_CHRT_GLD);
			glVertex3d(x1, y1, EMS_CHRT_GLD);
			glEnd();

			sprintf(message, "%3d", (int)(EMS_PHASE_LOW+tick*(90-EMS_PHASE_LOW)));
			setLabel((int)xt, (int)yt, message);
		}
		
		xt = winX/2 + (EMS_CC_BASE_X0+EMS_CC_BASE_X1)/2 - 70;
		yt = winY/2 - EMS_CC_BASE_Y1 + 30;
		setLabel((int)xt, (int)yt, "Phase Residual");
	}

	// draw ALIGNMENT-SHIFT histogram
	if ( functImage->dxBinMax > 0 ) {
		glLineWidth(1.0f);
		glColor3d(0.0, 1.0, 0.0);
		
		xt = fabs(EMS_DX_BASE_X0 - EMS_DX_BASE_X1);
		yt = fabs(EMS_DX_BASE_Y0 - EMS_DX_BASE_Y1);
		x0 = EMS_DX_BASE_X0;
		y0 = EMS_DX_BASE_Y0;
		
		for ( index=0; index<EMS_DX_BIN; index++ ) {
			x1 = EMS_DX_BASE_X0 + (1.0 * index / (EMS_DX_BIN-1)) * xt;
			y1 = EMS_DX_BASE_Y0 - yt * functImage->dxHist[index] / functImage->dxBinMax;
			
			glBegin(GL_LINES);
				glVertex3d(x0, y0, EMS_CHRT_GLD);
				glVertex3d(x1, y1, EMS_CHRT_GLD);
			glEnd();

			x0 = x1; y0 = y1;
		}
	}

	// draw ALIGNMENT-SHIFT threshold
	if ( functImage->flagFrealign )
		x0 = EMS_DX_BASE_X0 + 10 * functImage->falgnThres * EMS_DX_TICK;
	else
		x0 = EMS_DX_BASE_X0 + 10 * functImage->shiftThres * EMS_DX_TICK;

	y0 = EMS_DX_BASE_Y0;
	y1 = EMS_DX_BASE_Y1;

	glLineWidth(2.0f);
	glColor3d(0.0, 0.0, 1.0);
	glBegin(GL_LINES);
		glVertex3d(x0, y0, EMS_CHRT_GLD);
		glVertex3d(x0, y1, EMS_CHRT_GLD);
	glEnd();

	// draw horizontal axis
	glLineWidth(2.0f);
	glColor3d(1.0, 1.0, 0.0);

	glBegin(GL_LINES);
		glVertex3d(EMS_DX_BASE_X0-20, EMS_DX_BASE_Y0, EMS_CHRT_GLD);
		glVertex3d(EMS_DX_BASE_X1+20, EMS_DX_BASE_Y0, EMS_CHRT_GLD);
	glEnd();

	// draw tick bars
	y0 = EMS_DX_BASE_Y0 - 3;
	y1 = EMS_DX_BASE_Y0 + 3;
	yt = winY/2 - EMS_DX_BASE_Y0 - 16;
	glLineWidth(1.0f);

	for ( tick=0.0; tick<1.1; tick+=0.2 ) {
		x0 = x1 = EMS_DX_BASE_X0 + 10 * tick * EMS_DX_TICK;
		xt = winX/2 + x0 - 10;

		glBegin(GL_LINES);
			glVertex3d(x0, y0, EMS_CHRT_GLD);
			glVertex3d(x1, y1, EMS_CHRT_GLD);
		glEnd();

		sprintf(message, "%3.1lf", tick);
		setLabel((int)xt, (int)yt, message);
	}

	xt = winX/2 + (EMS_DX_BASE_X0+EMS_DX_BASE_X1)/2 - 70;
	yt = winY/2 - EMS_DX_BASE_Y1 + 30;
	setLabel((int)xt, (int)yt, "Alignment Shift");
}


inline void EMS_Display::drawMask(void)
{
	int x, y;
	double x0, x1, y0, y1, lnx0, lny0, lnx1, lny1, ang;
	
	// draw mask only in MCRGRAPH / PARTICLE / FILAMENT modes
	if ( project->dispMask.exist == EMS_FALSE ) return;
	if ( project->sysStatus.showMask == EMS_FALSE ) return;
	if ( ( project->sysStatus.emsdpMode != DP_MCRGRAPH_MODE ) &&
		 ( project->sysStatus.emsdpMode != DP_PARTICLE_MODE ) &&
		 ( project->sysStatus.emsdpMode != DP_2DMIMAGE_MODE ) ) return;
	
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_SRC_ALPHA);
	glColor4d(EMS_MSK_R, EMS_MSK_G, EMS_MSK_B, project->parameter.transp);
	
	if ( project->sysStatus.drawTools != EMS_FALSE ) {
		
		for ( y=0; y<filmY; y++ ) {
			for ( x=0; x<filmX; x++ ) {
				
				if ( project->dispMask.dataC[y*filmX+x] == EMS_TRUE ) {
					x0 = tranX - filmX/2 + x;
					y0 = tranY - filmY/2 + y;
					
					glBegin(GL_POINTS);
						glVertex3d(x0, y0, EMS_MASK_GLD);
					glEnd();
				}
			}
		}
	}
	else {
		x0 = tranX - filmX/2;
		x1 = tranX + filmX/2;
		y0 = tranY - filmY/2;
		y1 = tranY + filmY/2;
		
		glEnable(GL_TEXTURE_2D);
		glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
		glBindTexture(GL_TEXTURE_2D, texMaskName);
		
		glBegin(GL_QUADS);
			glTexCoord2d(0.0, 0.0); glVertex3d(x0, y0, EMS_MASK_GLD);
			glTexCoord2d(0.0, 1.0); glVertex3d(x0, y1, EMS_MASK_GLD);
			glTexCoord2d(1.0, 1.0); glVertex3d(x1, y1, EMS_MASK_GLD);
			glTexCoord2d(1.0, 0.0); glVertex3d(x1, y0, EMS_MASK_GLD);
		glEnd();
		
		glDisable(GL_TEXTURE_2D);
	}
	
	glDisable(GL_BLEND);

	// line drawing if necessary
	if ( project->sysStatus.drawTools == EMS_LINES ) {

		if ( (lineX0>=0) && (lineY0>=0) && (lineX1>=0) && (lineY1>=0) ) {
			lnx0 = (lineX0 - winX/2.0) / zoom;
			lny0 = (lineY0 - winY/2.0) / zoom;
			lnx1 = (lineX1 - winX/2.0) / zoom;
			lny1 = (lineY1 - winY/2.0) / zoom;
			
			glColor3d(EMS_MSKLN_R, EMS_MSKLN_G, EMS_MSKLN_B);
			glLineWidth((float)1.0);
			
			glBegin(GL_LINES);
				glVertex3d(lnx0, lny0, EMS_LINE_GLD);
				glVertex3d(lnx1, lny1, EMS_LINE_GLD);
			glEnd();
		}
	}

	// circle drawing if necessary
	else if ( project->sysStatus.drawTools == EMS_CIRCLE ) {

		glColor3d(EMS_MSKLN_R, EMS_MSKLN_G, EMS_MSKLN_B);
		glLineWidth((float)2.0);

		x0 = (circR + circX - winX/2.0) / zoom;
		y0 = (circY - winY/2.0) / zoom;

		for ( ang=0.01; ang<6.29; ang+=0.01 ) {
			x1 = (circR * cos(ang) + circX - winX/2.0) / zoom;
			y1 = (circR * sin(ang) + circY - winY/2.0) / zoom;

			glBegin(GL_LINES);
				glVertex3d(x0, y0, EMS_LINE_GLD);
				glVertex3d(x1, y1, EMS_LINE_GLD);
			glEnd();

			x0 = x1; y0 = y1;
		}
	}
}


inline void EMS_Display::drawPeak(void)
{
	int x, y;
	long index;
	
	// draw PEAK only in PARTICLE / FILAMENT modes
	if ( (project->sysStatus.emsdpMode!=DP_PARTICLE_MODE) &&
		 (project->sysStatus.emsdpMode!=DP_2DMIMAGE_MODE) ) return;
	if ( project->sysStatus.showPrtk == EMS_FALSE ) return;
	if ( project->parameter.totCount <= 0 ) return;
	if ( project->parameter.filaCount > 0 ) return;
	
	for ( y=0; y<project->film.dimY; y++ ) {
		for ( x=0; x<project->film.dimX; x++ ) {
			index = y * project->film.dimX + x;
			if ( project->record[index].select == EMS_SEL_COM ) {
				glColor3d(EMS_MKCOM_R, EMS_MKCOM_G, EMS_MKCOM_B);
				setMarker(x, y);
			}
			else if ( project->record[index].select == EMS_SEL_MAN ) {
				glColor3d(EMS_MKMAN_R, EMS_MKMAN_G, EMS_MKMAN_B);
				setMarker(x, y);
			}
			else if ( project->record[index].select == EMS_SEL_OUI ) {
				glColor3d(EMS_MKOUI_R, EMS_MKOUI_G, EMS_MKOUI_B);
				setMarker(x, y);
			}
		}
	}
}


inline void EMS_Display::drawFilaBoxing(void)
{
	double x0, y0, x1, y1, nx, ny, nr;
	double tx0, ty0, tx1, ty1, tx2, ty2, tx3, ty3;

	// draw filament selection only under FILAMENT mode
	if ( project->sysStatus.emsdpMode != DP_2DMIMAGE_MODE ) return;
	if ( (filaX0<0) || (filaY0<0) || (filaX1<0) || (filaY1<0) ) return;

	x0 = filaX0 + tranX - filmX/2.0;
	y0 = filaY0 + tranY - filmY/2.0;
	x1 = filaX1 + tranX - filmX/2.0;
	y1 = filaY1 + tranY - filmY/2.0;
	nr = sqrt((x1-x0)*(x1-x0) + (y1-y0)*(y1-y0));

	// draw filament cropping box
	if ( project->cropMode == EMS_TRUE ) {

		glColor3d(EMS_AXIS_R, EMS_AXIS_G, EMS_AXIS_B);
		glLineWidth((float)2.0);
		
		glBegin(GL_LINES);
			glVertex3d(x0, y0, EMS_LINE_GLD);
			glVertex3d(x1, y1, EMS_LINE_GLD);
		glEnd();
		
		if ( nr >= filaWd ) {
			nx = (x1 - x0 ) / nr;
			ny = (y1 - y0 ) / nr;
			
			tx0 = x0 + filaWd * ny; ty0 = y0 - filaWd * nx;
			tx1 = x0 - filaWd * ny; ty1 = y0 + filaWd * nx;
			tx2 = x1 - filaWd * ny; ty2 = y1 + filaWd * nx;
			tx3 = x1 + filaWd * ny; ty3 = y1 - filaWd * nx;
			
			glColor3d(0.0, 1.0, 0.0);
			glLineWidth((float)1.0);
			
			glBegin(GL_LINES);
				glVertex3d(tx0, ty0, EMS_LINE_GLD);
				glVertex3d(tx1, ty1, EMS_LINE_GLD);
			glEnd();
			glBegin(GL_LINES);
				glVertex3d(tx1, ty1, EMS_LINE_GLD);
				glVertex3d(tx2, ty2, EMS_LINE_GLD);
			glEnd();
			glBegin(GL_LINES);
				glVertex3d(tx2, ty2, EMS_LINE_GLD);
				glVertex3d(tx3, ty3, EMS_LINE_GLD);
			glEnd();
			glBegin(GL_LINES);
				glVertex3d(tx3, ty3, EMS_LINE_GLD);
				glVertex3d(tx0, ty0, EMS_LINE_GLD);
			glEnd();
		}
	}

	// draw rectangular cropping box
	else {
		glColor3d(0.0, 1.0, 0.0);
		glLineWidth((float)1.0);
		
		glBegin(GL_LINES);
			glVertex3d(x0, y0, EMS_LINE_GLD);
			glVertex3d(x0, y1, EMS_LINE_GLD);
		glEnd();
		glBegin(GL_LINES);
			glVertex3d(x0, y1, EMS_LINE_GLD);
			glVertex3d(x1, y1, EMS_LINE_GLD);
		glEnd();
		glBegin(GL_LINES);
			glVertex3d(x1, y1, EMS_LINE_GLD);
			glVertex3d(x1, y0, EMS_LINE_GLD);
		glEnd();
		glBegin(GL_LINES);
			glVertex3d(x1, y0, EMS_LINE_GLD);
			glVertex3d(x0, y0, EMS_LINE_GLD);
		glEnd();
	}
}


inline void EMS_Display::drawFilaTrace(void)
{
	int index, segid;
	double x0, y0, x1, y1, scaling;

	// draw filament trace only under FILAMENT mode
	if ( project->sysStatus.emsdpMode != DP_2DMIMAGE_MODE ) return;
	if ( project->sysStatus.showPrtk == EMS_FALSE ) return;
	if ( project->parameter.filaCount <= 0 ) return;

	scaling = 1.0 * project->parameter.downSize / project->parameter.dispScal;

	glColor3d(EMS_FILA_R, EMS_FILA_G, EMS_FILA_B);
	glLineWidth((float)EMS_FILA_LINE);

	for ( index=0; index<project->parameter.filaCount; index++ ) {

		x0 = project->filaAxis[index].x[0] * scaling + tranX - filmX/2;
		y0 = project->filaAxis[index].y[0] * scaling + tranY - filmY/2;

		for ( segid=0; segid<project->filaAxis[index].segid; segid++ ) {
			
			x1 = project->filaAxis[index].x[segid] * scaling + tranX - filmX/2;
			y1 = project->filaAxis[index].y[segid] * scaling + tranY - filmY/2;
			
			glBegin(GL_LINES);
				glVertex3d(x0, y0, EMS_FILA_GLD);
				glVertex3d(x1, y1, EMS_FILA_GLD);
			glEnd();

			x0 = x1; y0 = y1;
		}
	}
}


inline void EMS_Display::setMarker(int x, int y)
{
	double scaling, xf, yf;

	scaling = 1.0 * project->parameter.downSize / project->parameter.dispScal;
	xf = x * scaling + tranX - filmX/2;
	yf = y * scaling + tranY - filmY/2;

	glLineWidth((float)EMS_MARK_LINE);

	glBegin(GL_LINES);
		glVertex3d(xf-EMS_MARK_SIZE, yf, EMS_PEAK_GLD);
		glVertex3d(xf+EMS_MARK_SIZE, yf, EMS_PEAK_GLD);
	glEnd();

	glBegin(GL_LINES);
		glVertex3d(xf, yf-EMS_MARK_SIZE, EMS_PEAK_GLD);
		glVertex3d(xf, yf+EMS_MARK_SIZE, EMS_PEAK_GLD);
	glEnd();
}


inline void EMS_Display::drawFocus(void)
{
	double xf, yf, pixSize, boxSize, scaling;
	
	// safe-guard data availability
	if ( (focusX<0) || (focusX>=project->film.dimX) ) return;
	if ( (focusY<0) || (focusY>=project->film.dimY) ) return;

	pixSize = project->parameter.pixelSize * project->parameter.dispScal;
	scaling = 1.0 * project->parameter.downSize / project->parameter.dispScal;
	xf = (focusX * scaling + tranX - filmX/2);
	yf = (focusY * scaling + tranY - filmY/2);

	// box drawing for particle selection
	if ( project->sysStatus.emsdpMode == DP_PARTICLE_MODE ) {
		if ( project->inspectID < 0 )
			return;
		else if ( project->templates.exist == EMS_TRUE )
			boxSize = 0.5 * project->templates.dimX / project->parameter.dispScal;
		else
			boxSize = 0.5 * EMS_FOCUS_RAD * project->parameter.prtkSize / pixSize;
	}

	// box drawing for spectrum region selection
	else if ( project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE ) {
		if ( project->sysStatus.imgInform == EMS_HSTG_MODE )
			return;
		else
			boxSize = 0.5 * project->parameter.specSize / project->parameter.dispScal;
	}

	// no drawing otherwise
	else
		return;

	glLineWidth((float)EMS_FOCUS_LINE);
	glColor3d(EMS_FOCUS_R, EMS_FOCUS_G, EMS_FOCUS_B);

	glBegin(GL_LINES);
		glVertex3d(xf-boxSize, yf-boxSize, EMS_PEAK_GLD);
		glVertex3d(xf-boxSize, yf+boxSize, EMS_PEAK_GLD);
	glEnd();

	glBegin(GL_LINES);
		glVertex3d(xf-boxSize, yf+boxSize, EMS_PEAK_GLD);
		glVertex3d(xf+boxSize, yf+boxSize, EMS_PEAK_GLD);
	glEnd();

	glBegin(GL_LINES);
		glVertex3d(xf+boxSize, yf+boxSize, EMS_PEAK_GLD);
		glVertex3d(xf+boxSize, yf-boxSize, EMS_PEAK_GLD);
	glEnd();

	glBegin(GL_LINES);
		glVertex3d(xf+boxSize, yf-boxSize, EMS_PEAK_GLD);
		glVertex3d(xf-boxSize, yf-boxSize, EMS_PEAK_GLD);
	glEnd();
}


void EMS_Display::manuSelect(int x, int y)
{
	int xx, yy;
	double scaling, s_factor;
	
	if ( project->film.exist == EMS_FALSE ) return;
	
	// convert coordinates from GLWin to film[]
	scaling = 1.0 * project->parameter.dispScal / project->parameter.downSize;
	xx = (int)floor(((x - winX/2.0) / zoom - tranX + filmX/2.0) * scaling + 0.5);
	yy = (int)floor(((y - winY/2.0) / zoom - tranY + filmY/2.0) * scaling + 0.5);

	switch ( project->sysStatus.emsdpMode ) {
	// for particle selection
	case DP_PARTICLE_MODE:
		s_factor = project->prtkSelect(xx, yy);
		setInspectFocus(xx, yy, 0);
		EMS_DP->sldScore->value(s_factor);
		EMS_DP->dspCount->value(project->parameter.totCount);
		break;
	// for spectrum inspection
	case DP_MCRGRAPH_MODE:
		setInspectFocus(xx, yy, 0);
		cPanel->drawImgInfo(1);
		break;
	default:
		return;
	}

	redraw();
}


void EMS_Display::coopSelect(int x, int y)
{
	int xx, yy;
	double scaling, s_factor;
	
	if ( project->sysStatus.emsdpMode != DP_PARTICLE_MODE ) return;
	if ( project->film.exist == EMS_FALSE ) return;
	
	scaling = 1.0 * project->parameter.dispScal / project->parameter.downSize;
	xx = (int)floor(((x - winX/2) / zoom - tranX + filmX/2) * scaling + 0.5);
	yy = (int)floor(((y - winY/2) / zoom - tranY + filmY/2) * scaling + 0.5);

	s_factor = project->prtkCoopSelect(xx, yy);
	setInspectFocus(xx, yy, 0);
	redraw();

	EMS_DP->sldScore->value(s_factor);
	EMS_DP->dspCount->value(project->parameter.totCount);
}


void EMS_Display::manuDelete(int x, int y)
{
	int xx, yy;
	double scaling, s_factor;
	
	if ( project->sysStatus.emsdpMode != DP_PARTICLE_MODE ) return;
	if ( project->film.exist == EMS_FALSE ) return;
	
	scaling = 1.0 * project->parameter.dispScal / project->parameter.downSize;
	xx = (int)floor(((x - winX/2) / zoom - tranX + filmX/2) * scaling + 0.5);
	yy = (int)floor(((y - winY/2) / zoom - tranY + filmY/2) * scaling + 0.5);

	s_factor = project->prtkDelete(xx, yy);
	setInspectFocus(xx, yy, 0);
	redraw();

	EMS_DP->sldScore->value(s_factor);
	EMS_DP->dspCount->value(project->parameter.totCount);
}


void EMS_Display::setZoom(int delt)
{
	zoom += delt * EMS_ZSCALE;

	if ( zoom > EMS_Z1_LIM )
		zoom = EMS_Z1_LIM;
	
	if ( zoom < EMS_Z0_LIM )
		zoom = EMS_Z0_LIM;
}


void EMS_Display::setTrans(int dx, int dy)
{
	tranX += dx / zoom;
	tranY += dy / zoom;

	if ( EMS_DP->btnTemplate->value() ) {
		if ( tranY > 0.0 ) tranY = 0.0;
		tranX = 0.0;
		return;
	}

	if ( tranX > filmX/2 )
		tranX = 0.5 * filmX;
	else if ( tranX < -filmX/2 )
		tranX = -0.5 * filmX;

	if ( tranY > filmY/2 )
		tranY = 0.5 * filmY;
	else if ( tranY < -filmY/2 )
		tranY = -0.5 * filmY;
}


void EMS_Display::resetTransform(char flag)
{
	static char wasTemp=EMS_FALSE;
	static double tranX_0, tranY_0, zoom_0;

	if ( (flag==EMS_TRUE) && (wasTemp==EMS_FALSE) ) {
		zoom_0 = zoom;
		tranX_0 = tranX;
		tranY_0 = tranY;

		zoom = 1.0;
		tranX = 0.0;
		tranY = 0.0;

		wasTemp = EMS_TRUE;
	}

	if ( (flag==EMS_FALSE) && (wasTemp==EMS_TRUE) ) {
		zoom = zoom_0;
		tranX = tranX_0;
		tranY = tranY_0;

		wasTemp = EMS_FALSE;
	}
}


void EMS_Display::resetModelView(void)
{
	zoom3D = 1.0;
	tranX3 = deltaX = 0.0;
	tranY3 = deltaY = 0.0;
	rotaH3 = rotaV3 = 0.0;
	setIdentity(rotaMatrix, 4);
}


inline void EMS_Display::manuPaint(int x, int y)
{
	int x0, x1, y0, y1, xx, yy, dimX, dimY;
	long page;
	double br2;
	char *msk;

	if ( project->sysStatus.drawTools != EMS_PAINT ) return;
	br2 = project->parameter.brushSize * project->parameter.brushSize / 4.0;

	// template frame manual masking
	if ( (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) && project->sysStatus.editTemp ) {
		dimX = project->templates.dimX;
		dimY = project->templates.dimY;
		page = project->refer.editFrame * dimX * dimY;
		x = (int)((x - winX/2) / project->refer.scale + dimX/2);
		y = (int)((y - winY/2) / project->refer.scale + dimY/2);
		
		if ( (x<0) || (x>=dimX) ) return;
		if ( (y<0) || (y>=dimY) ) return;
		
		x0 = x - project->parameter.brushSize;
		if ( x0 < 0 ) x0 = 0;
		x1 = x + project->parameter.brushSize;
		if ( x1 >= dimX ) x1 = dimX - 1;
		y0 = y - project->parameter.brushSize;
		if ( y0 < 0 ) y0 = 0;
		y1 = y + project->parameter.brushSize;
		if ( y1 >= dimY ) y1 = dimY - 1;
		
		// determine which mask to draw
		if ( project->refer.maskID == EMS_MASK_L1 )
			msk = project->refer.L1Mask;
		else if ( project->refer.maskID == EMS_MASK_L2 )
			msk = project->refer.L2Mask;

		for ( yy=y0; yy<=y1; yy++ ) {
			for ( xx=x0; xx<=x1; xx++ ) {
				if ( ((xx-x)*(xx-x) + (yy-y)*(yy-y)) <= br2 )
					msk[page+yy*dimX+xx] = EMS_FALSE;
			}
		}
	}

	// micrograph masking
	else if ( (project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE) || 
		      (project->sysStatus.emsdpMode == DP_PARTICLE_MODE) ||
			  (project->sysStatus.emsdpMode == DP_2DMIMAGE_MODE) ) {

		x = (int)((x - winX/2) / zoom - tranX + filmX/2);
		y = (int)((y - winY/2) / zoom - tranY + filmY/2);
		
		if ( (x<0) || (x>=filmX) ) return;
		if ( (y<0) || (y>=filmY) ) return;
		
		x0 = x - project->parameter.brushSize;
		if ( x0 < 0 ) x0 = 0;
		x1 = x + project->parameter.brushSize;
		if ( x1 >= filmX ) x1 = filmX - 1;
		y0 = y - project->parameter.brushSize;
		if ( y0 < 0 ) y0 = 0;
		y1 = y + project->parameter.brushSize;
		if ( y1 >= filmY ) y1 = filmY - 1;
		
		for ( yy=y0; yy<=y1; yy++ ) {
			for ( xx=x0; xx<=x1; xx++ ) {
				if ( ((xx-x)*(xx-x) + (yy-y)*(yy-y)) <= br2 )
					project->dispMask.dataC[yy * filmX + xx] = EMS_TRUE;
			}
		}
	}
}


inline void EMS_Display::manuErase(int x, int y)
{
	int x0, x1, y0, y1, xx, yy, dimX, dimY;
	long page;
	double br2;
	char *msk;
	
	if ( project->sysStatus.drawTools != EMS_ERASE ) return;
	br2 = project->parameter.brushSize * project->parameter.brushSize / 4.0;

	// template frame manual masking
	if ( (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) && project->sysStatus.editTemp ) {
		dimX = project->templates.dimX;
		dimY = project->templates.dimY;
		page = project->refer.editFrame * dimX * dimY;
		x = (int)((x - winX/2) / project->refer.scale + dimX/2);
		y = (int)((y - winY/2) / project->refer.scale + dimY/2);
		
		if ( (x<0) || (x>=dimX) ) return;
		if ( (y<0) || (y>=dimY) ) return;
		
		x0 = x - project->parameter.brushSize;
		if ( x0 < 0 ) x0 = 0;
		x1 = x + project->parameter.brushSize;
		if ( x1 >= dimX ) x1 = dimX - 1;
		y0 = y - project->parameter.brushSize;
		if ( y0 < 0 ) y0 = 0;
		y1 = y + project->parameter.brushSize;
		if ( y1 >= dimY ) y1 = dimY - 1;

		// determine which mask to draw
		if ( project->refer.maskID == EMS_MASK_L1 )
			msk = project->refer.L1Mask;
		else if ( project->refer.maskID == EMS_MASK_L2 )
			msk = project->refer.L2Mask;

		for ( yy=y0; yy<=y1; yy++ ) {
			for ( xx=x0; xx<=x1; xx++ ) {
				if ( ((xx-x)*(xx-x) + (yy-y)*(yy-y)) <= br2 )
					msk[page+yy*dimX+xx] = EMS_TRUE;
			}
		}
	}

	// micrograph masking
	else if ( (project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE) || 
		      (project->sysStatus.emsdpMode == DP_PARTICLE_MODE) ||
			  (project->sysStatus.emsdpMode == DP_2DMIMAGE_MODE) ) {
		
		x = (int)((x - winX/2) / zoom - tranX + filmX/2);
		y = (int)((y - winY/2) / zoom - tranY + filmY/2);
		
		x0 = x - project->parameter.brushSize;
		x1 = x + project->parameter.brushSize;
		y0 = y - project->parameter.brushSize;
		y1 = y + project->parameter.brushSize;
		
		if ( (x0<0) || (x1>=filmX) ) return;
		if ( (y0<0) || (y1>=filmY) ) return;
		
		for ( yy=y0; yy<=y1; yy++ ) {
			for ( xx=x0; xx<=x1; xx++ ) {
				
				if ( ((xx-x)*(xx-x) + (yy-y)*(yy-y)) <= br2 )
					project->dispMask.dataC[yy * filmX + xx] = EMS_FALSE;
			}
		}
	}
}


inline void EMS_Display::manuLines(int x, int y, char lmode)
{
	int xx, yy, xm, ym, xp, yp, dimX, dimY;
	long page;
	static int x0=-1, x1=-1, y0=-1, y1=-1;
	double nr, nx, ny, xc, yc, dw, dh, dr0, dr1, lnwid;
	char tempEditFlag, *msk;
	
	if ( project->sysStatus.drawTools != EMS_LINES ) return;

	// template frame manual masking
	if ( (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) && project->sysStatus.editTemp ) {
		tempEditFlag = EMS_TRUE;
		dimX = project->templates.dimX;
		dimY = project->templates.dimY;
		xx = (int)((x - winX/2) / project->refer.scale + dimX/2);
		yy = (int)((y - winY/2) / project->refer.scale + dimY/2);
	}

	// micrograph masking
	else if ( (project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE) || 
		      (project->sysStatus.emsdpMode == DP_PARTICLE_MODE) ||
			  (project->sysStatus.emsdpMode == DP_2DMIMAGE_MODE) ) {

		tempEditFlag = EMS_FALSE;
		dimX = filmX; dimY = filmY;
		xx = (int)((x - winX/2) / zoom - tranX + dimX/2);
		yy = (int)((y - winY/2) / zoom - tranY + dimY/2);
	}
	else
		return;

	if ( (xx<0) || (xx>=dimX) ) {
		if ( lmode != 1 ) {
			x0 = y0 = x1 = y1 = -1;
			lineX0 = lineY0 = lineX1 = lineY1 = -1;
		}
		return;
	}
	if ( (yy<0) || (yy>=dimY) ) {
		if ( lmode != 1 ) {
			x0 = y0 = x1 = y1 = -1;
			lineX0 = lineY0 = lineX1 = lineY1 = -1;
		}
		return;
	}

	// (lmode=0) define the starting point of the line
	if ( lmode == 0 ) {
		x0 = xx; y0 = yy;
		lineX0 = x; lineY0 = y;
	}

	// (lmode=1) define the ending point of the line
	else if ( lmode == 1 ) {
		if ( (x0<0) || (y0<0) ) return;

		x1 = xx; y1 = yy;
		lineX1 = x; lineY1 = y;
	}

	// (lmode=-1) finalize line drawing
	else {
		if ( (x0<0) || (y0<0) ) return;

		x1 = xx; y1 = yy;
		lineX1 = x; lineY1 = y;

		xc = 0.5 * (x0 + x1);
		yc = 0.5 * (y0 + y1);
		nr = sqrt((x0-xc)*(x0-xc) + (y0-yc)*(y0-yc));
		if ( nr < 0.5 ) return;

		nx = (x0 - xc) / nr;
		ny = (y0 - yc) / nr;
		lnwid = project->parameter.brushSize / 2.0;

		xm = (x0 < x1)? x0 : x1;
		ym = (y0 < y1)? y0 : y1;
		xp = (x0 > x1)? x0 : x1;
		yp = (y0 > y1)? y0 : y1;

		xm = (int)floor(xm - lnwid);
		if ( xm < 0 ) xm = 0;
		ym = (int)floor(ym - lnwid);
		if ( ym < 0 ) ym = 0;
		xp = (int)ceil(xp + lnwid);
		if ( xp >= dimX ) xp = dimX - 1;
		yp = (int)ceil(yp + lnwid);
		if ( yp >= dimY ) yp = dimY - 1;

		// draw line for template frame masking
		if ( tempEditFlag ) {
			page = project->refer.editFrame * dimX * dimY;

			if ( project->refer.maskID == EMS_MASK_L1 )
				msk = project->refer.L1Mask;
			else if ( project->refer.maskID == EMS_MASK_L2 )
				msk = project->refer.L2Mask;

			for ( yy=ym; yy<=yp; yy++ ) {
				for ( xx=xm; xx<=xp; xx++ ) {
					
					// the line itself
					dw = fabs((xx-xc)*nx + (yy-yc)*ny);
					dh = fabs((xx-xc)*ny - (yy-yc)*nx);
					
					if ( (dw <= nr) && (dh <= lnwid) )
						msk[page+yy*dimX+xx] = EMS_FALSE;
					
					// round ending points
					dr0 = sqrt((xx-x0)*(xx-x0) + (yy-y0)*(yy-y0));
					dr1 = sqrt((xx-x1)*(xx-x1) + (yy-y1)*(yy-y1));
					
					if ( (dr0 <= lnwid) || (dr1 <= lnwid) )
						msk[page+yy*dimX+xx] = EMS_FALSE;
				}
			}
		}

		// draw line for micrograph masking
		else {
			for ( yy=ym; yy<=yp; yy++ ) {
				for ( xx=xm; xx<=xp; xx++ ) {
					
					// the line itself
					dw = fabs((xx-xc)*nx + (yy-yc)*ny);
					dh = fabs((xx-xc)*ny - (yy-yc)*nx);
					
					if ( (dw <= nr) && (dh <= lnwid) )
						project->dispMask.dataC[yy * dimX + xx] = EMS_TRUE;
					
					// round ending points
					dr0 = sqrt((xx-x0)*(xx-x0) + (yy-y0)*(yy-y0));
					dr1 = sqrt((xx-x1)*(xx-x1) + (yy-y1)*(yy-y1));
					
					if ( (dr0 <= lnwid) || (dr1 <= lnwid) )
						project->dispMask.dataC[yy * dimX + xx] = EMS_TRUE;
				}
			}
		}

		// prevent accidental mouse release and drawing
		x0 = y0 = x1 = y1 = -1;
		lineX0 = lineY0 = lineX1 = lineY1 = -1;
	}
}


inline void EMS_Display::manuCircle(int x, int y, char cmode)
{
	int xx, yy, dimX, dimY;
	long page;
	double xc, yc, r2;
	char *msk;
	
	if ( project->sysStatus.drawTools != EMS_CIRCLE ) return;

	// template frame manual masking
	if ( (project->sysStatus.emsdpMode==DP_TEMPLATE_MODE) && project->sysStatus.editTemp ) {
		dimX = project->templates.dimX;
		dimY = project->templates.dimY;

		// (cmode=0) initialize center and radius
		if ( cmode == 0 ) {
			circX = winX/2.0;
			circY = winY/2.0;
			circR = 0.25 * (dimX + dimY) * project->refer.scale;
		}

		// (cmode=1) translate the circle
		else if ( cmode == 1 ) {
			circX += x;
			circY += y;
		}
		
		// (cmode=2) change the radius
		else if ( cmode == 2 ) {
			circR += x;
			if ( circR < EMS_MINIR )
				circR = EMS_MINIR;
		}
		
		// (cmode=-1) finalize circle exclusion
		else {
			page = project->refer.editFrame * dimX * dimY;
			xc = (circX - winX/2) / project->refer.scale + dimX/2;
			yc = (circY - winY/2) / project->refer.scale + dimY/2;
			r2 = (circR / project->refer.scale) * (circR / project->refer.scale);

			if ( project->refer.maskID == EMS_MASK_L1 )
				msk = project->refer.L1Mask;
			else if ( project->refer.maskID == EMS_MASK_L2 )
				msk = project->refer.L2Mask;
			
			for ( yy=0; yy<project->templates.dimY; yy++ ) {
				for ( xx=0; xx<project->templates.dimX; xx++ ) {
					if ( ((xx-xc)*(xx-xc)+(yy-yc)*(yy-yc)) > r2 )
						msk[page+yy*dimX+xx] = EMS_FALSE;
				}
			}
		}
	}

	// micrograph masking
	else if ( (project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE) || 
		      (project->sysStatus.emsdpMode == DP_PARTICLE_MODE) ||
			  (project->sysStatus.emsdpMode == DP_2DMIMAGE_MODE) ) {
		
		// (cmode=0) initialize center and radius
		if ( cmode == 0 ) {
			circX = tranX * zoom + winX/2.0;
			circY = tranY * zoom + winY/2.0;
			circR = 0.25 * (filmX + filmY) * zoom;
		}
		
		// (cmode=1) translate the circle
		else if ( cmode == 1 ) {
			circX += x;
			circY += y;
		}
		
		// (cmode=2) change the radius
		else if ( cmode == 2 ) {
			circR += x;
			if ( circR < EMS_MINIR )
				circR = EMS_MINIR;
		}
		
		// (cmode=-1) finalize circle exclusion
		else {
			xc = (circX - winX/2) / zoom - tranX + filmX/2;
			yc = (circY - winY/2) / zoom - tranY + filmY/2;
			r2 = (circR / zoom) * (circR / zoom);
			
			for ( yy=0; yy<filmY; yy++ ) {
				for ( xx=0; xx<filmX; xx++ ) {
					if ( ((xx-xc)*(xx-xc)+(yy-yc)*(yy-yc)) > r2 )
						project->dispMask.dataC[yy * filmX + xx] = EMS_TRUE;
				}
			}
		}
	}
}


void EMS_Display::maskCircle(void)
{
	manuCircle(0, 0, -1);
	setMaskTexture(project->dispMask);
}


void EMS_Display::getFocusXY(int &x, int &y)
{
	x = focusX;
	y = focusY;
}


void EMS_Display::setFocusXY(int x, int y)
{
	focusX = x;
	focusY = y;
}


void EMS_Display::clearFocus(void)
{
	focusX = -1;
	focusY = -1;
}


void EMS_Display::getFilmDim(int &x, int &y)
{
	x = filmX;
	y = filmY;
}


inline void EMS_Display::zoomCenter(int x, int y)
{
	magX = x;
	magY = y;
}


inline void EMS_Display::drawZoomer(void)
{
	long index;
	int x, y, xc, yc, x0, y0, x1, y1, magSize;
	double intVal, xv, yv, s0, t0, s1, t1;
	double patikSize, axisLen, axisBar;
	double xf0, yf0, xf1, yf1, xfc, yfc;

	if ( project->sysStatus.showZoom == EMS_FALSE ) return;
	if ( (magX<0) || (magY<0) ) return;

	// find the location in the original micrograph (no resizing)
	xc = (int)(((magX - winX/2) / zoom - tranX + filmX/2) * project->parameter.dispScal + 0.5);
	yc = (int)(((magY - winY/2) / zoom - tranY + filmY/2) * project->parameter.dispScal + 0.5);

	patikSize = project->parameter.prtkSize / project->parameter.pixelSize;
	magSize = (int)(EMS_ZOOMSIZE * patikSize);
	x0 = xc - magSize/2; y0 = yc - magSize/2;
	x1 = xc + magSize/2; y1 = yc + magSize/2;

	// draw a bounding box (cyan)
	glColor3d(0.0, 0.8, 0.8);
	glLineWidth((float)1.0);

	s0 = (magX - magSize/2.0 - winX/2.0) / zoom;
	t0 = (magY - magSize/2.0 - winY/2.0) / zoom;
	s1 = (magX + magSize/2.0 - winX/2.0) / zoom;
	t1 = (magY + magSize/2.0 - winY/2.0) / zoom;

	glBegin(GL_LINES);
		glVertex3d(s0, t0, EMS_ZOOM_GLD);
		glVertex3d(s1, t0, EMS_ZOOM_GLD);
	glEnd();

	glBegin(GL_LINES);
		glVertex3d(s1, t0, EMS_ZOOM_GLD);
		glVertex3d(s1, t1, EMS_ZOOM_GLD);
	glEnd();

	glBegin(GL_LINES);
		glVertex3d(s1, t1, EMS_ZOOM_GLD);
		glVertex3d(s0, t1, EMS_ZOOM_GLD);
	glEnd();

	glBegin(GL_LINES);
		glVertex3d(s0, t1, EMS_ZOOM_GLD);
		glVertex3d(s0, t0, EMS_ZOOM_GLD);
	glEnd();

	// draw individual pixels in the original image
	for ( y=y0; y<=y1; y++ ) {
		yv = t0 + (y - y0) / zoom;
		for ( x=x0; x<=x1; x++ ) {
			xv = s0 + (x - x0) / zoom;

			if ( (x<0) || (x>=project->micrograph.dimX) || (y<0) || (y>=project->micrograph.dimY) ) {
				glColor3d(0.0, 0.0, 0.0);
				glBegin(GL_POINTS);
					glVertex3d(xv, yv, EMS_ZOOM_GLD);
				glEnd();

				continue;
			}
			else {
				index = y * project->micrograph.dimX + x;
				intVal = project->micrograph.density[index];

				// the original micrograph image has been normalized to N(0, 1)
				intVal /= (1.0 - project->parameter.flmCntr) * EMS_HSTG_SIGMA + EMS_ZERO;
				intVal += project->parameter.flmBrit;
				
				if ( intVal < 0.0 )
					intVal = 0.0;
				else if ( intVal > 1.0 )
					intVal = 1.0;

				glColor3d(intVal, intVal, intVal);
				glBegin(GL_POINTS);
					glVertex3d(xv, yv, EMS_ZOOM_GLD);
				glEnd();
			}
		}
	}

	// no cross-hair in MCRGRAPH module
	if ( project->sysStatus.emsdpMode == DP_MCRGRAPH_MODE ) return;

	// overlay cross-hair if btnCross1 is ON
	if ( project->sysStatus.mon1X == EMS_TRUE ) {
		
		glColor3d(0.0, 1.0, 0.0);
		glLineWidth((float)1.0);

		// draw major corss-hair
		axisLen = EMS_FOCUS_RAD * patikSize;
		xf0 = s0 + 0.5 * (magSize - axisLen) / zoom;
		xf1 = s0 + 0.5 * (magSize + axisLen) / zoom;
		yf0 = t0 + 0.5 * (magSize - axisLen) / zoom;
		yf1 = t0 + 0.5 * (magSize + axisLen) / zoom;
		xfc = s0 + 0.5 * magSize / zoom;
		yfc = t0 + 0.5 * magSize / zoom;

		glBegin(GL_LINES);
			glVertex3d(xf0, yfc, EMS_ZOOM_GLD-0.1);
			glVertex3d(xf1, yfc, EMS_ZOOM_GLD-0.1);
		glEnd();

		glBegin(GL_LINES);
			glVertex3d(xfc, yf0, EMS_ZOOM_GLD-0.1);
			glVertex3d(xfc, yf1, EMS_ZOOM_GLD-0.1);
		glEnd();

		// draw boundary bars (particle size)
		axisBar = patikSize * EMS_CROSS_BAR;
		xf0 = s0 + 0.5 * (magSize - axisBar) / zoom;
		xf1 = s0 + 0.5 * (magSize + axisBar) / zoom;
		yf0 = t0 + 0.5 * (magSize - axisBar) / zoom;
		yf1 = t0 + 0.5 * (magSize + axisBar) / zoom;

		xfc = s0 + 0.5 * (magSize - patikSize) / zoom;
		glBegin(GL_LINES);
			glVertex3d(xfc, yf0, EMS_ZOOM_GLD-0.1);
			glVertex3d(xfc, yf1, EMS_ZOOM_GLD-0.1);
		glEnd();

		xfc = s0 + 0.5 * (magSize + patikSize) / zoom;
		glBegin(GL_LINES);
			glVertex3d(xfc, yf0, EMS_ZOOM_GLD-0.1);
			glVertex3d(xfc, yf1, EMS_ZOOM_GLD-0.1);
		glEnd();

		yfc = t0 + 0.5 * (magSize - patikSize) / zoom;
		glBegin(GL_LINES);
			glVertex3d(xf0, yfc, EMS_ZOOM_GLD-0.1);
			glVertex3d(xf1, yfc, EMS_ZOOM_GLD-0.1);
		glEnd();

		yfc = t0 + 0.5 * (magSize + patikSize) / zoom;
		glBegin(GL_LINES);
			glVertex3d(xf0, yfc, EMS_ZOOM_GLD-0.1);
			glVertex3d(xf1, yfc, EMS_ZOOM_GLD-0.1);
		glEnd();
	}
}


void EMS_Display::resetZoomer(void)
{
	magX = magY = -1;
}


inline void EMS_Display::drawFilmMarker(void)
{
	int index;
	double xc, yc, x0, y0, x1, y1, scaling;

	if ( markNum == 0 ) return;
	if ( project->sysStatus.emsdpMode != DP_MCRGRAPH_MODE ) return;

	glColor3d(1.0, 0.0, 0.0);
	glLineWidth((float)1.0);
	scaling = 1.0 * project->parameter.downSize / project->parameter.dispScal;

	for ( index=0; index<markNum; index++ ) {
		
		xc = (int)(marker[index][0] * scaling + tranX - filmX/2.0);
		yc = (int)(marker[index][1] * scaling + tranY - filmY/2.0);
		x0 = xc - 0.5 * EMS_MARKER_SIZE;
		x1 = xc + 0.5 * EMS_MARKER_SIZE;
		y0 = yc - 0.5 * EMS_MARKER_SIZE;
		y1 = yc + 0.5 * EMS_MARKER_SIZE;
		
		glBegin(GL_LINES);
			glVertex3d(x0, y0, EMS_MARK_GLD);
			glVertex3d(x1, y0, EMS_MARK_GLD);
		glEnd();
		glBegin(GL_LINES);
			glVertex3d(x1, y0, EMS_MARK_GLD);
			glVertex3d(x1, y1, EMS_MARK_GLD);
		glEnd();
		glBegin(GL_LINES);
			glVertex3d(x1, y1, EMS_MARK_GLD);
			glVertex3d(x0, y1, EMS_MARK_GLD);
		glEnd();
		glBegin(GL_LINES);
			glVertex3d(x0, y1, EMS_MARK_GLD);
			glVertex3d(x0, y0, EMS_MARK_GLD);
		glEnd();
		glBegin(GL_LINES);
			glVertex3d(x0, y0, EMS_MARK_GLD);
			glVertex3d(x1, y1, EMS_MARK_GLD);
		glEnd();
		glBegin(GL_LINES);
			glVertex3d(x0, y1, EMS_MARK_GLD);
			glVertex3d(x1, y0, EMS_MARK_GLD);
		glEnd();
	}
}


void EMS_Display::setFilmMarker(void)
{
	if ( markNum > EMS_FILM_MARKER ) {
		errReport("Too many markers on the film!");
		return;
	}

	// focusX & focusY are in film[] coordinates
	if ( (focusX<0) || (focusX>=project->film.dimX) || 
		 (focusY<0) || (focusY>=project->film.dimY) ) {
		markNum = 0;
	}
	else {
		marker[markNum][0] = focusX;
		marker[markNum][1] = focusY;
		markNum ++;
	}

	redraw();
}


void EMS_Display::clearFilmMarker(void)
{
	markNum = 0;
	redraw();
}


inline void EMS_Display::manuBoxing(int x, int y, char bmode)
{
	int xx, yy, x0, y0, x1, y1;
	
	if ( project->film.exist == EMS_FALSE ) return;
	if ( project->sysStatus.drawTools != EMS_FALSE ) return;
	
	// in dispFilm[] coordinates
	xx = (int)(((x - winX/2.0) / zoom - tranX + filmX/2.0) + 0.5);
	yy = (int)(((y - winY/2.0) / zoom - tranY + filmY/2.0) + 0.5);
	if ( (xx<0) || (xx>=filmX) || (yy<0) || (yy>=filmY) ) return;

	// (bmode=0) define the starting point of the filament box
	if ( bmode == 0 ) {
		filaX0 = xx; filaY0 = yy;
		filaX1 = filaY1 = -1;
		
		filaWd = 0.5 * project->filaDiam;
		filaWd /= project->parameter.pixelSize * project->parameter.dispScal;
	}

	// (bmode=1) define the ending point of the filament box
	else if ( bmode == 1 ) {
		if ( (filaX0<0) || (filaY0<0) ) return;
		filaX1 = xx; filaY1 = yy;

		if ( project->cropMode == EMS_FALSE )
			filaWd = 0.5 * (filaY1 - filaY0) / project->parameter.dispScal;
	}

	// (bmode=-1) finalize filament boxing
	else {
		if ( (filaX0<0) || (filaY0<0) ) return;

		filaX1 = xx; filaY1 = yy;
		x0 = (int)(filaX0 * project->parameter.dispScal);
		y0 = (int)(filaY0 * project->parameter.dispScal);
		x1 = (int)(filaX1 * project->parameter.dispScal);
		y1 = (int)(filaY1 * project->parameter.dispScal);

		project->setFilaModel(x0, y0, x1, y1, 1);
	}
}


void EMS_Display::clearFilaBoxing(void)
{
	filaX0 = filaY0 = filaX1 = filaY1 = -1;
	redraw();
}


void EMS_Display::setFilaBoxing(double xc, double yc, int length, int width, double angle)
{
	double scaling, cosAng, sinAng;

	scaling = 1.0 / project->parameter.dispScal;

	// for filament cropping
	if ( project->cropMode == EMS_TRUE ) {
		cosAng = +cos(angle * DEG2RAD);
		sinAng = -sin(angle * DEG2RAD);
		filaX0 = scaling * (xc - 0.5 * length * cosAng);
		filaY0 = scaling * (yc - 0.5 * length * sinAng);
		filaX1 = scaling * (xc + 0.5 * length * cosAng);
		filaY1 = scaling * (yc + 0.5 * length * sinAng);
		filaWd = scaling * 0.5 * width;
	}

	// for rectangular cropping
	else {
		filaX0 = scaling * (xc - 0.5 * length);
		filaY0 = scaling * (yc - 0.5 * width);
		filaX1 = scaling * (xc + 0.5 * length);
		filaY1 = scaling * (yc + 0.5 * width);
		filaWd = scaling * 0.5 * width;
	}

	redraw();
}


void EMS_Display::setAlignCenter(int x, int y)
{
	int xx, yy;

	if ( project->sysStatus.editTemp == EMS_FALSE ) return;

	// convert coordinates from GLWin to film[]
	xx = (int)((x - winX/2.0)/project->refer.scale + project->templates.dimX/2.0 + 0.5);
	yy = (int)((y - winY/2.0)/project->refer.scale + project->templates.dimY/2.0 + 0.5);

	if ( (xx>=0) && (xx<project->templates.dimX) && (yy>=0) && (yy<project->templates.dimY) ) {
		project->refer.xc[project->refer.editFrame] = xx;
		project->refer.yc[project->refer.editFrame] = yy;
		project->sysStatus.mapRestore = EMS_FALSE;
		redraw();
	}
}


inline void EMS_Display::gelThreshold(int x, int y, char state)
{
	double mL, mH;
	static int x0, y0;
	static float massL, massH;
	static char grabL=EMS_FALSE;
	static char grabH=EMS_FALSE;

	if ( functImage->gelBinMax <= 0 ) return;

	// record starting point
	if ( state == 0 ) {
		x0 = x; y0 = y;
		mL = EMS_GEL_BINPIX * (functImage->gelLmass - functImage->gelMassAve) / functImage->gelMassStd;
		mH = EMS_GEL_BINPIX * (functImage->gelHmass - functImage->gelMassAve) / functImage->gelMassStd;
		mL = mL + winX/2 - x0;
		mH = x0 - winX/2 - mH;
		
		// detect low-end picking
		if ( (mL>=0) && (mL<=EMS_GEL_PICK) ) {
			grabL = EMS_TRUE;
			massL = functImage->gelLmass;
		}
		
		// detect high-end picking
		else if ( (mH>=0) && (mH<=EMS_GEL_PICK) ) {
			grabH = EMS_TRUE;
			massH = functImage->gelHmass;
		}

		else {
			grabL = EMS_FALSE;
			grabH = EMS_FALSE;
		}
	}

	// reset flags
	else if ( state == -1 ) {
		grabL = EMS_FALSE;
		grabH = EMS_FALSE;
	}

	// update gelLmass if picked
	if ( grabL ) {
		mL = massL + functImage->gelMassStd * (x-x0) / EMS_GEL_BINPIX;
		functImage->setGelLmass(mL);
		EMS_DP->gelMassLow->value(functImage->gelLmass);
		EMS_DP->gelMassHigh->value(functImage->gelHmass);
	}

	// update gelHmass if picked
	else if ( grabH ) {
		mH = massH + functImage->gelMassStd * (x-x0) / EMS_GEL_BINPIX;
		functImage->setGelHmass(mH);
		EMS_DP->gelMassLow->value(functImage->gelLmass);
		EMS_DP->gelMassHigh->value(functImage->gelHmass);
	}
}
